﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Renderer2D.Core;
using Renderer2D.Core.WinForms;
using System.ComponentModel;

namespace Renderer2D.GDI
{
    [Export(typeof(Renderer))]
    [DisplayName("GDI Renderer")]
    public class GDIRenderer : Renderer
    {
        public override bool IsHardwareAccelerated
        {
            get { return false; }
        }

        protected Graphics graphics;

        public override void BeginDraw(RenderContext renderContext)
        {
            base.BeginDraw(renderContext);

            var context = CheckRenderContext<WinFormsRenderContext>(renderContext);
            SetGraphics(context.RenderGraphics, true);
        }

        private void SetGraphics(Graphics gr, bool setAttributes)
        {
            graphics = gr;

            if (setAttributes)
            {
                graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
                graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.AntiAlias;

                // PixelOffsetMode set to workaround an off by one GDI bug
                // http://stackoverflow.com/questions/8431427/gdi-lineargradientbrush-bug#comment10418535_8431427
                graphics.PixelOffsetMode = System.Drawing.Drawing2D.PixelOffsetMode.Half;
            }

            if (Transform != null)
                graphics.Transform = Transform.ToSpecificMatrix();
        }

        protected override void ClearOverride(IColor color)
        {
            graphics.Clear(color.ToColor());
        }

        #region Brushes

        protected override Core.ISolidColorBrush CreateSolidColorBrushOverride(Core.IColor color)
        {
            return new GDISolidBrushContainer(color);
        }

        protected override Core.ILinearGradientBrush CreateLinearGradientBrushOverride(Core.IPoint startPoint,
                                                                                       Core.IPoint endPoint,
                                                                                       params Core.IGradientStop[] gradientStops)
        {
            return new GDILinearGradientBrushContainer(startPoint, endPoint, gradientStops);
        }

        protected override Core.IRadialGradientBrush CreateRadialGradientBrushOverride(Core.IPoint center,
                                                                                       Core.IPoint gradientOriginOffset,
                                                                                       double radiusX,
                                                                                       double radiusY,
                                                                                       params Core.IGradientStop[] gradientStops)
        {
            return new GDIRadialGradientBrushContainer(center, gradientOriginOffset, radiusX, radiusY, gradientStops);
        }

        #endregion

        #region Bitmaps

        protected override IBitmap CreateBitmapOverride(IBitmapDefinition bitmapDefinition)
        {
            return new GDIBitmapContainer(bitmapDefinition);
        }

        protected override void DrawBitmapOverride(IBitmap bitmap,
                                                   IRectangle destinationRectangle,
                                                   IRectangle sourceRectangle,
                                                   double opacity,
                                                   BitmapInterpolationMode interpolationMode)
        {
            var bc = (GDIBitmapContainer)bitmap;
            graphics.DrawImage(bc.NativeBitmap, destinationRectangle.ToRectangleF(), sourceRectangle.ToRectangleF(), GraphicsUnit.Pixel);
        }

        #endregion

        #region Text

        protected override ISize MeasureTextOverride(string text, ITextFormat textFormat)
        {
            using (var font = new System.Drawing.Font(textFormat.Font.FamilyName, (float)textFormat.Font.Size, textFormat.ToFontStyle()))
            {
                var size = TextRenderer.MeasureText(text, font);
                return new Core.Size(size.Width, size.Height);
            }
        }

        protected override void DrawTextOverride(string text,
                                                 ITextFormat textFormat,
                                                 IRectangle drawTextArea,
                                                 IBrush brush,
                                                 DrawTextOptions options)
        {
            using (var drawFont = new System.Drawing.Font(textFormat.Font.FamilyName, (float)textFormat.Font.Size, textFormat.ToFontStyle()))
            {
                var x = (float)drawTextArea.X;
                var y = (float)drawTextArea.Y;
                graphics.DrawString(text, drawFont, brush.ToBrush(), x, y);
            }
        }

        #endregion

        #region Shapes

        protected override void DrawLineOverride(IPoint point1,
            IPoint point2,
            IBrush brush,
            double strokeWidth,
            IStrokeStyle strokeStyle)
        {
            graphics.DrawLine(brush.ToPen(strokeWidth, strokeStyle), point1.ToPointF(), point2.ToPointF());
        }

        protected override void DrawBezierOverride(IPoint startPoint,
            IPoint controlPoint1,
            IPoint controlPoint2,
            IPoint endPoint,
            IBrush brush,
            double strokeWidth,
            IStrokeStyle strokeStyle)
        {
            graphics.DrawBezier(brush.ToPen(strokeWidth, strokeStyle),
                startPoint.ToPointF(),
                controlPoint1.ToPointF(),
                controlPoint2.ToPointF(),
                endPoint.ToPointF());
        }

        protected override void DrawEllipseOverride(IEllipse ellipse,
            IBrush brush,
            double strokeWidth,
            IStrokeStyle strokeStyle)
        {
            RectangleF rect = ellipse.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);
            graphics.DrawEllipse(brush.ToPen(strokeWidth, strokeStyle), rect);
        }

        protected override void DrawRectangleOverride(IRectangle rectangle,
            IBrush brush,
            double strokeWidth,
            IStrokeStyle strokeStyle)
        {
            RectangleF rect = rectangle.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);
            graphics.DrawRectangle(brush.ToPen(strokeWidth, strokeStyle), rectangle.ToRectangle());
        }

        protected override void DrawRoundedRectangleOverride(IRoundedRectangle rectangle,
            IBrush brush,
            double strokeWidth,
            IStrokeStyle strokeStyle)
        {
            RectangleF rect = rectangle.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);

            var path = CreateRoundedRectanglePath(rectangle);

            graphics.DrawPath(brush.ToPen(strokeWidth, strokeStyle), path);
        }

        protected override void FillEllipseOverride(IEllipse ellipse, IBrush brush)
        {
            RectangleF rect = ellipse.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);
            graphics.FillEllipse(brush.ToBrush(), rect);
        }

        protected override void FillRectangleOverride(IRectangle rectangle, IBrush brush)
        {
            RectangleF rect = rectangle.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);
            graphics.FillRectangle(brush.ToBrush(), rect);
        }

        protected override void FillRoundedRectangleOverride(IRoundedRectangle rectangle, IBrush brush)
        {
            RectangleF rect = rectangle.ToRectangleF();
            (brush as GDIBrushContainer).SetBoundingBox(rect);

            var path = CreateRoundedRectanglePath(rectangle);

            graphics.FillPath(brush.ToBrush(), path);
        }

        private static System.Drawing.Drawing2D.GraphicsPath CreateBottomRoundedRectanglePath(IRoundedRectangle rectangle)
        {
            var x0 = (float)(rectangle.X);
            var x1 = (float)(rectangle.X + rectangle.RadiusX);
            var x2 = (float)(rectangle.X + rectangle.Width - rectangle.RadiusX);
            var x3 = (float)(rectangle.X + rectangle.Width);
            var y0 = (float)(rectangle.Y);
            var y1 = (float)(rectangle.Y + rectangle.RadiusY);
            var y2 = (float)(rectangle.Y + rectangle.Height - rectangle.RadiusY);
            var y12 = (y1 + y2) / 2.0f;
            var y3 = (float)(rectangle.Y + rectangle.Height);
            var arcWidth = (float)(2.0 * rectangle.RadiusX);
            var arcHeight = (float)(2.0 * rectangle.RadiusY);
            var c1x = x0;
            var c1y = y0;
            var c2x = x3 - arcWidth;
            var c2y = y3 - arcHeight;

            var path = new System.Drawing.Drawing2D.GraphicsPath();
            bool drawArcs = arcWidth >= 1e-3 && arcHeight >= 1e-3;
            path.StartFigure();
            path.AddLine(x0, y3, x0, y1); if (drawArcs) path.AddArc(c1x, c1y, arcWidth, arcHeight, 180.0f, 90.0f);
            path.AddLine(x1, y0, x2, y0); if (drawArcs) path.AddArc(c2x, c1y, arcWidth, arcHeight, 270.0f, 90.0f);
            path.AddLine(x3, y1, x3, y3);
            path.CloseFigure();

            return path;
        }

        private static System.Drawing.Drawing2D.GraphicsPath CreateUpperRoundedRectanglePath(IRoundedRectangle rectangle)
        {
            var x0 = (float)(rectangle.X);
            var x1 = (float)(rectangle.X + rectangle.RadiusX);
            var x2 = (float)(rectangle.X + rectangle.Width - rectangle.RadiusX);
            var x3 = (float)(rectangle.X + rectangle.Width);
            var y0 = (float)(rectangle.Y);
            var y1 = (float)(rectangle.Y + rectangle.RadiusY);
            var y2 = (float)(rectangle.Y + rectangle.Height - rectangle.RadiusY);
            var y12 = (y1 + y2) / 2.0f;
            var y3 = (float)(rectangle.Y + rectangle.Height);
            var arcWidth = (float)(2.0 * rectangle.RadiusX);
            var arcHeight = (float)(2.0 * rectangle.RadiusY);
            var c1x = x0;
            var c1y = y0;
            var c2x = x3 - arcWidth;
            var c2y = y3 - arcHeight;

            var path = new System.Drawing.Drawing2D.GraphicsPath();
            bool drawArcs = arcWidth >= 1e-3 && arcHeight >= 1e-3;
            path.StartFigure();
            path.AddLine(x0, y3, x0, y1); if (drawArcs) path.AddArc(c1x, c1y, arcWidth, arcHeight, 180.0f, 90.0f);
            path.AddLine(x1, y0, x2, y0);  if (drawArcs) path.AddArc(c2x, c1y, arcWidth, arcHeight, 270.0f, 90.0f);
            path.AddLine(x3, y1, x3, y3);
            path.CloseFigure();

            return path;
        }

        private static System.Drawing.Drawing2D.GraphicsPath CreateRoundedRectanglePath(IRoundedRectangle rectangle)
        {
            var x0 = (float)(rectangle.X);
            var x1 = (float)(rectangle.X + rectangle.RadiusX);
            var x2 = (float)(rectangle.X + rectangle.Width - rectangle.RadiusX);
            var x3 = (float)(rectangle.X + rectangle.Width);
            var y0 = (float)(rectangle.Y);
            var y1 = (float)(rectangle.Y + rectangle.RadiusY);
            var y2 = (float)(rectangle.Y + rectangle.Height - rectangle.RadiusY);
            var y3 = (float)(rectangle.Y + rectangle.Height);
            var arcWidth = (float)(2.0 * rectangle.RadiusX);
            var arcHeight = (float)(2.0 * rectangle.RadiusY);
            var c1x = x0;
            var c1y = y0;
            var c2x = x3 - arcWidth;
            var c2y = y3 - arcHeight;

            var path = new System.Drawing.Drawing2D.GraphicsPath();
            bool drawArcs = arcWidth >= 1e-3 && arcHeight >= 1e-3;
            path.StartFigure();
            switch(rectangle.Style)
            {
                case CornerStyle.RoundedUpper:
                    path.AddLine(x0, y3, x0, y1); if (drawArcs) path.AddArc(c1x, c1y, arcWidth, arcHeight, 180.0f, 90.0f);
                    path.AddLine(x1, y0, x2, y0); if (drawArcs) path.AddArc(c2x, c1y, arcWidth, arcHeight, 270.0f, 90.0f);
                    path.AddLine(x3, y1, x3, y3);
                    break;

                case CornerStyle.RoundedBottom:
                    path.AddLine(x0, y0, x3, y0);
                    path.AddLine(x3, y0, x3, y2); if (drawArcs) path.AddArc(c2x, c2y, arcWidth, arcHeight, 0.0f, 90.0f);
                    path.AddLine(x2, y3, x1, y3); if (drawArcs) path.AddArc(c1x, c2y, arcWidth, arcHeight, 90.0f, 90.0f);
                    path.AddLine(x0, y2, x0, y0);
                    break;

                default:
                    path.AddLine(x1, y0, x2, y0); if (drawArcs) path.AddArc(c2x, c1y, arcWidth, arcHeight, 270.0f, 90.0f);
                    path.AddLine(x3, y1, x3, y2); if (drawArcs) path.AddArc(c2x, c2y, arcWidth, arcHeight, 0.0f, 90.0f);
                    path.AddLine(x2, y3, x1, y3); if (drawArcs) path.AddArc(c1x, c2y, arcWidth, arcHeight, 90.0f, 90.0f);
                    path.AddLine(x0, y2, x0, y1); if (drawArcs) path.AddArc(c1x, c1y, arcWidth, arcHeight, 180.0f, 90.0f);
                    break;
            }
            path.CloseFigure();

            return path;
        }

        #endregion

        private IMatrix transform;
        public override IMatrix Transform
        {
            get
            {
                return transform;
            }
            set
            {
                transform = value;
                if (graphics != null)
                    graphics.Transform = transform.ToSpecificMatrix();
            }
        }
    }
}
