﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct2D1;
using SharpDX.DirectWrite;
using System.IO;

namespace Renderer2D.Direct2D1
{
    public abstract class Direct2D1Renderer : Core.Renderer
    {
        protected RenderTarget renderTarget;

        public override bool IsHardwareAccelerated
        {
            get { return true; }
        }

        protected override void ClearOverride(Core.IColor color)
        {
            renderTarget.Clear(color.ToD2D1Color());
        }

        public override void BeginDraw(Core.RenderContext renderContext)
        {
            base.BeginDraw(renderContext);
            renderTarget.BeginDraw();
        }

        public override void EndDraw()
        {
            renderTarget.EndDraw();
            base.EndDraw();
        }

        protected abstract Core.ISize GetSystemDpi();

        protected override Core.IBitmap CreateBitmapOverride(Core.IBitmapDefinition bitmapDefinition)
        {
            var pixelFormat = new PixelFormat(SharpDX.DXGI.Format.B8G8R8A8_UNorm, AlphaMode.Premultiplied);
            var bitmapProperties = new BitmapProperties(pixelFormat, 96.0f, 96.0f);
            var pixelSize = new Size2((int)bitmapDefinition.Size.Width, (int)bitmapDefinition.Size.Height);
            var pitch = (int)bitmapDefinition.Size.Width * 4;

            var dataStream = DataStream.Create(bitmapDefinition.Data, true, false);
            var nativeBitmap = new Bitmap(renderTarget, pixelSize, dataStream, pitch, bitmapProperties);

            return new D2D1BitmapContainer(nativeBitmap);
        }

        protected override Core.ISolidColorBrush CreateSolidColorBrushOverride(Core.IColor color)
        {
            return new D2D1SolidColorBrushContainer(color, renderTarget);
        }

        protected override Core.ILinearGradientBrush CreateLinearGradientBrushOverride(Core.IPoint startPoint, Core.IPoint endPoint, params Core.IGradientStop[] gradientStops)
        {
            return new D2D1LinearGradientBrushContainer(renderTarget, startPoint, endPoint, gradientStops);
        }

        protected override Core.IRadialGradientBrush CreateRadialGradientBrushOverride(Core.IPoint center, Core.IPoint gradientOriginOffset, double radiusX, double radiusY, params Core.IGradientStop[] gradientStops)
        {
            return new D2D1RadialGradientBrushContainer(renderTarget, center, gradientOriginOffset, radiusX, radiusY, gradientStops);
        }

        protected override void DrawBitmapOverride(Core.IBitmap bitmap, Core.IRectangle destinationRectangle, Core.IRectangle sourceRectangle, double opacity, Core.BitmapInterpolationMode interpolationMode)
        {
            var b = bitmap as D2D1BitmapContainer;
            if (b == null)
                throw new ArgumentException(string.Format("Invalid bitmap (Unknown type '{0}')", bitmap.GetType().FullName));

            var bim = BitmapInterpolationMode.Linear;
            if (interpolationMode == Core.BitmapInterpolationMode.NearestNeighbor)
                bim = BitmapInterpolationMode.NearestNeighbor;

            renderTarget.DrawBitmap(b.NativeBitmap, destinationRectangle.ToD2D1Rect(), (float)opacity, bim, sourceRectangle.ToD2D1Rect());
        }

        protected abstract TextFormat CreateTextFormat(Core.ITextFormat textFormat);
        protected abstract TextLayout CreateTextLayout(string text, Core.ITextFormat textFormat, double maxWidth, double maxHeight);
        protected abstract PathGeometry CreatePathGeometry();

        protected override Core.ISize MeasureTextOverride(string text, Core.ITextFormat textFormat)
        {
            var systemDpi = GetSystemDpi();

            using (var textLayout = CreateTextLayout(text, textFormat, 100000.0, 100000.0))
            {
                var dipWidth = textLayout.Metrics.WidthIncludingTrailingWhitespace;
                var dipHeight = textLayout.Metrics.Height;

                return new Core.Size(dipWidth * systemDpi.Width / 96.0, dipHeight * systemDpi.Height / 96.0);
            }
        }

        protected override void DrawTextOverride(string text, Core.ITextFormat textFormat, Core.IRectangle drawTextArea, Core.IBrush brush, Core.DrawTextOptions options)
        {
            var b = CheckBrush(brush);
            using (var tf = CreateTextFormat(textFormat))
            {
                renderTarget.DrawText(text, tf, drawTextArea.ToD2D1Rect(), b.NativeBrush, options.ToD2D1DrawTextOptions());
            }
        }

        protected override void DrawEllipseOverride(Core.IEllipse ellipse, Core.IBrush brush, double strokeWidth, Core.IStrokeStyle strokeStyle)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(ellipse.ToRectangle());
            renderTarget.DrawEllipse(ellipse.ToD2D1Ellipse(), b.NativeBrush, (float)strokeWidth);
        }

        protected override void DrawLineOverride(Core.IPoint point1, Core.IPoint point2, Core.IBrush brush, double strokeWidth, Core.IStrokeStyle strokeStyle)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(new Core.Rectangle(point1, point2));
            renderTarget.DrawLine(point1.ToPoint2F(), point2.ToPoint2F(), b.NativeBrush, (float)strokeWidth);
        }

        protected override void DrawBezierOverride(Core.IPoint startPoint, Core.IPoint controlPoint1, Core.IPoint controlPoint2, Core.IPoint endPoint, Core.IBrush brush, double strokeWidth, Core.IStrokeStyle strokeStyle)
        {
            var b = CheckBrush(brush);

            using (var pathGeometry = CreatePathGeometry())
            {
                using (var sink = pathGeometry.Open())
                {
                    sink.BeginFigure(startPoint.ToPoint2F(), FigureBegin.Filled);
                    sink.AddBezier(new BezierSegment
                    {
                        Point1 = controlPoint1.ToPoint2F(),
                        Point2 = controlPoint2.ToPoint2F(),
                        Point3 = endPoint.ToPoint2F(),
                    });
                    sink.EndFigure(FigureEnd.Open);
                    sink.Close();
                }

                renderTarget.DrawGeometry(pathGeometry, b.NativeBrush, (float)strokeWidth);
            }
        }

        protected override void DrawRectangleOverride(Core.IRectangle rectangle, Core.IBrush brush, double strokeWidth, Core.IStrokeStyle strokeStyle)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(rectangle);
            renderTarget.DrawRectangle(rectangle.ToD2D1Rect(), b.NativeBrush, (float)strokeWidth);
        }

        protected override void DrawRoundedRectangleOverride(Core.IRoundedRectangle rectangle, Core.IBrush brush, double strokeWidth, Core.IStrokeStyle strokeStyle)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(rectangle);
            renderTarget.DrawRoundedRectangle(rectangle.ToD2D1RoundedRect(), b.NativeBrush, (float)strokeWidth);
        }

        protected override void FillEllipseOverride(Core.IEllipse ellipse, Core.IBrush brush)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(ellipse.ToRectangle());
            renderTarget.FillEllipse(ellipse.ToD2D1Ellipse(), b.NativeBrush);
        }

        protected override void FillRectangleOverride(Core.IRectangle rectangle, Core.IBrush brush)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(rectangle);
            renderTarget.FillRectangle(rectangle.ToD2D1Rect(), b.NativeBrush);
        }

        protected override void FillRoundedRectangleOverride(Core.IRoundedRectangle rectangle, Core.IBrush brush)
        {
            var b = CheckBrush(brush);
            b.SetBoundingBox(rectangle);
            renderTarget.FillRoundedRectangle(rectangle.ToD2D1RoundedRect(), b.NativeBrush);
        }

        private D2D1BrushContainer CheckBrush(Core.IBrush brush)
        {
            var b = brush as D2D1BrushContainer;
            if (b == null)
                throw new ArgumentException(string.Format("Invalid brush (Unknown type '{0}')", brush.GetType().FullName));
            return b;
        }

        private Core.IMatrix transform;
        public override Core.IMatrix Transform
        {
            get
            {
                return transform;
            }
            set
            {
                transform = value;
                renderTarget.Transform = transform.ToSpecificMatrix();
            }
        }
    }
}
