﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace Renderer2D.Core
{
    /// <summary>
    /// Represent an abstract render target.
    /// </summary>
    public abstract class Renderer
    {
        public virtual bool AntiAliasing { get; set; }
        public virtual bool TextAntiAliasing { get; set; }
        public abstract bool IsHardwareAccelerated { get; }

        public virtual void BeginDraw(RenderContext renderContext) { }
        public virtual void EndDraw() { }

        protected T CheckRenderContext<T>(RenderContext renderContext) where T : RenderContext
        {
            var typedRenderContext = renderContext as T;
            if (typedRenderContext == null)
            {
                var message = string.Format(Messages.EXCEPTION_INVALID_RENDER_CONTEXT_TYPE,
                    renderContext.GetType().FullName, typeof(T).FullName);
                throw new ArgumentException(message);
            }
            return typedRenderContext;
        }

        public virtual void OnViewportResized(ISize newSize)
        {
            var handler = ViewportResized;
            if (handler != null)
                handler(this, newSize);
        }

        public delegate void SizeHander(object sender, ISize size);
        public event SizeHander ViewportResized;

        public virtual void Initialize() { }

        public void Clear(IColor color)
        {
            if (color == null)
                throw new ArgumentNullException("color");

            ClearOverride(color);
        }

        protected abstract void ClearOverride(IColor color);

        public abstract IMatrix Transform { get; set; }

        #region CreateBitmap

        public IBitmap CreateBitmap(int width, int height, byte[] data)
        {
            if (data == null)
                throw new ArgumentNullException("data");

            return CreateBitmap(new BitmapDefinition(data, width, height));
        }

        public IBitmap CreateBitmap(IBitmapDefinition bitmapDefinition)
        {
            if (bitmapDefinition == null)
                throw new ArgumentNullException("bitmapDefinition");

            return CheckReturn(CreateBitmapOverride(bitmapDefinition), "CreateBitmapOverride");
        }

        protected abstract IBitmap CreateBitmapOverride(IBitmapDefinition bitmapDefinition);

        public IBitmap CreateBitmap(int width, int height, Stream dataStream)
        {
            if (dataStream == null)
                throw new ArgumentNullException("dataStream");

            if (dataStream.CanSeek)
            {
                var data = new byte[dataStream.Length];
                dataStream.Read(data, 0, data.Length);
                return CreateBitmap(width, height, data);
            }

            var ms = new MemoryStream();
            var buffer = new byte[4 * 1024 * 1024]; // 4 MB
            int length;

            do
            {
                length = dataStream.Read(buffer, 0, buffer.Length);
                ms.Write(buffer, 0, length);
            } while (length == buffer.Length);

            return CreateBitmap(width, height, ms.ToArray());
        }

        #endregion // CreateBitmap

        #region CreateBrushes

        public ISolidColorBrush CreateSolidColorBrush(IColor color)
        {
            if (color == null)
                throw new ArgumentNullException("color");

            return CheckReturn(CreateSolidColorBrushOverride(color), "CreateSolidColorBrushOverride");
        }

        public ILinearGradientBrush CreateLinearGradientBrush(IPoint startPoint, IPoint endPoint, params IGradientStop[] gradientStops)
        {
            if (startPoint == null)
                throw new ArgumentNullException("startPoint");
            if (endPoint == null)
                throw new ArgumentNullException("endPoint");
            if (gradientStops == null)
                throw new ArgumentNullException("gradientStops");

            return CheckReturn(CreateLinearGradientBrushOverride(startPoint, endPoint, gradientStops), "CreateLinearGradientBrushOverride");
        }

        public IRadialGradientBrush CreateRadialGradientBrush(IPoint center, IPoint gradientOriginOffset, double radiusX, double radiusY, params IGradientStop[] gradientStops)
        {
            if (center == null)
                throw new ArgumentNullException("center");
            if (gradientOriginOffset == null)
                throw new ArgumentNullException("gradientOriginOffset");
            if (gradientStops == null)
                throw new ArgumentNullException("gradientStops");

            return CheckReturn(CreateRadialGradientBrushOverride(center, gradientOriginOffset, radiusX, radiusY, gradientStops), "CreateRadialGradientBrushOverride");
        }

        protected abstract ISolidColorBrush CreateSolidColorBrushOverride(IColor color);
        protected abstract ILinearGradientBrush CreateLinearGradientBrushOverride(IPoint startPoint, IPoint endPoint, params IGradientStop[] gradientStops);
        protected abstract IRadialGradientBrush CreateRadialGradientBrushOverride(IPoint center, IPoint gradientOriginOffset, double radiusX, double radiusY, params IGradientStop[] gradientStops);

        #endregion // CreateBrushes

        #region DrawBitmap

        public void DrawBitmap(IBitmap bitmap, IRectangle destinationRectangle, IRectangle sourceRectangle, double opacity)
        {
            DrawBitmap(bitmap, destinationRectangle, sourceRectangle, opacity, BitmapInterpolationMode.NearestNeighbor);
        }

        public void DrawBitmap(IBitmap bitmap, IRectangle destinationRectangle, IRectangle sourceRectangle)
        {
            DrawBitmap(bitmap, destinationRectangle, sourceRectangle, 1.0);
        }

        public void DrawBitmap(IBitmap bitmap, IRectangle destinationRectangle)
        {
            DrawBitmap(bitmap, destinationRectangle, new Rectangle(bitmap.Size));
        }

        public void DrawBitmap(IBitmap bitmap, IRectangle destinationRectangle, IRectangle sourceRectangle, double opacity, BitmapInterpolationMode interpolationMode)
        {
            if (bitmap == null)
                throw new ArgumentNullException("bitmap");
            if (destinationRectangle == null)
                throw new ArgumentNullException("destinationRectangle");
            if (sourceRectangle == null)
                throw new ArgumentNullException("sourceRectangle");

            DrawBitmapOverride(bitmap, destinationRectangle, sourceRectangle, opacity, interpolationMode);
        }

        protected abstract void DrawBitmapOverride(IBitmap bitmap, IRectangle destinationRectangle, IRectangle sourceRectangle, double opacity, BitmapInterpolationMode interpolationMode);

        #endregion // DrawBitmap

        #region MeasureText

        public ISize MeasureText(string text, ITextFormat textFormat)
        {
            if (text == null)
                throw new ArgumentNullException("text");
            if (textFormat == null)
                throw new ArgumentNullException("textFormat");

            return CheckReturn(MeasureTextOverride(text, textFormat), "MeasureTextOverride");
        }

        protected abstract ISize MeasureTextOverride(string text, ITextFormat textFormat);

        #endregion // MeasureText

        #region DrawText

        public void DrawText(string text, ITextFormat textFormat, IRectangle drawTextArea, IBrush brush)
        {
            DrawText(text, textFormat, drawTextArea, brush, DrawTextOptions.None);
        }

        public void DrawText(string text, ITextFormat textFormat, IPoint position, ISize size, IBrush brush)
        {
            DrawText(text, textFormat, new Rectangle(position, size), brush, DrawTextOptions.None);
        }

        public void DrawText(string text, ITextFormat textFormat, IPoint position, IBrush brush)
        {
            DrawText(text, textFormat, new Rectangle(position, new Size(100000.0, 100000.0)), brush, DrawTextOptions.None);
        }

        public void DrawText(string text, ITextFormat textFormat, IRectangle drawTextArea, IBrush brush, DrawTextOptions options)
        {
            if (text == null)
                return;
            if (textFormat == null)
                throw new ArgumentNullException("textFormat");
            if (drawTextArea == null)
                throw new ArgumentNullException("drawTextArea");
            if (brush == null)
                throw new ArgumentNullException("brush");

            DrawTextOverride(text, textFormat, drawTextArea, brush, options);
        }

        protected abstract void DrawTextOverride(string text, ITextFormat textFormat, IRectangle drawTextArea, IBrush brush, DrawTextOptions options);

        public void DrawText(string text, ITextFormat textFormat, IPoint position, ISize size, IBrush brush, DrawTextOptions options)
        {
            DrawText(text, textFormat, new Rectangle(position, size), brush, options);
        }

        public void DrawText(string text, ITextFormat textFormat, IPoint position, IBrush brush, DrawTextOptions options)
        {
            DrawText(text, textFormat, new Rectangle(position, new Size()), brush, options);
        }

        #endregion // DrawText

        #region DrawEllipse

        public void DrawEllipse(IEllipse ellipse, IBrush brush, double strokeWidth)
        {
            DrawEllipse(ellipse, brush, strokeWidth, StrokeStyle.Default);
        }

        public void DrawEllipse(IEllipse ellipse, IBrush brush)
        {
            DrawEllipse(ellipse, brush, 1.0, StrokeStyle.Default);
        }

        public void DrawEllipse(IEllipse ellipse, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            if (ellipse == null)
                throw new ArgumentNullException("ellipse");
            if (brush == null)
                throw new ArgumentNullException("brush");
            if (strokeStyle == null)
                throw new ArgumentNullException("strokeStyle");

            DrawEllipseOverride(ellipse, brush, strokeWidth, strokeStyle);
        }

        protected abstract void DrawEllipseOverride(IEllipse ellipse, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle);

        public void DrawEllipse(IPoint center, double radiusX, double radiusY, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawEllipse(new Ellipse { Center = center, RadiusX = radiusX, RadiusY = radiusY }, brush, strokeWidth, strokeStyle);
        }

        public void DrawEllipse(double centerX, double centerY, double radiusX, double radiusY, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawEllipse(new Ellipse { Center = new Point(centerX, centerY), RadiusX = radiusX, RadiusY = radiusY }, brush, strokeWidth, strokeStyle);
        }

        #endregion // DrawEllipse

        #region DrawLine

        public void DrawLine(IPoint point1, IPoint point2, IBrush brush, double strokeWidth)
        {
            DrawLine(point1, point2, brush, strokeWidth, StrokeStyle.Default);
        }

        public void DrawLine(IPoint point1, IPoint point2, IBrush brush)
        {
            DrawLine(point1, point2, brush, 1.0, StrokeStyle.Default);
        }

        public void DrawLine(IPoint point1, IPoint point2, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            if (point1 == null)
                throw new ArgumentNullException("point1");
            if (point2 == null)
                throw new ArgumentNullException("point2");
            if (brush == null)
                throw new ArgumentNullException("brush");
            if (strokeStyle == null)
                throw new ArgumentNullException("strokeStyle");

            DrawLineOverride(point1, point2, brush, strokeWidth, strokeStyle);
        }

        protected abstract void DrawLineOverride(IPoint point1, IPoint point2, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle);

        public void DrawLine(double point1X, double point1Y, double point2X, double point2Y, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawLine(new Point(point1X, point1Y), new Point(point2X, point2Y), brush, strokeWidth, strokeStyle);
        }

        #endregion // DrawLine

        #region DrawBezier

        public void DrawBezier(IPoint startPoint, IPoint controlPoint1, IPoint controlPoint2, IPoint endPoint, IBrush brush, double strokeWidth)
        {
            DrawBezier(startPoint, controlPoint1, controlPoint2, endPoint, brush, strokeWidth, StrokeStyle.Default);
        }

        public void DrawBezier(IPoint startPoint, IPoint controlPoint1, IPoint controlPoint2, IPoint endPoint, IBrush brush)
        {
            DrawBezier(startPoint, controlPoint1, controlPoint2, endPoint, brush, 1.0, StrokeStyle.Default);
        }

        public void DrawBezier(IPoint startPoint, IPoint controlPoint1, IPoint controlPoint2, IPoint endPoint, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            if (startPoint == null)
                throw new ArgumentNullException("startPoint");
            if (controlPoint1 == null)
                throw new ArgumentNullException("controlPoint1");
            if (controlPoint2 == null)
                throw new ArgumentNullException("controlPoint2");
            if (endPoint == null)
                throw new ArgumentNullException("endPoint");
            if (brush == null)
                throw new ArgumentNullException("brush");
            if (strokeStyle == null)
                throw new ArgumentNullException("strokeStyle");

            DrawBezierOverride(startPoint, controlPoint1, controlPoint2, endPoint, brush, strokeWidth, strokeStyle);
        }

        protected abstract void DrawBezierOverride(IPoint startPoint, IPoint controlPoint1, IPoint controlPoint2, IPoint endPoint, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle);

        public void DrawBezier(
            double startPointX, double startPointY,
            double controlPoint1X, double controlPoint1Y,
            double controlPoint2X, double controlPoint2Y,
            double endPointX, double endPointY,
            IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawBezier(
                new Point(startPointX, startPointY),
                new Point(controlPoint1X, controlPoint1Y),
                new Point(controlPoint2X, controlPoint2Y),
                new Point(endPointX, endPointY),
                brush, strokeWidth, strokeStyle);
        }

        #endregion // Bezier

        #region DrawRectangle

        public void DrawRectangle(IRectangle rectangle, IBrush brush, double strokeWidth)
        {
            DrawRectangle(rectangle, brush, strokeWidth, StrokeStyle.Default);
        }

        public void DrawRectangle(IRectangle rectangle, IBrush brush)
        {
            DrawRectangle(rectangle, brush, 1.0, StrokeStyle.Default);
        }

        public void DrawRectangle(IRectangle rectangle, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");
            if (brush == null)
                throw new ArgumentNullException("brush");
            if (strokeStyle == null)
                throw new ArgumentNullException("strokeStyle");

            DrawRectangleOverride(rectangle, brush, strokeWidth, strokeStyle);
        }

        protected abstract void DrawRectangleOverride(IRectangle rectangle, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle);

        public void DrawRectangle(IPoint position, ISize size, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawRectangle(new Rectangle(position, size), brush, strokeWidth, strokeStyle);
        }

        public void DrawRectangle(double x, double y, double width, double height, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawRectangle(new Rectangle(x, y, width, height), brush, strokeWidth, strokeStyle);
        }

        #endregion // DrawRectangle

        #region DrawRoundedRectangle

        public void DrawRoundedRectangle(IRoundedRectangle rectangle, IBrush brush, double strokeWidth)
        {
            DrawRoundedRectangle(rectangle, brush, strokeWidth, StrokeStyle.Default);
        }

        public void DrawRoundedRectangle(IRoundedRectangle rectangle, IBrush brush)
        {
            DrawRoundedRectangle(rectangle, brush, 1.0, StrokeStyle.Default);
        }

        public void DrawRoundedRectangle(IRoundedRectangle rectangle, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");
            if (brush == null)
                throw new ArgumentNullException("brush");
            if (strokeStyle == null)
                throw new ArgumentNullException("strokeStyle");

            DrawRoundedRectangleOverride(rectangle, brush, strokeWidth, strokeStyle);
        }

        protected abstract void DrawRoundedRectangleOverride(IRoundedRectangle rectangle, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle);

        public void DrawRoundedRectangle(double x, double y, double width, double height, double radiusX, double radiusY, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawRoundedRectangle(new RoundedRectangle(x, y, width, height, radiusX, radiusY), brush, strokeWidth, strokeStyle);
        }

        public void DrawRoundedRectangle(IPoint position, ISize size, double radiusX, double radiusY, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawRoundedRectangle(new RoundedRectangle(position, size, radiusX, radiusY), brush, strokeWidth, strokeStyle);
        }

        public void DrawRoundedRectangle(IRectangle rectangle, double radiusX, double radiusY, IBrush brush, double strokeWidth, IStrokeStyle strokeStyle)
        {
            DrawRoundedRectangle(new RoundedRectangle(rectangle, radiusX, radiusY), brush, strokeWidth, strokeStyle);
        }

        public void DrawRoundedRectangle(IRectangle rectangle, double radiusX, double radiusY, IBrush brush)
        {
            DrawRoundedRectangle(new RoundedRectangle(rectangle, radiusX, radiusY), brush);
        }

        #endregion // DrawRoundedRectangle

        #region FillEllipse

        public void FillEllipse(IEllipse ellipse, IBrush brush)
        {
            if (ellipse == null)
                throw new ArgumentNullException("ellipse");
            if (brush == null)
                throw new ArgumentNullException("brush");

            FillEllipseOverride(ellipse, brush);
        }

        protected abstract void FillEllipseOverride(IEllipse ellipse, IBrush brush);

        public void FillEllipse(IPoint center, double radiusX, double radiusY, IBrush brush)
        {
            FillEllipse(new Ellipse { Center = center, RadiusX = radiusX, RadiusY = radiusY }, brush);
        }

        public void FillEllipse(double centerX, double centerY, double radiusX, double radiusY, IBrush brush)
        {
            FillEllipse(new Ellipse { Center = new Point(centerX, centerY), RadiusX = radiusX, RadiusY = radiusY }, brush);
        }

        #endregion // FillEllipse

        #region FillRectangle

        public void FillRectangle(IRectangle rectangle, IBrush brush)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");
            if (brush == null)
                throw new ArgumentNullException("brush");

            FillRectangleOverride(rectangle, brush);
        }

        protected abstract void FillRectangleOverride(IRectangle rectangle, IBrush brush);

        public void FillRectangle(IPoint position, ISize size, IBrush brush)
        {
            FillRectangle(new Rectangle(position, size), brush);
        }

        public void FillRectangle(double x, double y, double width, double height, IBrush brush)
        {
            FillRectangle(new Rectangle(x, y, width, height), brush);
        }

        #endregion // FillRectangle

        #region FillRoundedRectangle

        public void FillRoundedRectangle(IRoundedRectangle rectangle, IBrush brush)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");
            if (brush == null)
                throw new ArgumentNullException("brush");

            FillRoundedRectangleOverride(rectangle, brush);
        }

        protected abstract void FillRoundedRectangleOverride(IRoundedRectangle rectangle, IBrush brush);

        public void FillRoundedRectangle(double x, double y, double width, double height, double radiusX, double radiusY, IBrush brush)
        {
            FillRoundedRectangle(new RoundedRectangle(x, y, width, height, radiusX, radiusY), brush);
        }

        public void FillRoundedRectangle(IPoint position, ISize size, double radiusX, double radiusY, IBrush brush)
        {
            FillRoundedRectangle(new RoundedRectangle(position, size, radiusX, radiusY), brush);
        }

        public void FillRoundedRectangle(IRectangle rectangle, double radiusX, double radiusY, IBrush brush)
        {
            FillRoundedRectangle(new RoundedRectangle(rectangle, radiusX, radiusY), brush);
        }

        public void FillRoundedRectangle(IRectangle rectangle, double radiusX, double radiusY, CornerStyle style, IBrush brush)
        {
            FillRoundedRectangle(new RoundedRectangle(rectangle, radiusX, radiusY, style), brush);
        }

        #endregion // FillRoundedRectangle

        private static T CheckReturn<T>(T returnValue, string methodName) where T : class
        {
            if (returnValue == null)
                throw new InvalidOperationException(string.Format(Messages.EXCEPTION_MUST_NOT_RETURN_NULL, methodName));
            return returnValue;
        }
    }
}
