﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectDefinitions;

namespace EffectDefinitionsUnitTests
{
    [TestClass]
    public class EffectDefinitionsContainerTest
    {
        private static string basePath = "../../../../../TestData/UnitTests/Definitions/";
        private static string tempPath = Path.GetTempPath();

        private void CheckNumberOfErrors(EffectDefinitionsContainer defs, int expectedNumberOfErrors)
        {
            int numberOfErrors = defs.Errors.Count();
            if (numberOfErrors != expectedNumberOfErrors)
            {
                if (expectedNumberOfErrors == 0)
                {
                    foreach (var error in defs.Errors)
                    {
                        throw new Exception(error.Message);
                    }
                }
                else
                {
                    string errorMessages = string.Join(Environment.NewLine,
                        defs.Errors
                        .Where(x => string.IsNullOrEmpty(x.Message) == false)
                        .Select(x => x.Message));
                    throw new Exception(string.Format("Number of errors should be {0} but returned {1}:{2}",
                                                      expectedNumberOfErrors,
                                                      numberOfErrors,
                                                      Environment.NewLine + errorMessages));
                }
            }
        }

        private void CheckNumberOfDefinitions(EffectDefinitionsContainer defs,
                                             int expectedNumberOfFunctions,
                                             int expectedNumberOfBlocks,
                                             int expectedNumberOfTaggings,
                                             int expectedNumberOfTags)
        {
            int numberOfFunctions = defs.Functions.Count();
            if (numberOfFunctions != expectedNumberOfFunctions)
                throw new Exception(string.Format("Number of functions should be {0} but returned {1}",
                                                   expectedNumberOfFunctions,
                                                   numberOfFunctions));

            int numberOfBlocks = defs.Blocks.Count();
            if (numberOfBlocks != expectedNumberOfBlocks)
                throw new Exception(string.Format("Number of blocks should be {0} but returned {1}",
                                                  expectedNumberOfBlocks,
                                                  numberOfBlocks));

            int numberOfTaggings = defs.Taggings.Count();
            if (numberOfTaggings != expectedNumberOfTaggings)
                throw new Exception(string.Format("Number of taggings should be {0} but returned {1}",
                                                  expectedNumberOfTaggings,
                                                  numberOfTaggings));

            int numberOfTags = defs.Tags.Count();
            if (numberOfTags != expectedNumberOfTags)
                throw new Exception(string.Format("Number of tags should be {0} but returned {1}",
                                                  expectedNumberOfTags,
                                                  numberOfTags));
        }

        #region Number of elements tests

        [TestMethod]
        public void EmptyDefsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "empty-defs.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 0, 0, 0, 0);
        }

        [TestMethod]
        public void OneFunctionTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "just-one-function.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 1, 0, 0, 0);
        }

        [TestMethod]
        public void OneBlockTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "just-one-block.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 0, 1, 0, 0);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void TwoTagsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "just-two-tags.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 0, 0, 4, 2);
        }

        [TestMethod]
        public void OneFunctionWithTagsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "just-one-function-with-tags.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 1, 0, 2, 2);
        }

        [TestMethod]
        public void OneBlockTestWithOneFunction()
        {
            string[] paths = { Path.GetFullPath(basePath + "just-one-block-with-one-function.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfDefinitions(defs, 1, 1, 0, 0);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        #region Malformed XML tests

        [TestMethod]
        public void EmptyFileTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "empty.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedXmlClosingMissingTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-xml-closing-missing.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedXmlClosingTypo1Test()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-xml-closing-typo-1.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedXmlClosingTypo2Test()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-xml-closing-typo-2.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedXmlClosingTypo3Test()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-xml-closing-typo-3.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedXmlUnknownNode()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-xml-unknown-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            // This error is not fatal: the content should be loaded
            CheckNumberOfDefinitions(defs, 1, 1, 4, 2);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        #region Malformed function pathLoadInfo tests

        [TestMethod]
        public void MalformedFunctionMissingIdTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-id.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionMissingNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionDuplicateDescriptionNodeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-duplicate-description-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionMissingArgNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-arg-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionMissingArgTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-arg-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionMissingReturnTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-return-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionDuplicateReturnNodeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-duplicate-return-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionMissingTagNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-missing-tag-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionUnknownNodeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-unknown-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            // This error is not fatal: the function should be loaded
            CheckNumberOfDefinitions(defs, 1, 0, 0, 0);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedFunctionBouletPoweredTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-function-boulet-powered.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 11);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        #region Malformed block pathLoadInfo tests

        [TestMethod]
        public void MalformedBlockNoNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-no-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedBlockNoFuncidTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-no-funcid.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedBlockDuplicateFuncidTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-duplicate-funcid.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // One duplicate
            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();

            // Two functions to bind
            CheckNumberOfErrors(defs, 2);
        }

        [TestMethod]
        public void MalformedBlockNoArgTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-no-arg-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedBlockNoArgDisplayNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-no-arg-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedBlockNoArgTargetTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-no-arg-target.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedBlockUnknownNodeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-unknown-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            // This error is not fatal: the block should be loaded
            CheckNumberOfDefinitions(defs, 0, 1, 0, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // The block should complain about the missing function
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void MalformedBlockBouletPoweredTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-block-boulet-powered.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 10);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        [TestMethod]
        public void MalformedDefinitionUnknownAttributesTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-definition-unknown-attributes.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 8);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedDefinitionDuplicateAttributeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-definition-duplicate-attribute.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #region Malformed tag tests

        [TestMethod]
        public void MalformedTagMissingNameTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-tag-missing-name.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedTagMissingFuncIdTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-tag-missing-funcid.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedTagEmptyFuncIdTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-tag-empty-funcid.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedTagUnknownNodeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-tag-unknown-node.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 1);

            // This error is not fatal: the tag should be loaded
            CheckNumberOfDefinitions(defs, 0, 0, 3, 1);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedTaggingBouletPoweredTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "malformed-tag-boulet-powered.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 3);

            defs.ClearErrors();
            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        #region Duplicate tests

        [TestMethod]
        public void CorrectDefPoolTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedDefPoolDuplicateFunctionIDTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-duplicate-function-id.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // Two function id duplicates
            CheckNumberOfErrors(defs, 2);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedDefPoolDuplicateBlockGUIDTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-duplicate-block-guid.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // Two block guid duplicates
            CheckNumberOfErrors(defs, 2);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void CorrectDefPoolWithSemanticsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-with-semantics.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void CorrectDefPoolWithDuplicateInputSemanticsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-with-semantics-duplicate-input-semantic.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedDefPoolDuplicateOutputSemanticsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-with-semantics-duplicate-output-semantic.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // Three semantic with the same name
            CheckNumberOfErrors(defs, 3);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void MalformedDefPoolDuplicateOutputSemanticsFromSameBlockTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-with-semantics-duplicate-output-semantic-same-block.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // Two semantics with the same name
            CheckNumberOfErrors(defs, 2);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        #endregion

        #region Binding tests

        [TestMethod]
        public void BrokenBindingFunctionIDMissingTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-function-id-missing.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One missing function
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void BrokenBindingDuplicateFunctionIdTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-duplicate-function-id.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            // One duplicate binding
            CheckNumberOfErrors(defs, 1);

            defs.ClearErrors();
            defs.UpdateBindings();

            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void BrokenBindingUnmatchedFunctionInputTargetTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-unmatched-function-input-target.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One broken binding
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void BrokenBindingUnmatchedFunctionOutputTargetTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-unmatched-function-output-target.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One broken binding
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void BrokenBindingIncompatibleFunctionInputTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-incompatible-function-input-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One broken binding
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void BrokenBindingIncompatibleFunctionOutputTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-incompatible-function-output-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One broken binding
            CheckNumberOfErrors(defs, 1);
        }

        [TestMethod]
        public void BrokenBindingIncompatibleFunctionReturnTypeTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "broken-binding-incompatible-function-return-type.edml") };

            var defs = new EffectDefinitionsContainer();
            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);

            defs.ClearErrors();
            defs.UpdateBindings();

            // One broken binding
            CheckNumberOfErrors(defs, 1);
        }

        #endregion

        #region Reloading tests

        #region Event handlers

        private int directoryExplored;
        private void OnDirectoryExploringFinished(object source, FileIOEventArgs args)
        {
            ++directoryExplored;
        }


        private int functionDefinitionsAdded;
        private void OnFunctionDefinitionsAdded(object source, FunctionDefinitionEventArgs args)
        {
            functionDefinitionsAdded += args.Functions.Count();
        }

        private int functionDefinitionsUpdated;
        private void OnFunctionDefinitionsUpdated(object source, FunctionDefinitionEventArgs args)
        {
            functionDefinitionsUpdated += args.Functions.Count();
        }

        private int functionDefinitionsRemoved;
        private void OnFunctionDefinitionsRemoved(object source, FunctionDefinitionEventArgs args)
        {
            functionDefinitionsRemoved += args.Functions.Count();
        }


        private int blockDefinitionsAdded;
        private void OnBlockDefinitionsAdded(object source, BlockDefinitionEventArgs args)
        {
            blockDefinitionsAdded += args.Blocks.Count();
        }

        private int blockDefinitionsUpdated;
        private void OnBlockDefinitionsUpdated(object source, BlockDefinitionEventArgs args)
        {
            blockDefinitionsUpdated += args.Blocks.Count();
        }

        private int blockDefinitionsRemoved;
        private void OnBlockDefinitionsRemoved(object source, BlockDefinitionEventArgs args)
        {
            blockDefinitionsRemoved += args.Blocks.Count();
        }


        private int taggingsAdded;
        private void OnTaggingsAdded(object source, TaggingEventArgs args)
        {
            taggingsAdded += args.Taggings.Count();
        }

        private int taggingsRemoved;
        private void OnTaggingsRemoved(object source, TaggingEventArgs args)
        {
            taggingsRemoved += args.Taggings.Count();
        }


        private void SetHandlersToContainer(EffectDefinitionsContainer defs)
        {
            defs.DirectoryExploringFinished += OnDirectoryExploringFinished;

            defs.FunctionDefinitionsAdded += OnFunctionDefinitionsAdded;
            defs.FunctionDefinitionsUpdated += OnFunctionDefinitionsUpdated;
            defs.FunctionDefinitionsRemoved += OnFunctionDefinitionsRemoved;

            defs.BlockDefinitionsAdded += OnBlockDefinitionsAdded;
            defs.BlockDefinitionsUpdated += OnBlockDefinitionsUpdated;
            defs.BlockDefinitionsRemoved += OnBlockDefinitionsRemoved;

            defs.TaggingsAdded += OnTaggingsAdded;
            defs.TaggingsRemoved += OnTaggingsRemoved;
        }

        private void ClearEventsCount()
        {
            directoryExplored = 0;

            functionDefinitionsAdded = 0;
            functionDefinitionsUpdated = 0;
            functionDefinitionsRemoved = 0;

            blockDefinitionsAdded = 0;
            blockDefinitionsUpdated = 0;
            blockDefinitionsRemoved = 0;

            taggingsAdded = 0;
            taggingsRemoved = 0;
        }

        private void CheckNumberOfEvents(int expectedDirectoryExplored,
            int expectedFunctionDefinitionsAdded,
            int expectedFunctionDefinitionsUpdated,
            int expectedFunctionDefinitionsRemoved,
            int expectedBlockDefinitionsAdded,
            int expectedBlockDefinitionsUpdated,
            int expectedBlockDefinitionsRemoved,
            int expectedTaggingsAdded,
            int expectedTaggingsRemoved)
        {
            if (directoryExplored != expectedDirectoryExplored)
                throw new Exception(string.Format("Number of directory explored should be {0} but returned {1}",
                                                   expectedDirectoryExplored,
                                                   directoryExplored));


            if (functionDefinitionsAdded != expectedFunctionDefinitionsAdded)
                throw new Exception(string.Format("Number of functions added should be {0} but returned {1}",
                                                   expectedFunctionDefinitionsAdded,
                                                   functionDefinitionsAdded));

            if (functionDefinitionsUpdated != expectedFunctionDefinitionsUpdated)
                throw new Exception(string.Format("Number of functions updated should be {0} but returned {1}",
                                                   expectedFunctionDefinitionsUpdated,
                                                   functionDefinitionsUpdated));

            if (functionDefinitionsRemoved != expectedFunctionDefinitionsRemoved)
                throw new Exception(string.Format("Number of functions removed should be {0} but returned {1}",
                                                   expectedFunctionDefinitionsRemoved,
                                                   functionDefinitionsRemoved));


            if (blockDefinitionsAdded != expectedBlockDefinitionsAdded)
                throw new Exception(string.Format("Number of blocks added should be {0} but returned {1}",
                                                   expectedBlockDefinitionsAdded,
                                                   blockDefinitionsAdded));

            if (blockDefinitionsUpdated != expectedBlockDefinitionsUpdated)
                throw new Exception(string.Format("Number of blocks updated should be {0} but returned {1}",
                                                   expectedBlockDefinitionsUpdated,
                                                   blockDefinitionsUpdated));

            if (blockDefinitionsRemoved != expectedBlockDefinitionsRemoved)
                throw new Exception(string.Format("Number of blocks removed should be {0} but returned {1}",
                                                   expectedBlockDefinitionsRemoved,
                                                   blockDefinitionsRemoved));


            if (taggingsAdded != expectedTaggingsAdded)
                throw new Exception(string.Format("Number of taggings added should be {0} but returned {1}",
                                                   expectedTaggingsAdded,
                                                   taggingsAdded));

            if (taggingsRemoved != expectedTaggingsRemoved)
                throw new Exception(string.Format("Number of taggings removed should be {0} but returned {1}",
                                                   expectedTaggingsRemoved,
                                                   taggingsRemoved));
        }

        #endregion

        [TestMethod]
        public void LoadDefsEventsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml") };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void ReloadSameDefsTest()
        {
            string[] paths = { Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml") };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 0, 0, 0, 0, 0, 0, 0, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);
        }

        [TestMethod]
        public void ReloadNoDefsTest()
        {
            string originalPath = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml");
            string destPath = Path.GetFullPath(tempPath + "test.edml");

            File.Copy(originalPath, destPath, true);

            string[] paths = { destPath };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            File.Delete(destPath);

            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 0, 0, 5, 0, 0, 6, 0, 16);
            CheckNumberOfDefinitions(defs, 0, 0, 0, 0);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);

            File.Delete(destPath);
        }

        [TestMethod]
        public void ReloadMoreDefsTest()
        {
            string originalPath = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml");
            string originalPath2 = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags-version-more-of-it.edml");
            string destPath = Path.GetFullPath(tempPath + "test.edml");

            File.Copy(originalPath, destPath, true);

            string[] paths = { destPath };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            File.Delete(destPath);
            File.Copy(originalPath2, destPath);

            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 1, 0, 0, 2, 0, 0, 5, 0);
            CheckNumberOfDefinitions(defs, 6, 8, 21, 6);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);

            File.Delete(destPath);
        }

        [TestMethod]
        public void ReloadLessDefsTest()
        {
            string originalPath = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml");
            string originalPath2 = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags-version-less-of-it.edml");
            string destPath = Path.GetFullPath(tempPath + "test.edml");

            File.Copy(originalPath, destPath, true);

            string[] paths = { destPath };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            File.Delete(destPath);
            File.Copy(originalPath2, destPath);

            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 0, 0, 1, 0, 0, 2, 0, 6);
            CheckNumberOfDefinitions(defs, 4, 4, 10, 4);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);

            File.Delete(destPath);
        }

        [TestMethod]
        public void ReloadDifferentDefsTest()
        {
            string originalPath = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml");
            string originalPath2 = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags-version-different.edml");
            string destPath = Path.GetFullPath(tempPath + "test.edml");

            File.Copy(originalPath, destPath, true);

            string[] paths = { destPath };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            File.Delete(destPath);
            File.Copy(originalPath2, destPath);

            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 0, 5, 0, 0, 1, 0, 5, 5);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);

            File.Delete(destPath);
        }

        [TestMethod]
        public void ReloadDefsTest()
        {
            string originalPath = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags.edml");
            string originalPath2 = Path.GetFullPath(basePath + "group-of-functions-blocks-and-tags-version-with-various-changes.edml");
            string destPath = Path.GetFullPath(tempPath + "test.edml");

            File.Copy(originalPath, destPath, true);

            string[] paths = { destPath };

            var defs = new EffectDefinitionsContainer();
            SetHandlersToContainer(defs);
            ClearEventsCount();

            defs.SetDefinitionsPaths(CancellationToken.None, paths);

            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 5, 0, 0, 6, 0, 0, 16, 0);
            CheckNumberOfDefinitions(defs, 5, 6, 16, 5);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);


            File.Delete(destPath);
            File.Copy(originalPath2, destPath);

            ClearEventsCount();
            defs.ReloadDefinitions(CancellationToken.None);
            CheckNumberOfErrors(defs, 0);
            CheckNumberOfEvents(0, 1, 3, 0, 1, 3, 1, 7, 5);
            CheckNumberOfDefinitions(defs, 6, 6, 18, 6);

            defs.UpdateBindings();
            CheckNumberOfErrors(defs, 0);

            File.Delete(destPath);
        }

        #endregion
    }
}
