﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace EffectDefinitions
{
    /// <summary>
    /// Definition of a block: it has a name, a GUID, some input and output plugs, and the functions it represents.
    /// </summary>
    public class BlockDefinition
    {
        public BlockDefinition(DefinitionLocation source,
            string name,
            string guid,
            string group,
            ICollection<Plug> inputPlugs,
            ICollection<Plug> outputPlugs,
            UniformDefinition uniform,
            PreviewDefinition preview,
            params FunctionBinding[] functionBindings)
        {
            if (string.IsNullOrWhiteSpace(name))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "name"), "name");
            if (string.IsNullOrWhiteSpace(guid))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "guid"), "guid");
            if (inputPlugs == null)
                throw new ArgumentException("inputPlugs");
            if (outputPlugs == null)
                throw new ArgumentException("outputPlugs");
            if (preview == null)
                throw new ArgumentException("preview");

            // コメントノードは入出力ともに持たないので許容するように変更
            ////if (inputPlugs.Count + outputPlugs.Count == 0)
            ////    throw new ArgumentException(Messages.EXCEPTION_AT_LEAST_ONE_PLUG);

            this.Source = source;

            this.Name = name;
            this.Guid = guid;
            this.Group = group;
            this.InputPlugs = inputPlugs.ToArray();
            this.OutputPlugs = outputPlugs.ToArray();
            this.Uniform = uniform;
            this.Preview = preview;
            this.FunctionBindings = functionBindings;
        }

        public DefinitionLocation Source { get; private set; }

        public string Name { get; private set; }
        public string SubName { get; set; }
        public string Guid { get; private set; }
        public string Group { get; private set; }
        public FunctionBinding[] FunctionBindings { get; internal set; }
        public Plug[] InputPlugs { get; private set; }
        public Plug[] OutputPlugs { get; private set; }

        /// <summary>
        /// uniform 変数情報を取得します。
        /// </summary>
        public UniformDefinition Uniform { get; private set; }

        /// <summary>
        /// プレビュー情報を取得します。
        /// </summary>
        public PreviewDefinition Preview { get; private set; }

        public FunctionDefinition CurrentFunctionDefinition
        {
            get
            {
                if (this.FunctionBindings.Length == 0)
                {
                    return null;
                }

                return this.FunctionBindings[0].Definition;
            }
        }

        public bool IsBound
        {
            get
            {
                return this.FunctionBindings.Length > 0 &&
                    this.FunctionBindings.All(x => x.Definition != null);
            }
        }

        public string Description { get { return IsBound ? CurrentFunctionDefinition.Description : string.Empty; } }

        /// <summary>
        /// Tags of the active function definition.
        /// </summary>
        public virtual IEnumerable<string> Tags
        {
            get
            {
                return IsBound ? CurrentFunctionDefinition.Tags : Enumerable.Empty<string>();
            }
        }

        public bool HasSamePlugsTypesAs(BlockDefinition blockDef)
        {
            if (InputPlugs.Length != blockDef.InputPlugs.Length ||
                OutputPlugs.Length != blockDef.OutputPlugs.Length)
            {
                return false;
            }

            for (int i = 0; i < InputPlugs.Length; ++i)
            {
                if (InputPlugs[i].Type != blockDef.InputPlugs[i].Type)
                {
                    return false;
                }
            }

            for (int i = 0; i < OutputPlugs.Length; ++i)
            {
                if (OutputPlugs[i].Type != blockDef.OutputPlugs[i].Type)
                {
                    return false;
                }
            }

            return true;
        }

        public bool BindToFunctionDefinition(FunctionDefinition definition)
        {
            if (definition == null)
                throw new ArgumentNullException("definition");

            if (FunctionBindings.Length == 0)
                return false;

            var index = Array.FindIndex(this.FunctionBindings, x => x.Id == definition.Id);
            if (index < 0)
                return false;

            var inputParams = definition.Parameters
                .Where(x =>
                    x.Direction == ShaderTyping.ParameterDirection.Input ||
                    x.Direction == ShaderTyping.ParameterDirection.Reference);

            var outputParams = definition.Parameters
                .Where(x =>
                    x.Direction == ShaderTyping.ParameterDirection.Output ||
                    x.Direction == ShaderTyping.ParameterDirection.Reference);

            bool plugged =
                this.BindPlugsToArguments(this.InputPlugs, definition.ReturnType, inputParams) &&
                this.BindPlugsToArguments(this.OutputPlugs, definition.ReturnType, outputParams);

            if (plugged)
            {
                FunctionBindings[index].Definition = definition;
            }

            return plugged;
        }

        private bool BindPlugsToArguments(Plug[] plugs, ShaderTyping.ShaderTypeDefinition returnType, IEnumerable<ShaderTyping.ParameterDefinition> arguments)
        {
            var pluggedArguments = new List<ShaderTyping.ParameterDefinition>();

            bool pluggedAllPlugs = true;
            foreach (var plug in plugs)
            {
                if (plug.IsMachingReturnValue(returnType))
                    continue;

                bool plugged = false;
                foreach (var argument in arguments)
                {
                    if (plug.IsMatchingParameter(argument))
                    {
                        if (!pluggedArguments.Contains(argument))
                        {
                            pluggedArguments.Add(argument);
                        }

                        plugged = true;
                        break;
                    }
                }

                pluggedAllPlugs &= plugged;
            }

            bool pluggedAllArguments = (pluggedArguments.Count == arguments.Count());

            return pluggedAllPlugs && pluggedAllArguments;
        }

        public override string ToString()
        {
            return string.Format("{0} -> {1} ({2})",
                this.Name, string.Join(", ", this.FunctionBindings.Select(x => x.Id)), IsBound ? "bound" : "not bound");
        }

        #region Equals

        public override bool Equals(object obj)
        {
            if (obj is FunctionDefinition)
            {
                return Equals((FunctionDefinition)obj);
            }

            return false;
        }

        public virtual bool Equals(BlockDefinition other)
        {
            if ((object)other == null)
            {
                return false;
            }

            if (!(other is BlockDefinition))
            {
                return false;
            }

            return this.Name == other.Name &&
                this.Guid == other.Guid &&
                this.InputPlugs.SequenceEqual(other.InputPlugs) &&
                this.OutputPlugs.SequenceEqual(other.OutputPlugs) &&
                this.FunctionBindings.Select(x => x.Id).SequenceEqual(other.FunctionBindings.Select(x => x.Id));
        }

        public override int GetHashCode()
        {
            return (this.Name +
                this.Guid.ToString() +
                string.Concat(this.InputPlugs.Select(x => x.GetHashCode())) +
                string.Concat(this.OutputPlugs.Select(x => x.GetHashCode()))).GetHashCode();
        }

        public static bool operator ==(BlockDefinition t1, BlockDefinition t2)
        {
            if ((object)t1 == null)
            {
                return (object)t2 == null;
            }

            return t1.Equals(t2);
        }

        public static bool operator !=(BlockDefinition t1, BlockDefinition t2)
        {
            return !(t1 == t2);
        }

        #endregion

        #region Compatibility

        private class PlugEqualityComparer : IEqualityComparer<Plug>
        {
            public bool Equals(Plug x, Plug y)
            {
                return x.Type.Equals(y.Type);
            }

            public int GetHashCode(Plug obj)
            {
                return obj.GetHashCode();
            }
        }

        public bool AreInputPlugsSameAs(BlockDefinition otherDef)
        {
            return this.InputPlugs.Length == otherDef.InputPlugs.Length &&
                this.InputPlugs.SequenceEqual(otherDef.InputPlugs, new PlugEqualityComparer());
        }

        public bool AreOutputPlugsSameAs(BlockDefinition otherDef)
        {
            return this.OutputPlugs.Length == otherDef.OutputPlugs.Length &&
                this.OutputPlugs.SequenceEqual(otherDef.OutputPlugs, new PlugEqualityComparer());
        }

        #endregion
    }
}
