﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Usage.Descriptors;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using EffectDefinitions;
using EffectCombiner.Core.Extensions;
using Workflow.Core;

namespace EffectCombinerUnitTests
{
    [TestClass]
    public class ColorUsageDescriptorTest
    {
        [TestMethod]
        public void InputSubPlugBaseTargetTest()
        {
            var colorUsage = new ColorUsageDescriptor();
            var plug = Utility.CreateInputPlug("color", "vec4");
            var subPlugs = colorUsage.ExpandInput(plug);

            foreach (var subPlug in subPlugs)
                if (subPlug.BaseTarget != plug.BaseTarget)
                {
                    throw new Exception(string.Format("Sub-plug should have \"{0}\" as a base target but has \"{1}\"", plug.BaseTarget, subPlug.BaseTarget));
                }
        }

        [TestMethod]
        public void OutputSubPlugBaseTargetTest()
        {
            var colorUsage = new ColorUsageDescriptor();
            var plug = Utility.CreateOutputPlug("color", "vec4");
            var subPlugs = colorUsage.ExpandOutput(plug);

            foreach (var subPlug in subPlugs)
                if (subPlug.BaseTarget != plug.BaseTarget)
                {
                    throw new Exception(string.Format("Sub-plug should have \"{0}\" as a base target but has \"{1}\"", plug.BaseTarget, subPlug.BaseTarget));
                }
        }

        [TestMethod]
        public void InputExpandToOutputTest()
        {
            var colorUsage = new ColorUsageDescriptor();
            var plug = Utility.CreateInputPlug("color", "vec4");

            SubPlug[] subPlugs = null;
            try
            {
                subPlugs = colorUsage.ExpandOutput(plug);
            }
            catch
            {
                return;
            }
            if (subPlugs.IsNullOrEmpty() == false)
                throw new Exception("Calling ExpandOutput on an input plug was expected to fail");
        }

        [TestMethod]
        public void OutputExpandToInputTest()
        {
            var colorUsage = new ColorUsageDescriptor();
            var outputPlug = Utility.CreateOutputPlug("color", "vec4");

            SubPlug[] subPlugs = null;
            try
            {
                subPlugs = colorUsage.ExpandInput(outputPlug);
            }
            catch
            {
                return;
            }
            if (subPlugs.IsNullOrEmpty() == false)
                throw new Exception("Calling ExpandInput on an output plug was expected to fail");
        }

        #region Plug/block tools

        private static void CreateColorOutputBlock(out ColorUsageDescriptor colorUsage, out EffectOutputPlug outputPlug, out EffectOutputPlug[] outputSubPlugs)
        {
            colorUsage = new ColorUsageDescriptor();

            var plugRGBA = Utility.CreateOutputPlug("rgba", "vec4");
            var subPlugs = colorUsage.ExpandOutput(plugRGBA);
            var blockElement = Utility.CreateDummyOutputBlock(new Plug[] { plugRGBA }.Concat(subPlugs).ToArray());
            outputPlug = new EffectOutputPlug(plugRGBA, blockElement);
            outputSubPlugs = subPlugs.Select(x => new EffectOutputPlug(x, blockElement)).ToArray();
        }

        private static void CreateColorInputBlock(out ColorUsageDescriptor colorUsage, out EffectInputPlug inputPlug, out EffectInputPlug[] inputSubPlugs)
        {
            colorUsage = new ColorUsageDescriptor();

            var plugRGBA = Utility.CreateInputPlug("rgba", "vec4");
            var subPlugs = colorUsage.ExpandInput(plugRGBA);
            var blockElement = Utility.CreateDummyInputBlock(new Plug[] { plugRGBA }.Concat(subPlugs).ToArray());
            inputPlug = new EffectInputPlug(plugRGBA, blockElement);
            inputSubPlugs = subPlugs.Select(x => new EffectInputPlug(x, blockElement)).ToArray();
        }

        private static void CreateRgbaRgbRGBAInputBlock(out EffectInputPlug inputPlugRGBA, out EffectInputPlug inputPlugRGB, out EffectInputPlug inputPlugR, out EffectInputPlug inputPlugG, out EffectInputPlug inputPlugB, out EffectInputPlug inputPlugA)
        {
            var plugRGBA = Utility.CreateInputPlug("rgba", "vec4");
            var plugRGB = Utility.CreateInputPlug("rgb", "vec3");
            var plugR = Utility.CreateInputPlug("r", "float");
            var plugG = Utility.CreateInputPlug("g", "float");
            var plugB = Utility.CreateInputPlug("b", "float");
            var plugA = Utility.CreateInputPlug("a", "float");
            var blockElement = Utility.CreateDummyInputBlock(plugRGBA, plugRGB, plugR, plugG, plugB, plugA);
            inputPlugRGBA = new EffectInputPlug(plugRGBA, blockElement);
            inputPlugRGB = new EffectInputPlug(plugRGB, blockElement);
            inputPlugR = new EffectInputPlug(plugR, blockElement);
            inputPlugG = new EffectInputPlug(plugG, blockElement);
            inputPlugB = new EffectInputPlug(plugB, blockElement);
            inputPlugA = new EffectInputPlug(plugA, blockElement);
        }

        private static void CreateRgbaRgbRGBAOutputBlock(out EffectOutputPlug outputPlugRGBA, out EffectOutputPlug outputPlugRGB, out EffectOutputPlug outputPlugR, out EffectOutputPlug outputPlugG, out EffectOutputPlug outputPlugB, out EffectOutputPlug outputPlugA)
        {
            var plugRGBA = Utility.CreateOutputPlug("rgba", "vec4");
            var plugRGB = Utility.CreateOutputPlug("rgb", "vec3");
            var plugR = Utility.CreateOutputPlug("r", "float");
            var plugG = Utility.CreateOutputPlug("g", "float");
            var plugB = Utility.CreateOutputPlug("b", "float");
            var plugA = Utility.CreateOutputPlug("a", "float");
            var blockElement = Utility.CreateDummyOutputBlock(plugRGBA, plugRGB, plugR, plugG, plugB, plugA);
            outputPlugRGBA = new EffectOutputPlug(plugRGBA, blockElement);
            outputPlugRGB = new EffectOutputPlug(plugRGB, blockElement);
            outputPlugR = new EffectOutputPlug(plugR, blockElement);
            outputPlugG = new EffectOutputPlug(plugG, blockElement);
            outputPlugB = new EffectOutputPlug(plugB, blockElement);
            outputPlugA = new EffectOutputPlug(plugA, blockElement);
        }

        #endregion

        #region AreOutputsProperlyConnected

        [TestMethod]
        public void AreOutputsProperlyConnectedRGBATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputPlug, remoteRGBAInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedRGB_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteRGBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[4], remoteAInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedRGB_NotATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteRGBInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedRGB_R_G_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteRGBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[1], remoteRInputPlug);
            ConnectionManager.Connect(outputSubPlugs[2], remoteGInputPlug);
            ConnectionManager.Connect(outputSubPlugs[3], remoteBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[4], remoteAInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedRGB_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteRGBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[3], remoteBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[4], remoteAInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedR_G_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[1], remoteRInputPlug);
            ConnectionManager.Connect(outputSubPlugs[2], remoteGInputPlug);
            ConnectionManager.Connect(outputSubPlugs[3], remoteBInputPlug);
            ConnectionManager.Connect(outputSubPlugs[4], remoteAInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedR_G_B_A_ReplugTest()
        {
            ColorUsageDescriptor colorUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateColorOutputBlock(out colorUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteRGBAInputPlug;
            EffectInputPlug remoteRGBInputPlug;
            EffectInputPlug remoteRInputPlug;
            EffectInputPlug remoteGInputPlug;
            EffectInputPlug remoteBInputPlug;
            EffectInputPlug remoteAInputPlug;
            CreateRgbaRgbRGBAInputBlock(out remoteRGBAInputPlug, out remoteRGBInputPlug, out remoteRInputPlug, out remoteGInputPlug, out remoteBInputPlug, out remoteAInputPlug);

            ConnectionManager.Connect(outputSubPlugs[1], remoteRInputPlug);
            ConnectionManager.Connect(outputSubPlugs[2], remoteRInputPlug);
            ConnectionManager.Connect(outputSubPlugs[3], remoteRInputPlug);
            ConnectionManager.Connect(outputSubPlugs[4], remoteRInputPlug);

            Utility.OutputsShouldBeProperlyConnected(colorUsage, outputPlug, outputSubPlugs);
        }

        #endregion

        #region AreInputsProperlyConnected

        [TestMethod]
        public void AreInputsProperlyConnectedRGBATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteRGBAOutputPlug, inputPlug);

            Utility.InputsShouldBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedRGB_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteRGBOutputPlug, inputSubPlugs[0]);
            ConnectionManager.Connect(remoteAOutputPlug, inputSubPlugs[4]);

            Utility.InputsShouldBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedRGB_NotATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteRGBOutputPlug, inputSubPlugs[0]);

            Utility.InputsShouldNotBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedRGB_R_G_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteRGBOutputPlug, inputSubPlugs[0]);
            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[1]);
            ConnectionManager.Connect(remoteGOutputPlug, inputSubPlugs[2]);
            ConnectionManager.Connect(remoteBOutputPlug, inputSubPlugs[3]);
            ConnectionManager.Connect(remoteAOutputPlug, inputSubPlugs[4]);

            Utility.InputsShouldNotBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedRGB_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteRGBOutputPlug, inputSubPlugs[0]);
            ConnectionManager.Connect(remoteBOutputPlug, inputSubPlugs[3]);
            ConnectionManager.Connect(remoteAOutputPlug, inputSubPlugs[4]);

            Utility.InputsShouldNotBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedR_G_B_ATest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[1]);
            ConnectionManager.Connect(remoteGOutputPlug, inputSubPlugs[2]);
            ConnectionManager.Connect(remoteBOutputPlug, inputSubPlugs[3]);
            ConnectionManager.Connect(remoteAOutputPlug, inputSubPlugs[4]);

            Utility.InputsShouldBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedR_G_B_A_ReplugTest()
        {
            ColorUsageDescriptor colorUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateColorInputBlock(out colorUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteRGBAOutputPlug;
            EffectOutputPlug remoteRGBOutputPlug;
            EffectOutputPlug remoteROutputPlug;
            EffectOutputPlug remoteGOutputPlug;
            EffectOutputPlug remoteBOutputPlug;
            EffectOutputPlug remoteAOutputPlug;
            CreateRgbaRgbRGBAOutputBlock(out remoteRGBAOutputPlug, out remoteRGBOutputPlug, out remoteROutputPlug, out remoteGOutputPlug, out remoteBOutputPlug, out remoteAOutputPlug);

            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[1]);
            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[2]);
            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[3]);
            ConnectionManager.Connect(remoteROutputPlug, inputSubPlugs[4]);

            Utility.InputsShouldNotBeProperlyConnected(colorUsage, inputPlug, inputSubPlugs);
        }

        #endregion
    }
}
