﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define USE_BLINKING

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Threading.Tasks;
using System.Threading;

namespace EffectCombiner.Primitives
{
    public static class QuickInfoManager
    {
        public static void Initialize(Control quickInfoControl)
        {
            Globals.SetQuickInfo(new QuickInfo(quickInfoControl));
        }
    }

    public class QuickInfo
    {
        private readonly Control control;
        private System.Threading.Timer quickInfoTimer;
        private Color quickInfoOriginalForeColor;
        private Color quickInfoOriginalBackColor;

        public QuickInfo(Control control)
        {
            if (control == null)
                throw new ArgumentNullException("control");
            if (control.Parent == null)
                throw new ArgumentException(Messages.EXCEPTION_ROOT_OR_UNPARENTED_CHILD_CONTROL_FORBIDDEN, "control");

            this.control = control;
        }

        public void SetQuickInfo(string message, bool autoClear, Color? color = null)
        {
            if (control.Parent == null)
                throw new InvalidOperationException(Messages.EXCEPTION_ROOT_OR_UNPARENTED_CHILD_CONTROL_FORBIDDEN);

            if (quickInfoTimer == null)
            {
                quickInfoTimer = new System.Threading.Timer(HideQuickInfo, null, -1, -1);
                quickInfoOriginalForeColor = control.ForeColor;
                quickInfoOriginalBackColor = control.BackColor;
            }

            var workingColor = color != null ? color.Value : quickInfoOriginalForeColor;

#if USE_BLINKING
            var blinkColor = Color.FromArgb(255, 255 - workingColor.R, 255 - workingColor.G, 255 - workingColor.B);
#endif // USE_BLINKING

            control.Invoke((Action)delegate
            {
                control.Visible = true;
                control.Text = message;
                control.Left = control.Parent.ClientSize.Width - control.Width - 6;
                control.ForeColor = workingColor;
#if !USE_BLINKING
                if (autoClear)
                    quickInfoTimer.Change(2000, -1);
#endif // USE_BLINKING
            });

#if USE_BLINKING
            Task.Factory.StartNew(() =>
                {
                    for (var i = 0; i < 10; i++)
                    {
                        control.Invoke((Action)delegate { control.BackColor = blinkColor; });
                        Thread.Sleep(100);
                        control.Invoke((Action)delegate { control.BackColor = quickInfoOriginalBackColor; });
                        Thread.Sleep(100);
                    }
                })
                .ContinueWith(t =>
                {
                    control.Invoke((Action)delegate
                    {
                        control.ForeColor = workingColor;
                        if (autoClear)
                            quickInfoTimer.Change(2000, -1);
                    });
                });
#endif // USE_BLINKING
        }

        private void HideQuickInfo(object state)
        {
            control.BeginInvoke((Action)delegate
            {
                control.ForeColor = Color.Transparent;
                control.Visible = false;
            });
        }
    }
}
