﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Blocks.Core;
using EffectDefinitions;
using Renderer2D.Core;
using Renderer2D.Core.WinForms;
using EffectCombiner.Primitives.Blocks.BlockRenderers;
using EffectCombiner.Primitives.Extensions;
using EffectCombiner.Primitives.Generation;

namespace EffectCombiner.Primitives.Blocks
{
    public class WorkspaceRendering
    {
        private const double GridSpacing = 128.0;

        private readonly IWorkspaceManager workspaceManager;

        private readonly RegularBlockRenderer blockRenderer;

        public WorkspaceRendering(IWorkspaceManager workspaceManager)
        {
            if (workspaceManager == null)
                throw new ArgumentNullException("workspaceManager");

            this.workspaceManager = workspaceManager;
            blockRenderer = new RegularBlockRenderer();
        }

        public void RenderDraggingBlocks(Renderer renderer)
        {
            var mouseInputManager = workspaceManager.MouseInputManager;

            if (mouseInputManager.DraggingBlocks == null)
            {
                return;
            }

            double x = mouseInputManager.DraggingPosition.X;
            double y = mouseInputManager.DraggingPosition.Y;

            var bri = new BlockRenderInfo
            {
                IsUsable = false,
            };

            foreach (var block in mouseInputManager.DraggingBlocks)
            {
                var size = blockRenderer.Measure(renderer, bri, block, null);
                blockRenderer.Render(renderer, bri, block, new Rectangle(new Point(x, y), size));
                x += 32.0;
                y += 32.0;
            }
        }

        public void RenderGrid(Renderer renderer)
        {
            var mtx = (renderer.Transform ?? Matrix.Identity);
            var origin = mtx.TransformPoint(0.0, 0.0);

            var zoomedGridSpacing = GridSpacing * workspaceManager.Zoom;

            var startX = origin.X % zoomedGridSpacing;
            var startY = origin.Y % zoomedGridSpacing;

            var originalMatrix = renderer.Transform;
            renderer.Transform = Matrix.Identity;

            var width = workspaceManager.ViewportSize.Width;
            var height = workspaceManager.ViewportSize.Height;

            for (var x = startX; x < width; x += zoomedGridSpacing)
                renderer.DrawLine(new Point(x, 0.0), new Point(x, height), Globals.VisualResources.GridLineBrush);
            for (var y = startY; y < height; y += zoomedGridSpacing)
                renderer.DrawLine(new Point(0.0, y), new Point(width, y), Globals.VisualResources.GridLineBrush);

            renderer.Transform = originalMatrix;
        }

        #region Links rendering

        /// <summary>
        /// 接続済みのエッジを描画します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        public void RenderLinks(Renderer renderer)
        {
            foreach (EffectBlockElementBase blockElement in workspaceManager.BlockManager.BlockElements.Where(b => b.IsVisible))
            {
                foreach (EffectOutputPlug outputPlug in blockElement.WorkflowItem.OutputPlugs)
                {
                    foreach (EffectInputPlug remoteInputPlug in outputPlug.RemoteInputPlugs)
                    {
                        var startPoint = outputPlug.BlockElement.BlockRenderInfo.OutputPlugPositions[outputPlug.Index].ToRendererPoint();
                        var liftOffset1 = (outputPlug.BlockElement.IsDragging ? Globals.VisualResources.RegularDraggedBlockLiftOffset : 0);

                        startPoint = new Point(
                            startPoint.X + outputPlug.BlockElement.Left - liftOffset1,
                            startPoint.Y + outputPlug.BlockElement.Top - liftOffset1);

                        var endPoint = remoteInputPlug.BlockElement.BlockRenderInfo.InputPlugPositions[remoteInputPlug.Index].ToRendererPoint();
                        var liftOffset2 = (remoteInputPlug.BlockElement.IsDragging ? Globals.VisualResources.RegularDraggedBlockLiftOffset : 0);

                        endPoint = new Point(
                            endPoint.X + remoteInputPlug.BlockElement.Left - liftOffset2,
                            endPoint.Y + remoteInputPlug.BlockElement.Top - liftOffset2);

                        IBrush lineBrush;

                        if (outputPlug.BlockElement.BlockRenderInfo.IsPartOfCircularReference &&
                            remoteInputPlug.BlockElement.BlockRenderInfo.IsPartOfCircularReference)
                        {
                            lineBrush = Globals.VisualResources.CircularReferenceBrush;
                        }
                        else if (workspaceManager.MouseInputManager.IsDisconnecting &&
                            workspaceManager.MouseInputManager.DisconnectingSource.Identifier == outputPlug.Identifier &&
                            workspaceManager.MouseInputManager.DisconnectingTarget.Identifier == remoteInputPlug.Identifier)
                        {
                            lineBrush = Globals.VisualResources.DisconnectingLineBrush;
                        }
                        else
                        {
                            lineBrush = Globals.VisualResources.ConnectionLineBrush;
                        }

                        RenderLink(renderer, startPoint, endPoint, lineBrush);
                    }
                }
            }
        }

        /// <summary>
        /// 接続中のエッジを描画します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        public void RenderConnectingLink(Renderer renderer)
        {
            var mouseInputManager = workspaceManager.MouseInputManager;

            if (mouseInputManager.IsConnecting == false)
                return;

            IPoint startPoint;
            IPoint endPoint;
            if (mouseInputManager.BlocksConnectionInfo.PlugType == PlugHitTestResult.Input)
            {
                startPoint = mouseInputManager.CurrentMousePosition;
                if (mouseInputManager.MouseSnap != null)
                {
                    var startBlock = mouseInputManager.MouseSnap.BlockElement;
                    var plugType = mouseInputManager.MouseSnap.PlugType;
                    var index = mouseInputManager.MouseSnap.PlugIndex;
                    if (plugType == PlugHitTestResult.Input)
                    {
                        startPoint = new Point(startBlock.Left + startBlock.BlockRenderInfo.InputPlugPositions[index].X,
                            startBlock.Top + startBlock.BlockRenderInfo.InputPlugPositions[index].Y);
                    }
                    else
                    {
                        startPoint = new Point(startBlock.Left + startBlock.BlockRenderInfo.OutputPlugPositions[index].X,
                            startBlock.Top + startBlock.BlockRenderInfo.OutputPlugPositions[index].Y);
                    }
                }

                endPoint = mouseInputManager.BlocksConnectionInfo.BlockElement.BlockRenderInfo.InputPlugPositions[mouseInputManager.BlocksConnectionInfo.PlugIndex].ToRendererPoint();
                endPoint = new Point(
                    endPoint.X + mouseInputManager.BlocksConnectionInfo.BlockElement.Left,
                    endPoint.Y + mouseInputManager.BlocksConnectionInfo.BlockElement.Top);
            }
            else
            {
                startPoint = mouseInputManager.BlocksConnectionInfo.BlockElement.BlockRenderInfo.OutputPlugPositions[mouseInputManager.BlocksConnectionInfo.PlugIndex].ToRendererPoint();
                startPoint = new Point(
                    startPoint.X + mouseInputManager.BlocksConnectionInfo.BlockElement.Left,
                    startPoint.Y + mouseInputManager.BlocksConnectionInfo.BlockElement.Top);

                endPoint = mouseInputManager.CurrentMousePosition;
                if (mouseInputManager.MouseSnap != null)
                {
                    var endBlock = mouseInputManager.MouseSnap.BlockElement;
                    var plugType = mouseInputManager.MouseSnap.PlugType;
                    var index = mouseInputManager.MouseSnap.PlugIndex;
                    if (plugType == PlugHitTestResult.Input)
                    {
                        endPoint = new Point(endBlock.Left + endBlock.BlockRenderInfo.InputPlugPositions[index].X,
                            endBlock.Top + endBlock.BlockRenderInfo.InputPlugPositions[index].Y);
                    }
                    else
                    {
                        endPoint = new Point(endBlock.Left + endBlock.BlockRenderInfo.OutputPlugPositions[index].X,
                            endBlock.Top + endBlock.BlockRenderInfo.OutputPlugPositions[index].Y);
                    }
                }
            }

            IBrush lineBrush;

            if (mouseInputManager.IsMakingCircularReference || mouseInputManager.IsInvalidPlug)
                lineBrush = Globals.VisualResources.CircularReferenceBrush;
            else if (mouseInputManager.IsValidPlug)
                lineBrush = Globals.VisualResources.ValidConnectionBrush;
            else if (mouseInputManager.IsConnecting)
                lineBrush = Globals.VisualResources.DraggedConnectionBrush;
            else
                lineBrush = Globals.VisualResources.ConnectionLineBrush;

            RenderLink(renderer, startPoint, endPoint, lineBrush);
        }

        private static void RenderLink(Renderer renderer, IPoint startPoint, IPoint endPoint, IBrush lineBrush)
        {
            IPoint controlPoint1;
            IPoint controlPoint2;
            ComputeBezierPoints(startPoint, endPoint, out controlPoint1, out controlPoint2);

            renderer.DrawBezier(startPoint, controlPoint1, controlPoint2, endPoint,
                                lineBrush, Globals.VisualResources.ConnectionLineStroke);
        }

        private const double TangentOffsetRatio = 0.5;

        private static void ComputeBezierPoints(IPoint startPoint, IPoint endPoint, out IPoint controlPoint1, out IPoint controlPoint2)
        {
            var dx = endPoint.X - startPoint.X;
            var dy = endPoint.Y - startPoint.Y;

            var tangentOffset = Math.Sqrt(dx * dx + dy * dy) * TangentOffsetRatio;

            controlPoint1 = new Point(startPoint.X + tangentOffset, startPoint.Y);
            controlPoint2 = new Point(endPoint.X - tangentOffset, endPoint.Y);
        }

        #endregion

        /// <summary>
        /// ブロックサイズを計算します。
        /// </summary>
        /// <param name="renderer">レンダラ</param>
        public void CalculateBlockSize(Renderer renderer)
        {
            var blockManager = this.workspaceManager.BlockManager;

            lock (blockManager.BlocksSyncRoot)
            {
                for (var i = blockManager.BlockElements.Count - 1; i >= 0; i--)
                {
                    var blockElement = (EffectBlockElementBase)blockManager.BlockElements[i];
                    ISize blockSize = this.blockRenderer.Measure(
                                        renderer,
                                        blockElement.BlockRenderInfo,
                                        blockElement.BlockDefinition,
                                        new Rectangle(new Point(blockElement.Left, blockElement.Top), new Size(blockElement.Width, blockElement.Height)));

                    blockElement.SetComputedSize(blockSize.Width, blockSize.Height);
                }
            }
        }

        /// <summary>
        /// ブロックを描画します。
        /// </summary>
        /// <param name="renderer">レンダラー</param>
        public void RenderBlocks(Renderer renderer)
        {
            var blockManager = this.workspaceManager.BlockManager;

            lock (blockManager.BlocksSyncRoot)
            {
                for (var i = blockManager.BlockElements.Count - 1; i >= 0; i--)
                {
                    var blockElement = (EffectBlockElementBase)blockManager.BlockElements[i];

                    if (blockElement.IsVisible)
                    {
                        blockElement.OnBeforeRendering();

                        var rect = new Rectangle(new Point(blockElement.Left, blockElement.Top), new Size(blockElement.Width, blockElement.Height));
                        this.blockRenderer.Render(renderer, blockElement.BlockRenderInfo, blockElement.BlockDefinition, rect);

                        blockElement.OnAfterRendering();
                    }
                }
            }
        }

        /// <summary>
        /// ブロックの外枠を描画します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        public void RenderSideIndicators(Renderer renderer)
        {
            var viewportSize = workspaceManager.ViewportSize;

            if (viewportSize == null)
                return;

            var blockManager = workspaceManager.BlockManager;

            var topLeft = 0.0;
            var top = 0.0;
            var topRight = 0.0;
            var right = 0.0;
            var bottomRight = 0.0;
            var bottom = 0.0;
            var bottomLeft = 0.0;
            var left = 0.0;

            var mtx = (renderer.Transform ?? Matrix.Identity);

            lock (blockManager.BlocksSyncRoot)
            {
                foreach (var blockElement in blockManager.BlockElements.Where(b => b.IsVisible))
                {
                    var blockTop = TransformScalarY(mtx, blockElement.Top - FrameMargin);
                    var blockBottom = TransformScalarY(mtx, blockElement.Top + blockElement.Height + FrameMargin);
                    var blockRight = TransformScalarX(mtx, blockElement.Left + blockElement.Width + FrameMargin);
                    var blockLeft = TransformScalarX(mtx, blockElement.Left - FrameMargin);

                    if (blockBottom < 0.0)
                    {
                        if (blockRight < 0.0)
                            topLeft = Math.Max(topLeft, DistanceAlpha(0.0, 0.0, blockRight, blockBottom));
                        else if (blockLeft > viewportSize.Width)
                            topRight = Math.Max(topRight, DistanceAlpha(viewportSize.Width, 0.0, blockLeft, blockBottom));
                        else
                            top = Math.Max(top, DistanceAlpha(0.0, 0.0, 0.0, blockBottom));
                    }
                    else if (blockTop > viewportSize.Height)
                    {
                        if (blockRight < 0.0)
                            bottomLeft = Math.Max(bottomLeft, DistanceAlpha(0.0, viewportSize.Height, blockRight, blockTop));
                        else if (blockLeft > viewportSize.Width)
                            bottomRight = Math.Max(bottomRight, DistanceAlpha(viewportSize.Width, viewportSize.Height, blockLeft, blockTop));
                        else
                            bottom = Math.Max(bottom, DistanceAlpha(0.0, viewportSize.Height, 0.0, blockTop));
                    }
                    else if (blockRight < 0.0)
                        left = Math.Max(left, DistanceAlpha(0.0, 0.0, blockRight, 0.0));
                    else if (blockLeft > viewportSize.Width)
                        right = Math.Max(right, DistanceAlpha(viewportSize.Width, 0.0, blockLeft, 0.0));
                }
            }

            var topLeftBrush = GetAlphaBrush(renderer, topLeft);
            var topBrush = GetAlphaBrush(renderer, top);
            var topRightBrush = GetAlphaBrush(renderer, topRight);
            var rightBrush = GetAlphaBrush(renderer, right);
            var bottomRightBrush = GetAlphaBrush(renderer, bottomRight);
            var bottomBrush = GetAlphaBrush(renderer, bottom);
            var bottomLeftBrush = GetAlphaBrush(renderer, bottomLeft);
            var leftBrush = GetAlphaBrush(renderer, left);

            var originalMatrix = renderer.Transform;
            renderer.Transform = Matrix.Identity;

            // top left
            renderer.FillRectangle(new Rectangle(
                0.0,
                0.0,
                CornderIndicatorLength,
                IndicatorThickness),
                topLeftBrush);
            renderer.FillRectangle(new Rectangle(
                0.0,
                IndicatorThickness,
                IndicatorThickness,
                CornderIndicatorLength - IndicatorThickness),
                topLeftBrush);

            // top right
            renderer.FillRectangle(new Rectangle(
                viewportSize.Width - CornderIndicatorLength,
                0.0,
                CornderIndicatorLength,
                IndicatorThickness),
                topRightBrush);
            renderer.FillRectangle(new Rectangle(
                viewportSize.Width - IndicatorThickness,
                IndicatorThickness,
                IndicatorThickness,
                CornderIndicatorLength - IndicatorThickness),
                topRightBrush);

            // bottom left
            renderer.FillRectangle(new Rectangle(
                0.0,
                viewportSize.Height - IndicatorThickness,
                CornderIndicatorLength,
                IndicatorThickness),
                bottomLeftBrush);
            renderer.FillRectangle(new Rectangle(
                0.0,
                viewportSize.Height - CornderIndicatorLength,
                IndicatorThickness,
                CornderIndicatorLength - IndicatorThickness),
                bottomLeftBrush);

            // bottom right
            renderer.FillRectangle(new Rectangle(
                viewportSize.Width - CornderIndicatorLength,
                viewportSize.Height - IndicatorThickness,
                CornderIndicatorLength,
                IndicatorThickness),
                bottomRightBrush);
            renderer.FillRectangle(new Rectangle(
                viewportSize.Width - IndicatorThickness,
                viewportSize.Height - CornderIndicatorLength,
                IndicatorThickness,
                CornderIndicatorLength - IndicatorThickness),
                bottomRightBrush);

            renderer.FillRectangle(new Rectangle(
                CornderIndicatorLength,
                0.0,
                viewportSize.Width - CornderIndicatorLength * 2.0,
                IndicatorThickness),
                topBrush);

            renderer.FillRectangle(new Rectangle(
                0.0,
                CornderIndicatorLength,
                IndicatorThickness,
                viewportSize.Height - CornderIndicatorLength * 2.0),
                leftBrush);

            renderer.FillRectangle(new Rectangle(
                viewportSize.Width - IndicatorThickness,
                CornderIndicatorLength,
                IndicatorThickness,
                viewportSize.Height - CornderIndicatorLength * 2.0),
                rightBrush);

            renderer.FillRectangle(new Rectangle(
                CornderIndicatorLength,
                viewportSize.Height - IndicatorThickness,
                viewportSize.Width - CornderIndicatorLength * 2.0,
                IndicatorThickness),
                bottomBrush);

            renderer.Transform = originalMatrix;
        }

        /// <summary>
        /// 選択領域を描画します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        public void RenderRectangularSelection(Renderer renderer)
        {
            var mouseInputManager = workspaceManager.MouseInputManager;

            if (mouseInputManager.DraggingMode == DraggingMode.RectangularSelection)
            {
                var selectionRectangle = new Rectangle(mouseInputManager.MouseDownPosition, mouseInputManager.CurrentMousePosition);
                renderer.FillRectangle(selectionRectangle, Globals.VisualResources.RectangularSelectionBrush);
                renderer.DrawRectangle(selectionRectangle, Globals.VisualResources.RectangularSelectionBorderBrush);
            }
        }

        private IBrush GetAlphaBrush(Renderer renderer, double alpha)
        {
            var intAlpha = Math.Min(Math.Max(0, (int)(alpha * 255)), 255);
            var color = System.Drawing.Color.FromArgb(intAlpha, System.Drawing.Color.CornflowerBlue);
            return renderer.CreateSolidColorBrush(color.ToRendererColor());
        }

        private double DistanceAlpha(double x1, double y1, double x2, double y2)
        {
            var distance = Distance(x1, y1, x2, y2);
            return Math.Max(MinDistanceAlphaRatio, distance / MaxDistance);
        }

        private double Distance(double x1, double y1, double x2, double y2)
        {
            var dx = Math.Abs(x1 - x2);
            var dy = Math.Abs(y1 - y2);
            return Math.Sqrt(dx * dx + dy * dy);
        }

        private const double MaxDistance = 128.0;
        private const double MinDistanceAlphaRatio = 0.2;
        private const double FrameMargin = 8.0;

        private const double IndicatorThickness = 12.0;
        private const double CornderIndicatorLength = 64.0;

        private static double TransformScalarX(IMatrix matrix, double x)
        {
            return (x * matrix.M11) + matrix.M31;
        }

        private static double TransformScalarY(IMatrix matrix, double y)
        {
            return (y * matrix.M22) + matrix.M32;
        }
    }
}
