﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using OperationManager.Core;
using Blocks.Core;
using EffectCombiner.Core.Extensions;
using EffectCombiner.Primitives.Operations;
using EffectCombiner.Primitives.Blocks.BlockHitTesters;
using EffectCombiner.Primitives.Extensions;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using ShaderGenerator.GLSL;
using Workflow.Core;

namespace EffectCombiner.Primitives.Blocks
{
    public class CompositionBlockManager : BlockManagerBase
    {
        private readonly BlockElementGenerator blockElementGenerator = new BlockElementGenerator();
        private IWorkspaceManager workspaceManager;

        private readonly List<BlockElementBase> copiedElements = new List<BlockElementBase>();
        public IList<BlockElementBase> CopiedElements { get; private set; }

        private readonly Dictionary<uint, InputPlugInfo[]> copiedInputs = new Dictionary<uint, InputPlugInfo[]>();
        private readonly Dictionary<uint, OutputPlugInfo[]> copiedOutputs = new Dictionary<uint, OutputPlugInfo[]>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CompositionBlockManager()
        {
            CopiedElements = new ReadOnlyCollection<BlockElementBase>(copiedElements);
            ConnectionManager.EndWiringTransaction += ConnectionManagerEndWiringTransaction;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workspaceManager">WorkspaceManager</param>
        public CompositionBlockManager(IWorkspaceManager workspaceManager)
        {
            this.workspaceManager = workspaceManager;
            CopiedElements = new ReadOnlyCollection<BlockElementBase>(copiedElements);
            ConnectionManager.EndWiringTransaction += ConnectionManagerEndWiringTransaction;
        }

        public override BlockElementBase CreateBlockElement(BlockDefinition blockDefinition)
        {
            var blockElement = blockElementGenerator.CreateBlockElement(blockDefinition);
            OnBlockElementCreated(new BlockElementEventArgs(blockElement));

            return blockElement;
        }

        public event EventHandler<BlockElementEventArgs> BlockElementCreated;

        protected virtual void OnBlockElementCreated(BlockElementEventArgs e)
        {
            var be = (EffectBlockElementBase)e.BlockElement;

            be.SetHitTester(new BlockElementHitTester());
            be.SetupPlugPositions();

            // ブロックのサイズ変更に伴う当たり判定領域更新をイベントハンドラで処理する
            be.SizeChanged += (sender, args) => be.SetupPlugPositions();

            var handler = BlockElementCreated;
            if (handler != null)
                handler(this, e);
        }

        #region Events

        public class ShaderProducedEventArgs : EventArgs
        {
            public string SourceCode { get; private set; }
            public bool IsNewlyGenerated { get; private set; }
            public bool IsBuildForced { get; private set; }

            public ShaderProducedEventArgs(string sourceCode, bool isNewlyGenerated, bool isForced)
            {
                if (string.IsNullOrWhiteSpace(sourceCode))
                    throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "sourceCode"), "sourceCode");

                SourceCode = sourceCode;
                IsNewlyGenerated = isNewlyGenerated;
                IsBuildForced = isForced;
            }
        }

        public event EventHandler<ShaderProducedEventArgs> ShaderCodeProduced;

        protected virtual void OnShaderCodeProduced(ShaderProducedEventArgs e)
        {
            var handler = ShaderCodeProduced;
            if (handler != null)
                handler(this, e);
        }

        /// <summary>
        /// 描画要求イベントです。
        /// </summary>
        public event EventHandler RequestRender;

        /// <summary>
        /// 描画が要求されたときの処理を行います。
        /// </summary>
        protected virtual void OnRequestRender()
        {
            this.RequestRender?.Invoke(this, EventArgs.Empty);
        }

        /// <summary>
        /// 描画内容を無効化して再描画を促します。
        /// </summary>
        public void InvalidateRender()
        {
            this.OnRequestRender();
        }

        #endregion

        /// <summary>
        /// ブロックエレメントの constant 値が変更されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<ConstantValuesChangedEventArgs> BlockElementConstantValuesChanged;

        /// <summary>
        /// ブロックエレメントの constant 値が変更されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockElementConstantValuesChanged(ConstantValuesChangedEventArgs e)
        {
            this.BlockElementConstantValuesChanged?.Invoke(this, e);
        }

        /// <summary>
        /// 管理しているブロックエレメントから発生した ConstantValuesChanged イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementConstantValuesChangedHook(object sender, ConstantValuesChangedEventArgs e)
        {
            this.OnBlockElementConstantValuesChanged(e);
        }

        /// <summary>
        /// ブロックエレメントが追加されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnBlockAdded(BlockElementEventArgs e)
        {
            if (e.BlockElement is ConstantBlockElement)
            {
                ConstantBlockElement constantBlockElement = (ConstantBlockElement)e.BlockElement;
                constantBlockElement.ValuesChanged += this.BlockElementConstantValuesChangedHook;
            }

            base.OnBlockAdded(e);
        }

        /// <summary>
        /// ブロックエレメントが削除されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnBlockRemoved(BlockElementEventArgs e)
        {
            if (e.BlockElement is ConstantBlockElement)
            {
                ConstantBlockElement constantBlockElement = (ConstantBlockElement)e.BlockElement;
                constantBlockElement.ValuesChanged -= this.BlockElementConstantValuesChangedHook;
            }

            ConnectionManager.Disconnect(((EffectBlockElementBase)e.BlockElement).WorkflowItem);

            base.OnBlockRemoved(e);
        }

        /// <summary>
        /// ブロックエレメントが置換されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnBlockReplaced(BlockElementReplacedEventArgs e)
        {
            ConnectionManager.Disconnect(((EffectBlockElementBase)e.OldBlockElement).WorkflowItem);

            if (e.OldBlockElement is ConstantBlockElement)
            {
                ConstantBlockElement constantBlockElement = (ConstantBlockElement)e.OldBlockElement;
                constantBlockElement.ValuesChanged -= this.BlockElementConstantValuesChangedHook;
            }

            if (e.NewBlockElement is ConstantBlockElement)
            {
                ConstantBlockElement constantBlockElement = (ConstantBlockElement)e.NewBlockElement;
                constantBlockElement.ValuesChanged += this.BlockElementConstantValuesChangedHook;
            }

            base.OnBlockReplaced(e);
        }

        #region Shader generation

        private readonly object generationSyncRoot = new object();
        private bool isGenerating;

        private string[] latestProducedShaders;

        private int codeGenerationLockCount;

        public void BeginShaderCodeGenerationLock()
        {
            Interlocked.Increment(ref codeGenerationLockCount);
        }

        public void EndShaderCodeGenerationLock()
        {
            if (Interlocked.Decrement(ref codeGenerationLockCount) < 0)
                throw new InvalidOperationException(Messages.EXCEPTION_SHADER_GENERATION_LOCK_COUNT_MISMATCH);
        }

        /// <summary>
        /// シェーダコードの生成処理を非同期で行います。
        /// </summary>
        /// <param name="forceBuild">ビルドを強制するかどうか</param>
        /// <param name="forceCheckGenerateShader">シェーダ生成ができるかチェックするかどうか</param>
        public void RunShaderGenerationProcessAsync(bool forceBuild = false, bool forceCheckGenerateShader = false)
        {
            string shaderMainName = Globals.Options.ShaderGeneration.MainFunctionName;
            var task = ProduceShaderCodesAsync(true, forceCheckGenerateShader, shaderMainName);

            if (task == null)
                return;

            task.ContinueWith(t =>
            {
                foreach (var code in t.Result)
                    OnShaderCodeProduced(new ShaderProducedEventArgs(code, true, forceBuild));
            }, TaskScheduler.FromCurrentSynchronizationContext());
        }

        /// <summary>
        /// シェーダコードの生成処理を同期で行います。
        /// </summary>
        /// <param name="forceBuild">ビルドを強制するかどうか</param>
        /// <param name="forceCheckGenerateShader">シェーダ生成ができるかチェックするかどうか</param>
        public void RunShaderGenerationProcessSync(bool forceBuild = false, bool forceCheckGenerateShader = false)
        {
            string shaderMainName = Globals.Options.ShaderGeneration.MainFunctionName;
            if (codeGenerationLockCount > 0)
            {
                return;
            }

            latestProducedShaders = ProduceShaderCodesSync(true, forceCheckGenerateShader, shaderMainName);

            foreach (var code in latestProducedShaders)
            {
                OnShaderCodeProduced(new ShaderProducedEventArgs(code, true, forceBuild));
            }
        }

        /// <summary>
        /// 非同期でシェーダコードを生成します。
        /// </summary>
        /// <param name="addHeaderFiles">シェーダコードにヘッダーファイルを追加するかどうか</param>
        /// <param name="forceCheckGenerateShader">シェーダ生成ができるかチェックするかどうか</param>
        /// <param name="shaderMainName">シェーダコードのmain関数名</param>
        /// <returns>シェーダコードの生成タスクを返します。</returns>
        public Task<string[]> ProduceShaderCodesAsync(bool addHeaderFiles, bool forceCheckGenerateShader, string shaderMainName)
        {
            if (isGenerating || codeGenerationLockCount > 0)
                return null;

            lock (generationSyncRoot)
            {
                if (isGenerating || codeGenerationLockCount > 0)
                    return null;

                isGenerating = true;
            }

            return Task.Factory.StartNew(() =>
                {
                    try
                    {
                        latestProducedShaders = ProduceShaderCodesSync(addHeaderFiles, forceCheckGenerateShader, shaderMainName);
                        return latestProducedShaders;
                    }
                    finally
                    {
                        isGenerating = false;
                    }
                });
        }

        /// <summary>
        /// 同期でシェーダコードを生成します。
        /// </summary>
        /// <param name="addHeaderFiles">シェーダコードにヘッダーファイルを追加するかどうか</param>
        /// <param name="forceCheckGenerateShader">シェーダ生成ができるかチェックするかどうか</param>
        /// <param name="shaderMainName">シェーダコードのmain関数名</param>
        /// <returns>シェーダコードを返します。</returns>
        public string[] ProduceShaderCodesSync(bool addHeaderFiles, bool forceCheckGenerateShader, string shaderMainName)
        {
            EffectBlockElementBase[] coldBlocks;

            lock (BlocksSyncRoot)
            {
                coldBlocks = BlockElements.Cast<EffectBlockElementBase>().ToArray();
            }

            var allConnections = Globals.Options.ShaderGeneration.AllowShaderGenerationOnNotProperlyConnectedGraph == false;

            var query = from blockElement in coldBlocks
                        where blockElement.WorkflowItem.IsGraphOutput
                        let connectedGroup = blockElement.WorkflowItem.GetConnectedGroup(allConnections)
                        select RunConnectedGroupShaderGeneration(blockElement.WorkflowItem, connectedGroup.ToArray(), addHeaderFiles, forceCheckGenerateShader, shaderMainName)
                            into code
                        where string.IsNullOrWhiteSpace(code) == false
                        select code;

            return query.ToArray();
        }

        /// <summary>
        /// 接続情報からシェーダコードを生成します。
        /// </summary>
        /// <param name="outputWorkflowItem">出力ブロックのワークフローアイテム</param>
        /// <param name="connectedGroup">出力ブロックに接続されているワークフローアイテム情報</param>
        /// <param name="addHeaderFiles">シェーダコードにヘッダーファイルを追加するかどうか</param>
        /// <param name="forceCheckGenerateShader">シェーダ生成ができるかチェックするかどうか</param>
        /// <param name="shaderMainName">シェーダコードのmain関数名</param>
        /// <returns>シェーダコードを返します。</returns>
        private string RunConnectedGroupShaderGeneration(WorkflowItem<PlugValue> outputWorkflowItem, WorkflowItem<PlugValue>[] connectedGroup, bool addHeaderFiles, bool forceCheckGenerateShader, string shaderMainName)
        {
            if (connectedGroup
                .Cast<EffectWorkflowItem>()
                .Any(w => w.BlockElement.BlockDefinition.IsBound == false))
                return null;

            if (Globals.Options.ShaderGeneration.AllowShaderGenerationOnNotProperlyConnectedGraph == false ||
                forceCheckGenerateShader == true)
            {
                if (EffectWorkflowItem.IsGraphProperlyConnected(connectedGroup, false) == false)
                    return null;
            }

            ResourceLocator resourceLocator;
            lock (BlocksSyncRoot)
            {
                resourceLocator = new ResourceLocator(BlockElements
                    .SelectMany(be => be.BlockDefinition.FunctionBindings
                        .Select(fb => fb.Definition)
                        .Where(fd => fd != null))
                    .ToArray());
            }

            var glslGenerationContext = new GlslGenerationContext(resourceLocator, Globals.ContentRepository, shaderMainName)
            {
                DefinitionMode = DependencyMode.Insert,
            };

            foreach (var workflowItem in connectedGroup.Cast<EffectWorkflowItem>())
                workflowItem.SetGenerationContext(glslGenerationContext);

            glslGenerationContext.BeginGeneration();
            outputWorkflowItem.PollInputValues();
            glslGenerationContext.EndGeneration();

            if (string.IsNullOrWhiteSpace(glslGenerationContext.FullShader))
                return null;

            var sb = new StringBuilder();

            if (addHeaderFiles)
            {
                foreach (var p in Globals.Options.ShaderGeneration.LinkDefinitionsShaderFileNames)
                {
                    if (string.IsNullOrWhiteSpace(p.Path))
                        continue;
                    sb.AppendLine(Globals.ContentRepository.GetContent(new Uri(p.Path), Encoding.UTF8));
                }
            }

            sb.AppendLine(glslGenerationContext.FullShader);

            return sb.ToString();
        }

        public bool ReviewLastProducedShaders()
        {
            if (latestProducedShaders == null || latestProducedShaders.Length == 0)
                return false;

            foreach (var code in latestProducedShaders)
                OnShaderCodeProduced(new ShaderProducedEventArgs(code, false, false));

            return true;
        }

        #endregion

        public bool RemoveSelectedBlocks()
        {
            var removeBlockElements = BlockElements.Where(b => b.IsSelected).Cast<EffectBlockElementBase>();

            if (removeBlockElements.Any() == false)
            {
                return false;
            }

            var operation = new RemoveBlockOperation(this, removeBlockElements);

            BeginShaderCodeGenerationLock();

            Globals.MainOperationManager.Add(operation, true);

            EndShaderCodeGenerationLock();

            return true;
        }

        public void OnBlockDefinitionsDropped(double x, double y, BlockDefinition[] blockDefinitions)
        {
            if (blockDefinitions.Length == 0)
            {
                return;
            }

            BeginShaderCodeGenerationLock();

            try
            {
                var blockElements = new BlockElementBase[blockDefinitions.Length];

                for (int i = 0; i < blockDefinitions.Length; ++i)
                {
                    blockElements[i] = CreateBlockElement(blockDefinitions[i]);
                    blockElements[i].SetPosition(x, y);

                    x += 32.0;
                    y += 32.0;
                }

                var operation = new AddBlockOperation(this, blockElements);

                Globals.MainOperationManager.Add(operation, true);
            }
            catch (Exception ex)
            {
                // schedule exception rethrow from UI thread to UI thread O_o
                SynchronizationContext.Current.RunAsync(() => { throw ex; });
            }
            finally
            {
                EndShaderCodeGenerationLock();

                Globals.WorkspaceManager.InvalidateRender();
            }
        }

        public void CopyBlocks()
        {
            lock (BlocksSyncRoot)
            {
                copiedElements.Clear();
                var copiedDic = new Dictionary<uint, uint>();
                foreach (var item in SelectedItems)
                {
                    var clonedItem = CreateBlockElement(item.BlockDefinition);
                    copiedDic[item.InstanceIdentifier] = clonedItem.InstanceIdentifier;
                    clonedItem.SetPosition(item.Left, item.Top);

                    var constItem = clonedItem as ConstantBlockElement;
                    if (constItem != null)
                    {
                        var origConst = (ConstantBlockElement)item;
                        constItem.Values = (string[,])origConst.Values.Clone();
                    }

                    var commentItem = clonedItem as CommentBlockElement;
                    if (commentItem != null)
                    {
                        var origComment = (CommentBlockElement)item;
                        commentItem.Text = origComment.Text;
                        commentItem.SetComputedSize(item.Width, item.Height);
                    }

                    copiedElements.Add(clonedItem);
                }

                CopyConnections(copiedDic);
            }
        }

        public List<BlockElementBase> PasteBlocks()
        {
            var pastedList = new List<BlockElementBase>();
            var pastedDic = new Dictionary<uint, uint>();
            foreach (var item in copiedElements)
            {
                // ペーストのたびに右下へずれるようにする
                item.SetPosition(item.Left + 32, item.Top + 32);
                var clonedItem = CreateBlockElement(item.BlockDefinition);
                pastedDic[item.InstanceIdentifier] = clonedItem.InstanceIdentifier;
                clonedItem.SetPosition(item.Left, item.Top);

                var constItem = clonedItem as ConstantBlockElement;
                if (constItem != null)
                {
                    var origConst = (ConstantBlockElement)item;
                    constItem.Values = (string[,])origConst.Values.Clone();
                }

                var commentItem = clonedItem as CommentBlockElement;
                if (commentItem != null)
                {
                    var origComment = (CommentBlockElement)item;
                    commentItem.Text = origComment.Text;
                    commentItem.SetComputedSize(item.Width, item.Height);
                }

                pastedList.Add(clonedItem);
            }

            var operations = new List<OperationBase>();
            if (pastedList.Any())
            {
                {
                    var addBlockOperation = new AddBlockOperation(this, pastedList);
                    addBlockOperation.Execute();

                    operations.Add(addBlockOperation);
                }

                foreach (var item in copiedElements.OfType<EffectBlockElementBase>())
                {
                    var pastedItem = BlockElements.OfType<EffectBlockElementBase>().First(b => b.InstanceIdentifier == pastedDic[item.InstanceIdentifier]);

                    var myInputs = copiedInputs[item.InstanceIdentifier];
                    foreach (var input in myInputs)
                    {
                        var remoteItem = BlockElements.OfType<EffectBlockElementBase>().First(b => b.InstanceIdentifier == pastedDic[input.Remote.InstanceIdentifier]);

                        var pastedInputPlug = pastedItem.WorkflowItem.InputPlugs[input.Index];
                        var remotePlug = remoteItem.WorkflowItem.OutputPlugs[input.Remote.Index];

                        ConnectionManager.Connect(remotePlug, pastedInputPlug);
                        operations.Add(new ConnectionOperation(this, true, (EffectInputPlug)pastedInputPlug, (EffectOutputPlug)remotePlug));
                    }

                    var myOutputs = copiedOutputs[item.InstanceIdentifier];
                    foreach (var output in myOutputs)
                    {
                        foreach (var remote in output.Remotes)
                        {
                            var remoteItem = BlockElements.OfType<EffectBlockElementBase>().First(b => b.InstanceIdentifier == pastedDic[remote.InstanceIdentifier]);

                            var pastedOutputPlug = pastedItem.WorkflowItem.OutputPlugs[output.Index];
                            var remotePlug = remoteItem.WorkflowItem.InputPlugs[remote.Index];

                            ConnectionManager.Connect(pastedOutputPlug, remotePlug);
                            operations.Add(new ConnectionOperation(this, true, (EffectInputPlug)remotePlug, (EffectOutputPlug)pastedOutputPlug));
                        }
                    }
                }

                Globals.MainOperationManager.Add(new AggregateOperation(OperationType.PasteBlock, operations));
            }

            return pastedList;
        }

        private void CopyConnections(Dictionary<uint, uint> originalToCopiedMap)
        {
            copiedInputs.Clear();
            copiedOutputs.Clear();

            foreach (var item in SelectedItems.OfType<EffectBlockElementBase>())
            {
                // 入力接続情報を列挙
                {
                    // 入力接続情報を列挙する
                    var inputConnections = item.WorkflowItem.InputPlugs      // 入力プラグリストについて
                        .Cast<EffectInputPlug>()                             // IEnumerable<EffectInputPlug> にキャスト
                        .Where(p => p.BlockDefinitionPlug.Semantic == null)  // セマンティックが未指定のものを列挙して
                        .Select((p, i) => Tuple.Create(p, i))                // この並びでインデックスを付ける
                        .Where(t => t.Item1.RemoteOutputPlug != null &&      // 出力プラグに未接続のものは省く (インデックスにはカウントする)
                            this.SelectedItems.Any(s => s.InstanceIdentifier == t.Item1.BlockElement.InstanceIdentifier) &&                      // さらに、未選択のブロックを省く
                            originalToCopiedMap.ContainsKey(((EffectWorkflowItem)t.Item1.RemoteWorkflowItem).BlockElement.InstanceIdentifier));  // 出力先のブロックがコピーされていないものを省く

                    // 列挙した入力接続情報を InpugPlugInfo にまとめる
                    var inputPlugInfoArray = inputConnections.Select(t => new InputPlugInfo(
                        t.Item2,
                        new PlugInfo(
                            t.Item1.RemoteOutputPlug.Index,
                            originalToCopiedMap[((EffectWorkflowItem)t.Item1.RemoteWorkflowItem).BlockElement.InstanceIdentifier]
                        )
                    )).ToArray();

                    copiedInputs.Add(originalToCopiedMap[item.InstanceIdentifier], inputPlugInfoArray);
                }

                // 出力接続情報を列挙
                {
                    // 出力接続情報を列挙する
                    var outputConnections = item.WorkflowItem.OutputPlugs    // 出力プラグリストについて
                        .Cast<EffectOutputPlug>()                            // IEnumerable<EffectOutputPlug> にキャスト
                        .Where(p => p.Plug.Semantic == null)                 // セマンティックが未指定のものを列挙して
                        .Select((p, i) => Tuple.Create(p, i))                // この並びでインデックスを付ける
                        .Where(t => t.Item1.RemoteInputPlugs.Length > 0 &&   // 入力プラグに未接続のものは省く (インデックスにはカウントする)
                            this.SelectedItems.Any(s => s.InstanceIdentifier == t.Item1.BlockElement.InstanceIdentifier));  // さらに、未選択のブロックを省く

                    // 列挙した出力接続情報を OutputPlugInfo にまとめる
                    var outputPlugInfoArray = outputConnections.Select(t => new OutputPlugInfo(
                        t.Item2,
                        t.Item1.RemoteInputPlugs.Cast<EffectInputPlug>()
                            .Where(p => originalToCopiedMap.ContainsKey(p.BlockElement.InstanceIdentifier))
                            .Select(r => new PlugInfo(
                                r.Index,
                                originalToCopiedMap[r.BlockElement.InstanceIdentifier]
                            )).ToArray())
                    ).ToArray();

                    copiedOutputs.Add(originalToCopiedMap[item.InstanceIdentifier], outputPlugInfoArray);
                }
            }
        }

        private static bool CheckPlugShaderTypeMatch(BlockHitInfo plug1, BlockHitInfo plug2)
        {
            if (plug1 == null)
                throw new ArgumentNullException("plug1");
            if (plug2 == null)
                throw new ArgumentNullException("plug2");

            if (plug1.PlugType == PlugHitTestResult.None || plug2.PlugType == PlugHitTestResult.None)
                return false;

            if (plug1.PlugType == PlugHitTestResult.Input)
            {
                return DetermineConnectability(
                    plug2.BlockElement, plug2.PlugIndex,
                    plug1.BlockElement, plug1.PlugIndex) != Connectability.Invalid;
            }

            if (plug1.PlugType == PlugHitTestResult.Output)
            {
                return DetermineConnectability(
                    plug1.BlockElement, plug1.PlugIndex,
                    plug2.BlockElement, plug2.PlugIndex) != Connectability.Invalid;
            }

            return false;
        }

        public static Connectability DetermineConnectability(
               EffectBlockElementBase outputBlock, int outputIndex,
               EffectBlockElementBase inputBlock, int inputIndex)
        {
            var outputDef = outputBlock.BlockDefinition;
            var inputDef = inputBlock.BlockDefinition;

            var polymorphicOutput = outputBlock as PolymorphicBlockElement;
            var polymorphicInput = inputBlock as PolymorphicBlockElement;

            var outPlugName = outputDef.OutputPlugs[outputIndex].Name;
            var inPlugName = inputDef.InputPlugs[inputIndex].Name;

            ShaderTyping.ShaderTypeDefinition[] possibleOutputTypes;
            ShaderTyping.ShaderTypeDefinition[] possibleInputTypes;

            if (polymorphicOutput != null)
                possibleOutputTypes = polymorphicOutput.GetPossibleTypes(outputBlock.WorkflowItem, false, outputIndex);
            else
                possibleOutputTypes = new ShaderTyping.ShaderTypeDefinition[] { outputDef.OutputPlugs[outputIndex].Type };

            if (polymorphicInput != null)
                possibleInputTypes = polymorphicInput.GetPossibleTypes(inputBlock.WorkflowItem, true, inputIndex);
            else
                possibleInputTypes = new[] { inputDef.InputPlugs[inputIndex].Type };

            if (Generation.AutoGen.OperatorDefinitionSetUtility.TypesMatch(possibleOutputTypes, possibleInputTypes))
            {
                if (outPlugName == inPlugName)
                    return Connectability.Best;
                return Connectability.Good;
            }

            return Connectability.Invalid;
        }

        /// <summary>
        /// エッジの接続操作が終わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void ConnectionManagerEndWiringTransaction(object sender, EventArgs e)
        {
            this.RunShaderGenerationProcessSync(false, true);
        }
    }
}
