﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation
{
    /// <summary>
    /// ユーザ定義ブロックです。
    /// </summary>
    public class UserDefinitionBlockElement : RegularEffectBlockElement
    {
        /// <summary>
        /// uniform 変数値です。
        /// </summary>
        private UniformData uniformValue;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockDefinition">ブロック定義</param>
        public UserDefinitionBlockElement(UserDefinition blockDefinition)
            : base(blockDefinition)
        {
            this.InitializeData();
        }

        /// <summary>
        /// uniform 変数値を取得または設定します。
        /// </summary>
        public UniformData UniformValue
        {
            get
            {
                return this.uniformValue;
            }

            set
            {
                if (value != null)
                {
                    this.uniformValue = (UniformData)value.Clone();
                }
                else
                {
                    this.uniformValue = null;
                }
            }
        }

        /// <summary>
        /// データの初期化を行います。
        /// </summary>
        public void InitializeData()
        {
            if (this.BlockDefinition.Uniform != null)
            {
                var uniformDefinition = BlockDefinition.Uniform;
                this.UniformValue = Globals.UniformManager.GetUniformParameter(uniformDefinition.Name);

                this.UpdateParameterText();
            }
        }

        /// <summary>
        /// ユーザ定義ブロックのコピーを行います。
        /// </summary>
        /// <param name="blockElement">コピー元</param>
        public void CopyUserDefinitionBlockElement(UserDefinitionBlockElement blockElement)
        {
            this.UpdateBlockDefinition(new UserDefinition(this.BlockDefinition));

            if (!((UserDefinition)BlockDefinition).CanEditValue)
            {
                return;
            }

            var uniformType = (ShaderTypeDefinition)blockElement.BlockDefinition.CurrentFunctionDefinition.ReturnType;
            var uniformTypeStr = GlslTypingUtility.GetShaderTypeFromGlslTypeName(uniformType.TypeString);

            BlockDefinition.SubName = uniformTypeStr.ToString();

            this.UniformValue = blockElement.UniformValue;
        }

        /// <summary>
        /// パラメータを更新します。
        /// </summary>
        public void UpdateData()
        {
            if (!((UserDefinition)BlockDefinition).CanEditValue)
            {
                return;
            }

            this.UpdateParameterText();
        }

        /// <summary>
        /// パラメータのテキストを更新します。
        /// </summary>
        private void UpdateParameterText()
        {
            if (!((UserDefinition)BlockDefinition).CanEditValue)
            {
                return;
            }

            if (this.UniformValue == null || this.UniformValue is UniformDataFile)
            {
                return;
            }

            var shaderType = (ShaderTypeDefinition)BlockDefinition.CurrentFunctionDefinition.ReturnType;
            var glslShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(shaderType.TypeString);

            var userDefinition = BlockDefinition as UserDefinition;

            var displayName = BlockDefinition.Uniform.Displayname;
            var strArray = UniformDataHelper.UniformDataToStringArray(this.UniformValue);
            userDefinition.ParameterText = GlslTypingUtility.GetShaderParamFromShaderType(glslShaderType, strArray, new string[] { displayName });
        }
    }
}
