﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using EffectCombiner.Core;

namespace EffectCombiner.Primitives.Generation
{
    /// <summary>
    ///  uniform 変数の値を管理するマネージャクラスです。
    /// </summary>
    public class UniformManager
    {
        /// <summary>
        /// uniform 変数リストです。
        /// </summary>
        private Dictionary<string, UniformData> uniforms;

        /// <summary>
        /// 読み取り専用の uniform 変数ディクショナリです。
        /// </summary>
        private ReadOnlyDictionary<string, UniformData> readonlyUniforms;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformManager()
        {
            this.uniforms = new Dictionary<string, UniformData>();
        }

        /// <summary>
        /// uniform 変数のデータ変更時に呼ばれるイベントです。
        /// </summary>
        public event EventHandler<UniformDataChangedEventArgs> UniformDataChanged;

        /// <summary>
        /// uniform 変数の初期化時に呼ばれるイベントです。
        /// </summary>
        public event EventHandler UniformDataInitialized;

        /// <summary>
        /// uniform 変数です。
        /// </summary>
        public ReadOnlyDictionary<string, UniformData> Uniforms
        {
            get
            {
                if (this.readonlyUniforms == null)
                {
                    this.readonlyUniforms = new ReadOnlyDictionary<string, UniformData>(this.uniforms);
                }

                return this.readonlyUniforms;
            }
        }

        /// <summary>
        /// uniform 変数を初期化します。
        /// </summary>
        public void InitializeValues()
        {
            // uniform 変数リストをデフォルト値で初期化
            foreach (string name in this.uniforms.Keys.ToArray())
            {
                this.uniforms[name] = UniformDataHelper.CreateUniformData(this.uniforms[name].Type);
            }

            this.OnUniformInitializedValue();
        }

        /// <summary>
        /// 指定値で uniform 変数を初期化します。
        /// 指定がない場合デフォルト値を設定します。
        /// </summary>
        /// <param name="uniformDataList">uniform 変数の初期値</param>
        public void InitializeValues(IEnumerable<KeyValuePair<string, UniformData>> uniformDataList)
        {
            // uniform 変数リストをデフォルト値で初期化
            foreach (string name in this.uniforms.Keys.ToArray())
            {
                this.uniforms[name] = UniformDataHelper.CreateUniformData(this.uniforms[name].Type);
            }

            // 指定された値を uniform 変数に設定
            foreach (var uniform in uniformDataList)
            {
                this.uniforms[uniform.Key] = uniform.Value;
            }

            this.OnUniformInitializedValue();
        }

        /// <summary>
        /// uniform 変数を追加します。
        /// </summary>
        /// <param name="uniformName">uniform 変数名</param>
        /// <param name="uniformType">uniform 変数の型</param>
        public void AddUniformParameter(string uniformName, string uniformType)
        {
            this.uniforms[uniformName] = UniformDataHelper.CreateUniformData(uniformType);
        }

        /// <summary>
        /// uniform 値を設定します。
        /// </summary>
        /// <param name="name">変数名</param>
        /// <param name="value">値</param>
        public void SetUniformParameter(string name, UniformData value)
        {
            UniformData oldValue = null;

            this.uniforms.TryGetValue(name, out oldValue);

            this.uniforms[name] = value;

            this.OnUniformDataChanged(new UniformDataChangedEventArgs(name, oldValue, value));
        }

        /// <summary>
        /// uniform 値を取得します。
        /// </summary>
        /// <param name="name">変数名</param>
        /// <returns>uniform 値を返します。変数名に対応する値が存在しない場合は null を返します。</returns>
        public UniformData GetUniformParameter(string name)
        {
            UniformData result;
            this.uniforms.TryGetValue(name, out result);

            return result;
        }

        /// <summary>
        /// uniform 変数が存在するかどうかを取得します。
        /// </summary>
        /// <param name="name">uniform 変数名</param>
        /// <returns>存在するかどうか。</returns>
        public bool ExistUniformParam(string name)
        {
            return this.uniforms.ContainsKey(name);
        }

        /// <summary>
        /// uniform 変数を全て削除します。
        /// </summary>
        public void Clear()
        {
            this.InitializeValues();
            this.uniforms.Clear();
        }

        /// <summary>
        /// uniform 変数が変更されたときに呼ばれます。
        /// </summary>
        /// <param name="e">uniform 変数の変更イベント</param>
        private void OnUniformDataChanged(UniformDataChangedEventArgs e)
        {
            this.UniformDataChanged?.Invoke(this, e);
        }

        /// <summary>
        /// uniform 変数の値が初期化されたときに呼ばれます。
        /// </summary>
        private void OnUniformInitializedValue()
        {
            this.UniformDataInitialized?.Invoke(this, EventArgs.Empty);
        }
    }
}
