﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectCombiner.Primitives.Generation
{
    #region UniformData

    /// <summary>
    /// uniform 変数データです。
    /// </summary>
    public abstract class UniformData : ICloneable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="type">uniform の型</param>
        public UniformData(string type)
        {
            this.Type = type;
        }

        #region ICloneable members

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public abstract bool Set(object src);

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public abstract object Clone();

        #endregion

        /// <summary>
        /// uniform 変数のデータ型を取得します。
        /// </summary>
        public string Type { get; private set; }

        /// <summary>
        /// 値を取得します。
        /// </summary>
        public object Value { get; protected set; }
    }

    #endregion

    #region UniformDataFile

    /// <summary>
    /// file の uniform 変数データです。
    /// </summary>
    public class UniformDataFile : UniformData
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformDataFile()
            : base("file")
        {
            this.Value = string.Empty;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">初期値</param>
        public UniformDataFile(string value)
            : this()
        {
            this.Value = value;
        }

        #region CopyConstructor, ICloneable members

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public UniformDataFile(UniformDataFile obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            UniformDataFile srcData = src as UniformDataFile;

            if (srcData == null)
            {
                return false;
            }

            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new UniformDataFile(this);
        }

        #endregion

        /// <summary>
        /// 文字列で指定された値をパースして変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string s, out UniformDataFile result)
        {
            result = new UniformDataFile(s);

            return true;
        }

        /// <summary>
        /// 値を文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public override string ToString()
        {
            return this.Value;
        }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public new string Value
        {
            get
            {
                return (string)base.Value;
            }

            set
            {
                base.Value = value;
            }
        }
    }

    #endregion

    #region UniformDataFloat

    /// <summary>
    /// float の uniform 変数データです。
    /// </summary>
    public class UniformDataFloat : UniformData
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformDataFloat()
            : base("float")
        {
            this.Value = 0.0f;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">初期値</param>
        public UniformDataFloat(float value)
            : this()
        {
            this.Value = value;
        }

        #region CopyConstructor, ICloneable members

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public UniformDataFloat(UniformDataFloat obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            UniformDataFloat srcData = src as UniformDataFloat;

            if (srcData == null)
            {
                return false;
            }

            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new UniformDataFloat(this);
        }

        #endregion

        /// <summary>
        /// 文字列で指定された値をパースして変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string s, out UniformDataFloat result)
        {
            result = null;

            float val;
            bool resParse = float.TryParse(s, out val);

            if (resParse == false)
            {
                return false;
            }

            result = new UniformDataFloat(val);

            return true;
        }

        /// <summary>
        /// 値を文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public override string ToString()
        {
            return this.Value.ToString();
        }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public new float Value
        {
            get
            {
                return (float)base.Value;
            }

            set
            {
                base.Value = value;
            }
        }
    }

    #endregion

    #region UniformDataVec2

    /// <summary>
    /// vec2 の uniform 変数データです。
    /// </summary>
    public class UniformDataVec2 : UniformData
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformDataVec2()
            : base("vec2")
        {
            base.Value = new float[] { 0.0f, 0.0f };
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">初期値</param>
        public UniformDataVec2(float[] value)
            : this()
        {
            if (value == null)
                throw new ArgumentNullException(nameof(value));
            if (value.Length != 2)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, nameof(value)), nameof(value));

            this.Value = value;
        }

        #region CopyConstructor, ICloneable members

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public UniformDataVec2(UniformDataVec2 obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            UniformDataVec2 srcData = src as UniformDataVec2;

            if (srcData == null)
            {
                return false;
            }

            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new UniformDataVec2(this);
        }

        #endregion

        /// <summary>
        /// 文字列で指定された値をパースして変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string s, out UniformDataVec2 result)
        {
            result = null;

            string[] elements = s.Split(';');
            float[] val = new float[2];

            if (elements.Length != val.Length)
            {
                return false;
            }

            for (int i = 0; i < val.Length; i++)
            {
                bool resParse = float.TryParse(elements[i], out val[i]);
                if (resParse == false)
                {
                    return false;
                }
            }

            result = new UniformDataVec2(val);

            return true;
        }

        /// <summary>
        /// 値を文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public override string ToString()
        {
            return string.Join(";", this.Value.Select(x => x.ToString()));
        }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public new float[] Value
        {
            get
            {
                return (float[])base.Value;
            }

            set
            {
                if (value != null && value.Length == 2)
                {
                    float[] val = (float[])base.Value;

                    val[0] = value[0];
                    val[1] = value[1];
                }
            }
        }
    }

    #endregion

    #region UniformDataVec3

    /// <summary>
    /// vec3 の uniform 変数データです。
    /// </summary>
    public class UniformDataVec3 : UniformData
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformDataVec3()
            : base("vec3")
        {
            base.Value = new float[] { 0.0f, 0.0f, 0.0f };
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">初期値</param>
        public UniformDataVec3(float[] value)
            : this()
        {
            if (value == null)
                throw new ArgumentNullException(nameof(value));
            if (value.Length != 3)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, nameof(value)), nameof(value));

            this.Value = value;
        }

        #region CopyConstructor, ICloneable members

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public UniformDataVec3(UniformDataVec3 obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            UniformDataVec3 srcData = src as UniformDataVec3;

            if (srcData == null)
            {
                return false;
            }

            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new UniformDataVec3(this);
        }

        #endregion

        /// <summary>
        /// 文字列で指定された値をパースして変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string s, out UniformDataVec3 result)
        {
            result = null;

            string[] elements = s.Split(';');
            float[] val = new float[3];

            if (elements.Length != val.Length)
            {
                return false;
            }

            for (int i = 0; i < val.Length; i++)
            {
                bool resParse = float.TryParse(elements[i], out val[i]);
                if (resParse == false)
                {
                    return false;
                }
            }

            result = new UniformDataVec3(val);

            return true;
        }

        /// <summary>
        /// 値を文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public override string ToString()
        {
            return string.Join(";", this.Value.Select(x => x.ToString()));
        }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public new float[] Value
        {
            get
            {
                return (float[])base.Value;
            }

            set
            {
                if (value != null && value.Length == 3)
                {
                    float[] val = (float[])base.Value;

                    val[0] = value[0];
                    val[1] = value[1];
                    val[2] = value[2];
                }
            }
        }
    }

    #endregion

    #region UniformDataVec4

    /// <summary>
    /// vec4 の uniform 変数データです。
    /// </summary>
    public class UniformDataVec4 : UniformData
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformDataVec4()
            : base("vec4")
        {
            base.Value = new float[] { 1.0f, 1.0f, 1.0f, 1.0f };
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">初期値</param>
        public UniformDataVec4(float[] value)
            : this()
        {
            if (value == null)
                throw new ArgumentNullException(nameof(value));
            if (value.Length != 4)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, nameof(value)), nameof(value));

            this.Value = value;
        }

        #region CopyConstructor, ICloneable members

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public UniformDataVec4(UniformDataVec4 obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            UniformDataVec4 srcData = src as UniformDataVec4;

            if (srcData == null)
            {
                return false;
            }

            this.Value = srcData.Value;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new UniformDataVec4(this);
        }

        #endregion

        /// <summary>
        /// 文字列で指定された値をパースして変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string s, out UniformDataVec4 result)
        {
            result = null;

            string[] elements = s.Split(';');
            float[] val = new float[4];

            if (elements.Length != val.Length)
            {
                return false;
            }

            for (int i = 0; i < val.Length; i++)
            {
                bool resParse = float.TryParse(elements[i], out val[i]);
                if (resParse == false)
                {
                    return false;
                }
            }

            result = new UniformDataVec4(val);

            return true;
        }

        /// <summary>
        /// 値を文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public override string ToString()
        {
            return string.Join(";", this.Value.Select(x => x.ToString()));
        }

        /// <summary>
        /// 値を取得または設定します。
        /// </summary>
        public new float[] Value
        {
            get
            {
                return (float[])base.Value;
            }

            set
            {
                if (value != null && value.Length == 4)
                {
                    float[] val = (float[])base.Value;

                    val[0] = value[0];
                    val[1] = value[1];
                    val[2] = value[2];
                    val[3] = value[3];
                }
            }
        }
    }

    #endregion

    #region UniformDataHelper

    /// <summary>
    /// UniformData についての処理を行うヘルパークラスです。
    /// </summary>
    public static class UniformDataHelper
    {
        /// <summary>
        /// UniformData を作成します。
        /// </summary>
        /// <param name="type">型</param>
        /// <returns>作成した UniformData を返します。</returns>
        public static UniformData CreateUniformData(string type)
        {
            switch (type)
            {
                case "file":
                    return new UniformDataFile();
                case "float":
                    return new UniformDataFloat();
                case "vec2":
                    return new UniformDataVec2();
                case "vec3":
                    return new UniformDataVec3();
                case "vec4":
                    return new UniformDataVec4();
            }

            return null;
        }

        /// <summary>
        /// 文字列で指定された UniformData の値をパースして変換します。
        /// </summary>
        /// <param name="type">型</param>
        /// <param name="s">文字列</param>
        /// <param name="result">変換したデータ</param>
        /// <returns>パースが成功したときは true、失敗したときは false を返します。</returns>
        public static bool TryParse(string type, string s, out UniformData result)
        {
            switch (type)
            {
                case "file":
                {
                    UniformDataFile data;
                    bool resParse = UniformDataFile.TryParse(s, out data);
                    result = data;

                    return resParse;
                }

                case "float":
                {
                    UniformDataFloat data;
                    bool resParse = UniformDataFloat.TryParse(s, out data);
                    result = data;

                    return resParse;
                }

                case "vec2":
                {
                    UniformDataVec2 data;
                    bool resParse = UniformDataVec2.TryParse(s, out data);
                    result = data;

                    return resParse;
                }

                case "vec3":
                {
                    UniformDataVec3 data;
                    bool resParse = UniformDataVec3.TryParse(s, out data);
                    result = data;

                    return resParse;
                }

                case "vec4":
                {
                    UniformDataVec4 data;
                    bool resParse = UniformDataVec4.TryParse(s, out data);
                    result = data;

                    return resParse;
                }
            }

            result = null;

            return false;
        }

        /// <summary>
        /// 文字列配列データを UniformData に変換します。
        /// </summary>
        /// <param name="type">型</param>
        /// <param name="value">文字列配列データ</param>
        /// <returns>UniformData を返します。</returns>
        public static UniformData StringArrayToUniformData(string type, string[,] value)
        {
            int dimX = 0;
            int dimY = 0;

            if (value != null)
            {
                dimX = value.GetLength(0);
                dimY = value.GetLength(1);
            }

            if (type == "file")
            {
                UniformDataFile data = new UniformDataFile();

                if (dimX >= 1 && dimY >= 1)
                {
                    data.Value = value[0, 0];
                }

                return data;
            }
            else if (type == "float")
            {
                UniformDataFloat data = new UniformDataFloat();

                if (dimX >= 1 && dimY >= 1)
                {
                    float val;
                    float.TryParse(value[0, 0], out val);

                    data.Value = val;
                }

                return data;
            }
            else if (type == "vec2")
            {
                UniformDataVec2 data = new UniformDataVec2();

                if (dimX >= 2 && dimY >= 1)
                {
                    for (int i = 0; i < 2; ++i)
                    {
                        float val;
                        float.TryParse(value[i, 0], out val);

                        data.Value[i] = val;
                    }
                }

                return data;
            }
            else if (type == "vec3")
            {
                UniformDataVec3 data = new UniformDataVec3();

                if (dimX >= 3 && dimY >= 1)
                {
                    for (int i = 0; i < 3; ++i)
                    {
                        float val;
                        float.TryParse(value[i, 0], out val);

                        data.Value[i] = val;
                    }
                }

                return data;
            }
            else if (type == "vec4")
            {
                UniformDataVec4 data = new UniformDataVec4();

                if (dimX >= 4 && dimY >= 1)
                {
                    for (int i = 0; i < 4; ++i)
                    {
                        float val;
                        float.TryParse(value[i, 0], out val);

                        data.Value[i] = val;
                    }
                }

                return data;
            }

            return null;
        }

        /// <summary>
        /// UniformData を文字列配列データに変換します。
        /// </summary>
        /// <param name="value">UniformData</param>
        /// <returns>文字列配列データを返します。</returns>
        public static string[,] UniformDataToStringArray(UniformData value)
        {
            if (value is UniformDataFile)
            {
                UniformDataFile data = (UniformDataFile)value;

                string[,] strArray = new string[,]
                {
                    { data.Value }
                };

                return strArray;
            }
            else if (value is UniformDataFloat)
            {
                UniformDataFloat data = (UniformDataFloat)value;

                string[,] strArray = new string[,]
                {
                    { data.Value.ToString() }
                };

                return strArray;
            }
            else if (value is UniformDataVec2)
            {
                UniformDataVec2 data = (UniformDataVec2)value;

                string[,] strArray = new string[,]
                {
                    { data.Value[0].ToString() },
                    { data.Value[1].ToString() }
                };

                return strArray;
            }
            else if (value is UniformDataVec3)
            {
                UniformDataVec3 data = (UniformDataVec3)value;

                string[,] strArray = new string[,]
                {
                    { data.Value[0].ToString() },
                    { data.Value[1].ToString() },
                    { data.Value[2].ToString() }
                };

                return strArray;
            }
            else if (value is UniformDataVec4)
            {
                UniformDataVec4 data = (UniformDataVec4)value;

                string[,] strArray = new string[,]
                {
                    { data.Value[0].ToString() },
                    { data.Value[1].ToString() },
                    { data.Value[2].ToString() },
                    { data.Value[3].ToString() }
                };

                return strArray;
            }

            return null;
        }
    }

    #endregion
}
