﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using EffectDefinitions;
using ShaderGenerator.GLSL;
using EffectCombiner.Core;

namespace EffectCombiner.Primitives.Generation.AutoGen
{
    public static class OperatorDefinitionSetUtility
    {
        public static IAutoDefinitionSet[] InstanceAllDefinitionSets(IExceptionReporter reporter)
        {
            var assembly = Assembly.GetExecutingAssembly();
            return InstanceAllDefinitionSets(assembly, reporter).ToArray();
        }

        private static IEnumerable<IAutoDefinitionSet> InstanceAllDefinitionSets(Assembly assembly, IExceptionReporter reporter)
        {
            var list = new List<IAutoDefinitionSet>();

            var types = assembly.GetTypes()
                .Where(t => t.IsInterface == false && t.IsAbstract == false)
                .Where(t => typeof(IAutoDefinitionSet).IsAssignableFrom(t));

            foreach (var t in types)
            {
                try
                {
                    var instance = (IAutoDefinitionSet)Activator.CreateInstance(t);
                    list.Add(instance);
                }
                catch (Exception ex)
                {
                    if (reporter != null)
                        reporter.Report(ex);
                    else
                        throw;
                }
            }

            var assemblies = from an in assembly.GetReferencedAssemblies()
                             select Assembly.Load(an)
                                 into asm
                                 where asm.GlobalAssemblyCache == false
                                 select asm;

            foreach (var sub in assemblies)
                list.AddRange(InstanceAllDefinitionSets(sub, reporter));

            return list;
        }

        public static bool TypesMatch(IEnumerable<ShaderTyping.ShaderTypeDefinition> a,
            IEnumerable<ShaderTyping.ShaderTypeDefinition> b)
        {
            return a.Any(ta => b.Any(ta.Equals));
        }

        public static bool TypesMatch(IEnumerable<ShaderTyping.ShaderTypeDefinition> a,
            ShaderTyping.ShaderTypeDefinition b)
        {
            return a.Any(ta => ta.Equals(b));
        }

        public static FunctionDefinition[] CreateFunctions(IAutoDefinitionSet definitionSet, string[][] types, string[] paramNames)
        {
            return types
                .Select((ts, i) => CreateFunction(definitionSet, i, ts, paramNames))
                .ToArray();
        }

        public static FunctionDefinition CreateFunction(IAutoDefinitionSet definitionSet, int index, string[] types, string[] paramNames)
        {
            var parameters = types
                .Skip(1)
                .Select((t, i) => new ParameterDefinition(null, paramNames[i],
                    new ShaderTypeDefinition(t),
                    ShaderTyping.ParameterDirection.Input,
                    null, null))
                .ToArray();

            var funcId = string.Format("{0}_{1}_{2}", definitionSet.FunctionName, definitionSet.Guid, index);

            return new FunctionDefinition(null,
                funcId,
                definitionSet.FunctionName,
                new ShaderTypeDefinition(types[0]), null, null, null, parameters);
        }

        public static string CreateShaderCode(ShaderTyping.FunctionDefinition func, string op)
        {
            if (func == null)
                throw new ArgumentNullException("func");
            if (func.Parameters.Length != 2)
            {
                var a = string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "func");
                var b = string.Format(Messages.EXCEPTION_MUST_CONTAIN_N_PARAMS, 2);
                throw new ArgumentException(string.Format("{0} ({1})", a, b));
            }

            var sb = new StringBuilder();

            sb.AppendFormat("{0} {1}(", ((ShaderTypeDefinition)func.ReturnType).TypeString, func.Name);
            sb.AppendFormat("{0}, {1}", func.Parameters[0].GetPrototype(), func.Parameters[1].GetPrototype());
            sb.AppendLine(")");

            sb.AppendLine("{");
            sb.AppendLine(string.Format("  return {0} {1} {2};",
                func.Parameters[0].Name,
                op,
                func.Parameters[1].Name));
            sb.AppendLine("}");

            return sb.ToString();
        }
    }
}
