﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Win32;
using System.Diagnostics;
using System.IO;
using System.Threading.Tasks;
using System.Xml;

namespace EffectCombiner.PreBuild
{
    public class Subversion
    {
        public static readonly bool IsAvailable;

        private static readonly string processFilename;

        static Subversion()
        {
            processFilename = GetProcessFilename();
            IsAvailable = processFilename != null;
        }

        public static string GetCurrentRevision(string path)
        {
            var output = RunCommand("info --xml -r BASE", path);
            if (output == null)
                return null;

            try
            {
                var doc = new XmlDocument();
                doc.LoadXml(output);

                var node = doc.SelectSingleNode("info/entry/commit");
                if (node == null)
                {
                    Console.WriteLine(Messages.CANNOT_FIND_NODE, "info/entry/commit");
                    return null;
                }

                if (node.Attributes == null)
                {
                    Console.WriteLine(Messages.NO_ATTRIBUTE);
                    return null;
                }

                var attr = node.Attributes["revision"];
                if (attr == null)
                {
                    Console.WriteLine(Messages.CANNOT_FIND_ATTRIBUTE, "revision", "info/entry/commit");
                    return null;
                }

                return attr.Value.Trim();
            }
            catch (Exception ex)
            {
                Common.DumpException(ex);
                return null;
            }
        }

        private static string RunCommand(string command, string path)
        {
            if (string.IsNullOrWhiteSpace(command))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "command"), "command");
            if (string.IsNullOrWhiteSpace(path))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "path"), "path");

            if (processFilename == null)
                return null;

            var args = string.Format("{0} \"{1}\"", command, path);
            Console.WriteLine(Messages.RUN_COMMAND, processFilename, args);

            var process = Process.Start(new ProcessStartInfo
                {
                    Arguments = args,
                    CreateNoWindow = true,
                    FileName = processFilename,
                    RedirectStandardOutput = true,
                    UseShellExecute = false,
                    WindowStyle = ProcessWindowStyle.Hidden,
                });

            var result = process.StandardOutput.ReadToEnd();

            if (string.IsNullOrWhiteSpace(result))
            {
                Console.WriteLine(Messages.RETURNED_EMPTY_CONTENT);
                result = null;
            }
            else
            {
                Console.WriteLine(Messages.RETURNED_FOLLOWING_CONTENT);
                Console.WriteLine(result);
            }

            return result;
        }

        private static string GetProcessFilename()
        {
            try
            {
                var pathVar = Environment.GetEnvironmentVariable("PATH");
                if (pathVar == null)
                {
                    Console.WriteLine(Messages.CANNOT_GET_PATH_ENV_VAR);
                    return null;
                }

                foreach (var path in pathVar.Split(';').Select(x => x.Trim()))
                {
                    var filename = Path.GetFullPath(Path.Combine(path, "svn.exe"));
                    if (File.Exists(filename))
                        return filename;
                }

                Console.WriteLine(Messages.CANNOT_FIND_SVN_IN_PATH_ENV_VAR);
            }
            catch (Exception ex)
            {
                Common.DumpException(ex);
            }

            return null;
        }
    }
}
