﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using EffectCombiner.Core;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.AutoGen;
using EffectDefinitions;
using ShaderGenerator.Core;

namespace EffectCombiner.Generator
{
    public class ShaderGen
    {
        private readonly ShaderGenSettings settings;

        private IContentRepository contentRepository;
        private IEnumerable<BlockDefinition> blockDefinitions;

        public ShaderGen(ShaderGenSettings settings)
        {
            if (settings == null)
                throw new ArgumentNullException("settings");

            this.settings = settings;
        }

        public void Initialize()
        {
            InitializeEffectDefinitions();
        }

        public Task InitializeAsync()
        {
            return Task.Factory.StartNew(Initialize);
        }

        /// <summary>
        /// シェーダコードを生成します。
        /// </summary>
        /// <param name="projectFile">プロジェクトファイル</param>
        /// <param name="shaderMainName">シェーダコードのmain関数名</param>
        /// <returns>生成したシェーダコードを返します。</returns>
        public string[] ProduceShaderCodes(string projectFile, string shaderMainName)
        {
            if (string.IsNullOrWhiteSpace(projectFile))
                throw new ArgumentException("Invalid 'projectFile' argument.", "projectFile");

            if (blockDefinitions == null)
                throw new InvalidOperationException("Not fully initialized. (Block Definitions not ready)");

            var definitionsDic = blockDefinitions.ToDictionary(bd => bd.Guid);
            if (definitionsDic.Count == 0)
                throw new InvalidOperationException("No block definition available.");

            var project = ProjectLoader.LoadProject(projectFile);

            var workflow = WorkflowLoader.GetWorkflow(project, definitionsDic);

            return CodeGenerator.GetShaderCodes(workflow, contentRepository, settings, shaderMainName);
        }

        private void InitializeEffectDefinitions()
        {
            var container = new EffectDefinitionsContainer();
            contentRepository = new CustomContentRepository(CacheMode.Compressed);

            GenerateOperators(container);
            GenerateConstantBlock(container);
            GenerateCommentBlock(container);

            container.SetDefinitionsPaths(CancellationToken.None,
                settings.EffectDefinitionsPaths.Paths.ToArray());

            container.UpdateBindings();

            //foreach (var ex in container.Errors)
            //{
            //    ex.Type == EffectDefinitions.ErrorType.MissingInformation
            //}

            blockDefinitions = container.Blocks;
        }

        private void GenerateOperators(EffectDefinitionsContainer container)
        {
            var operatorDefinitionSets = OperatorDefinitionSetUtility.InstanceAllDefinitionSets(NullExceptionReporter.Default);

            foreach (var defSet in operatorDefinitionSets)
            {
                var funcDefs = defSet.BlockDefinition.FunctionDefinitions;

                container.AddFunctionDefinitions(funcDefs);
                container.AddBlockDefinitions(defSet.BlockDefinition);

                if (defSet.ShaderCodes == null || defSet.ShaderCodes.Length == 0)
                    continue;

                for (var i = 0; i < funcDefs.Length; i++)
                {
                    var funcDef = funcDefs[i];
                    ((CustomContentRepository)contentRepository).SetCustomContent(
                        funcDef.GetMangling(),
                        defSet.ShaderCodes[i]);
                }
            }
        }

        private void GenerateConstantBlock(EffectDefinitionsContainer container)
        {
            var def = new ConstantBlockDefinition(new ShaderTypeDefinition("float"));
            container.AddFunctionDefinitions(def.FunctionDefinition);
            container.AddBlockDefinitions(def);
        }

        /// <summary>
        /// コメントノードを生成
        /// </summary>
        private void GenerateCommentBlock(EffectDefinitionsContainer container)
        {
            var def = new CommentBlockDefinition();
            container.AddFunctionDefinitions(def.FunctionDefinition);
            container.AddBlockDefinitions(def);
        }
    }
}
