﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using EffectCombiner.Data;
using EffectCombiner.Core;

namespace EffectCombiner.Generator
{
    public static class ProjectLoader
    {
        private const ushort CurrentVersion = 2;

        public static object LoadProject(string xmlFile)
        {
            ushort version;
            if (WorkflowDataHelper.FindVersion(xmlFile, out version) == false)
                throw new FormatException("Impossible to determine project file version");

            var processor = WorkflowDataHelper.GetProcessor(version);
            if (processor == null)
                throw new InvalidOperationException(string.Format("There is no project file processor for version {0}.", version));

            object data;

            var reporter = new DefaultWorkflowDataEventReporter();

            processor.SetReporter(reporter);
            using (var fs = File.OpenRead(xmlFile))
                data = processor.Load(fs);

            if (data == null)
                throw new AggregateException(reporter.EventReports.Select(CreateException));

            if (version < CurrentVersion)
            {
                var converter = WorkflowDataHelper.GetUpgradeConverter(version, CurrentVersion);
                data = converter.Convert(data);
            }
            else if (version > CurrentVersion)
            {
                var converter = WorkflowDataHelper.GetDowngradeConverter(version, CurrentVersion);
                data = converter.Convert(data);
            }

            if (data == null)
            {
                var message = string.Format("Conversion from version {0} to version {1} failed.", version, CurrentVersion);
                throw new FormatException(message);
            }

            return data;
        }

        private static Exception CreateException(WorkflowDataEventReport report)
        {
            if (report.IsError)
                return new ErrorException(report.Message(), report.Exception);
            return new WarningException(report.Message(), report.Exception);
        }
    }

    public class DefaultWorkflowDataEventReporter : IWorkflowDataEventReporter
    {
        public List<WorkflowDataEventReport> EventReports = new List<WorkflowDataEventReport>();

        public void Report(WorkflowDataEventReport eventReport)
        {
            EventReports.Add(eventReport);
        }
    }
}
