﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Editor.Controls;
using EffectCombiner.Editor.Properties;
using EffectCombiner.Primitives;
using Renderer2D.Core;
using System.Threading.Tasks;

namespace EffectCombiner.Editor
{
    public class RendererManager
    {
        public Renderer CurrentRenderer { get; private set; }
        public RenderSurface RenderSurface { get; private set; }

        private Renderer[] renderers;
        private readonly Panel renderSurfaceContainer;
        private readonly ToolStripComboBox rendererComboBox;

        public RendererManager(ToolStripComboBox rendererComboBox, Panel renderSurfaceContainer)
        {
            this.renderSurfaceContainer = renderSurfaceContainer;
            this.rendererComboBox = rendererComboBox;
            rendererComboBox.SelectedIndexChanged += RendererIndexChanged;
        }

        public event EventHandler CurrentRendererChanged;
        public event EventHandler RenderSurfaceCreated;

        public void LoadRenderersAsync()
        {
            Task.Factory.StartNew(() =>
                {
                    var renderersPath = CoreUtility.UsePath("Renderers");

                    var composer = new Composer<Renderer>();

                    composer.AddPathLookup(renderersPath, false, "*.dll");
                    renderers = composer.Compose(new ReportingExceptionReporter(ReportCategory.RendererComposition));

                }).ContinueWith(t => CreateRenderersMenu(), TaskScheduler.FromCurrentSynchronizationContext());
        }

        private void CreateRenderersMenu()
        {
            rendererComboBox.Items.Clear();

            foreach (var renderer in renderers)
            {
                var displayNames = CoreUtility.GetAttribute<DisplayNameAttribute>(renderer);

                string name;
                if (displayNames != null)
                    name = displayNames[0].DisplayName;
                else
                    name = renderer.GetType().Name;

                rendererComboBox.Items.Add(new ComboBoxItem<Renderer>(name, renderer));
            }

            if (rendererComboBox.Items.Count == 0)
                return;

            var item = rendererComboBox.Items
                .Cast<ComboBoxItem<Renderer>>()
                .FirstOrDefault(m => m.ToString() == "GDI Renderer");

            if (item == null)
            {
                item = rendererComboBox.Items
                    .Cast<ComboBoxItem<Renderer>>()
                    .FirstOrDefault(m => m.ToString() == "Direct2D Renderer");
            }

            if (item != null)
                rendererComboBox.SelectedItem = item;
            else
                rendererComboBox.SelectedIndex = 0;
        }

        private void RendererIndexChanged(object sender, EventArgs e)
        {
            using (new DrawingSuspender(RenderSurface))
            {
                var item = (ComboBoxItem<Renderer>)rendererComboBox.SelectedItem;
                CurrentRenderer = item.Item;
                CreateRenderSurface(CurrentRenderer);
            }

            var handler = CurrentRendererChanged;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private void CreateRenderSurface(Renderer renderer)
        {
            if (RenderSurface != null)
                RenderSurface.Dispose();

            renderSurfaceContainer.Controls.Clear();
            RenderSurface = new RenderSurface
            {
                AllowDrop = true,
                BackColor = System.Drawing.Color.Gray,
                Dock = DockStyle.Fill,
            };
            renderSurfaceContainer.Controls.Add(RenderSurface);

            renderer.Initialize();
            Globals.VisualResources.Dispose();

            var resourcesFilename = AppDomain.CurrentDomain.BaseDirectory + "\\resources.xml";
            LoadVisualResourcesFromFile(resourcesFilename, renderer);

            if (watcher == null)
            {
                watcher = new System.IO.FileSystemWatcher
                {
                    Path = System.IO.Path.GetDirectoryName(resourcesFilename),
                    Filter = System.IO.Path.GetFileName(resourcesFilename),
                };
                watcher.Changed += (ss, ee) =>
                {
                    LoadVisualResourcesFromFile(ee.FullPath, RenderSurface.Renderer);
                    RenderSurface.Invalidate();
                };
                watcher.EnableRaisingEvents = true;
            }

            Globals.VisualResources.SetInputNodeBitmap(renderer, Resources.input);
            Globals.VisualResources.SetOutputNodeBitmap(renderer, Resources.output);
            Globals.VisualResources.SetDefaultPreviewBitmap(renderer, Resources.preview);
            Globals.VisualResources.SetDefaultPreviewErrorBitmap(renderer, Resources.previewError);

            RenderSurface.SetRenderer(renderer);

            var handler = RenderSurfaceCreated;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private static System.IO.FileSystemWatcher watcher;

        /// <summary>
        /// RendererやVisualResourceSetに関する設定を読み込む
        /// </summary>
        /// <param name="resourcesFilename">リソースファイル名</param>
        /// <param name="renderer">Renderer</param>
        private static void LoadVisualResourcesFromFile(string resourcesFilename, Renderer renderer)
        {
            var resourceParser = new VisualResourceParser();

            var needSetupDefault = true;
            var needGenerate = false;

            // VisualResourceSetに関するデータを読み込む.
            if (System.IO.File.Exists(resourcesFilename))
            {
                try
                {
                    resourceParser.LoadFile(resourcesFilename, renderer, Globals.VisualResources);
                    needSetupDefault = false;
                }
                catch
                {
                }
            }
            else
            {
                needGenerate = true;
            }

            // リソースファイルの読み込みに失敗した場合は、ソースコードに直打ちしてある設定を使用する.
            if (needSetupDefault)
                Globals.VisualResources.SetupDefault(renderer);

            // BlockのBrushをセットする.
            Globals.VisualResources.SetUpBlockBackgroundBrushes(renderer);

            // リソースファイルがなかったら、現在の設定でリソースファイルを作ってしまう.
            if (needGenerate)
                resourceParser.GenerateFile(resourcesFilename, Globals.VisualResources);
        }
    }
}
