﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#if !DEBUG
    #define HANDLE_ALL_EXCEPTION
#endif

using System;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Threading;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Core.IO;
using EffectCombiner.Editor.CommandLine;
using EffectCombiner.Editor.Properties;

namespace EffectCombiner.Editor
{
    static class Program
    {
        /// <summary>
        /// 起動引数に指定されたプロジェクトファイルのパスを保持します。
        /// </summary>
        public static string argProjectConfigPath = String.Empty;

        /// <summary>The flag indicating if the fatal error had been handled.</summary>
        private static bool fatalErrorHandled = false;

        /// <summary>
        /// コマンドラインモードかどうか取得または設定します。
        /// </summary>
        public static bool IsCommandLineMode { get; set; }

        /// <summary>
        /// 同期コンテキストを取得します。
        /// </summary>
        public static SynchronizationContext SyncContext { get; private set; }

        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [STAThread]
        private static int Main(string[] args)
        {
            int result = 0;

            #if HANDLE_ALL_EXCEPTION
            {
                System.Windows.Forms.Application.ThreadException += OnThreadException;

                Application.ThreadException += ApplicationThreadException;
                AppDomain.CurrentDomain.UnhandledException += CurrentDomainUnhandledException;
            }
            #endif

            try
            {
                Program.IsCommandLineMode = CommandLineApplication.ShouldEnableCommandLine(args);

                if (Program.IsCommandLineMode)
                {
                    result = CommandLineApplication.Run(args);
                }
                else
                {
                    result = RunWindowApplication(args);
                }
            }
            #if HANDLE_ALL_EXCEPTION
            catch (Exception ex)
            {
                OnFatalErrorOccured(ex, false);

                result = 1;
            }
            #else
            finally
            {
            }
            #endif

            return result;
        }

        /// <summary>
        /// CombinerEditor をウィンドウモードで実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>終了コードを返します。</returns>
        private static int RunWindowApplication(string[] args)
        {
            CommandLineParams parameters = CommandLineApplication.ParseArgs(args, true);

            // パラメータで指定されたプロジェクトコンフィグを設定
            if (parameters != null && parameters.ProjectConfigFile != null)
            {
                argProjectConfigPath = PathUtility.ToAbsolutePath(parameters.ProjectConfigFile, Environment.CurrentDirectory);
            }

            // GUI の描画方法を指定
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            // 作業フォルダを作成
            IOUtility.SafeCreateTemporaryDirectory(IOConstants.AppDataWorkPath, IOUtility.TemporaryDirectoryUsage.ForMyProcess);

            // メインフォームを作成
            MainForm mainForm = new MainForm();
            Program.SyncContext = SynchronizationContext.Current;

            // 実行！
            Application.Run(mainForm);

            // 一時ファイルをまとめて削除
            IOUtility.SafeDeleteTemporaryDirectories();

            return 0;
        }

        private static void CurrentDomainUnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var filename = GenerateCrashDumpFilename();

            if (filename == null)
                return;

            MessageBox.Show(Localization.Messages.CRASH_DUMP_GENERATION_BEGIN,
                Localization.Messages.UNRECOVERABLE_CRASH_TITLE,
                MessageBoxButtons.OK, MessageBoxIcon.Error);

            CoreDump.Dump(filename);

            var message = string.Format("{0}\r\n{1}",
                Localization.Messages.UNRECOVERABLE_CRASH_TITLE,
                string.Format(Localization.Messages.CRASH_DUMP_GENERATED_AT, filename));

            MessageBox.Show(message,
                Localization.Messages.UNRECOVERABLE_CRASH_TITLE,
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private static void ApplicationThreadException(object sender, ThreadExceptionEventArgs e)
        {
            var filename = GenerateCrashDumpFilename();

            var comment = string.Empty;
            if (filename != null)
            {
                CoreDump.Dump(filename);
                comment = string.Format(Localization.Messages.CRASH_DUMP_GENERATED_AT, filename);
            }

            Reporting.Report(new EventReport(
                () => string.Format(Localization.Messages.EXCEPTION_APPLICATION_LEVEL, e.Exception.Message),
                ReportLevel.Fatal,
                ReportCategory.Application,
                comment,
                () => Localization.Messages.APPLICATION_IS_UNSTABLE,
                e.Exception));
        }

        private static string GenerateCrashDumpFilename()
        {
            var filename = string.Format("crashdump.dmp");

            try
            {
                return System.IO.Path.Combine(AppDomain.CurrentDomain.BaseDirectory, filename);
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        /// Handle ThreadException event from application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnThreadException(object sender, ThreadExceptionEventArgs e)
        {
            OnFatalErrorOccured(e.Exception, true);
            System.Windows.Forms.Application.Exit();
        }

        /// <summary>
        /// Handle fatal error.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <param name="fromThreadException">True when called from OnThreadException.</param>
        private static void OnFatalErrorOccured(Exception error, bool fromThreadException)
        {
            // Only handle once.
            if (fatalErrorHandled == true)
            {
                return;
            }

            fatalErrorHandled = true;

            // Output error message.
            try
            {
                string logFilePath = Path.Combine(Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), "error.log");

                string exceptionMessage = GetErrorMessage(error);
                string stackTrace = GetStackTrace(error);

                // Delete the log file if exceeds the maximum size 100KB.
                FileInfo info = new FileInfo(logFilePath);
                if (info.Exists == true && info.Length >= 100 * 1000)
                {
                    File.Delete(logFilePath);
                }

                System.Text.StringBuilder sb = new System.Text.StringBuilder();

                using (var writer = new StreamWriter(logFilePath, true))
                {
                    sb.AppendLine("--------------------------------------------------------------------------------");

                    // ビルド日時
                    Assembly entry = Assembly.GetEntryAssembly();
                    Version version = entry.GetName().Version;
                    DateTime buildDate = new DateTime(2000, 1, 1, 0, 0, 0);
                    buildDate = buildDate.AddDays((double)version.Build);
                    buildDate = buildDate.AddSeconds((double)version.Revision * 2.0);
                    sb.AppendFormat(
                        "Build date：{0} {1}",
                        buildDate.ToShortDateString(),
                        buildDate.ToShortTimeString());
                    sb.AppendLine(string.Empty);

                    // エラーが起きた日時
                    sb.AppendFormat(
                        "Error occurred at {0} {1}",
                        DateTime.Now.ToShortDateString(),
                        DateTime.Now.ToShortTimeString());
                    sb.AppendLine(string.Empty);

                    sb.AppendLine("--------------------------------------------------------------------------------");
                    sb.AppendLine("[From]:");
                    sb.AppendLine("   " + (fromThreadException ? "ThreadException" : "WinMain"));
                    sb.AppendLine("[Message]:");
                    sb.AppendLine("   " + exceptionMessage);
                    sb.AppendLine("[StackTrace]:");
                    sb.AppendLine(stackTrace);

                    writer.Write(sb.ToString());
                }

                if (Program.IsCommandLineMode)
                {
                    OutputFatalErrorMessage(sb.ToString());
                }
                else
                {
                    ShowFatalErrorDialog(sb.ToString());
                }
            }
            catch
            {
                // Do nothing, we are already handling a fatal error...
            }
        }

        /// <summary>
        /// 致命的なエラーをダイアログに表示します。
        /// </summary>
        /// <param name="log">エラーログ。</param>
        private static void ShowFatalErrorDialog(string log)
        {
            var pos = new Point(20, 20);
            const int DlgWidth = 700;

            // ダイアログ作成
            Form dlg = new Form();
            dlg.SuspendLayout();

            // ラベル：致命的なエラーが発生しました。アプリケーションを終了します。
            Label label1 = new Label();
            label1.Text = Resources.ApplicationFatalErrorMessage;
            label1.Location = pos;
            label1.AutoSize = true;

            // ラベル：Error log:
            pos.Y += 50;
            Label label2 = new Label();
            label2.Text = "Error log:";
            label2.Location = pos;
            label2.AutoSize = true;

            // テキストボックス：エラーログ
            pos.Y += 20;
            TextBox txtBox = new TextBox();
            txtBox.Text = log;
            txtBox.Location = pos;
            txtBox.Size = new System.Drawing.Size(DlgWidth - 60, 200);
            txtBox.Multiline = true;
            txtBox.ReadOnly = true;
            txtBox.SelectionStart = 0;
            txtBox.ScrollBars = ScrollBars.Both;
            txtBox.Font = new Font(txtBox.Font.FontFamily, 7);

            // ボタン：ＯＫ
            pos.Y += txtBox.Size.Height + 20;
            Button btnOk = new Button() { Text = "OK" };
            btnOk.Click += (sender, e) => { dlg.Close(); };
            btnOk.Location = pos;

            // ボタン：ＣＯＰＹ
            Button btnCopy = new Button() { Text = "Copy" };
            btnCopy.Click += (sender, e) => { Clipboard.SetDataObject(log, true); };
            btnCopy.Location = new Point(pos.X + btnOk.Size.Width + 20, pos.Y);

            // コントロール配置
            dlg.Controls.Add(label1);
            dlg.Controls.Add(label2);
            dlg.Controls.Add(txtBox);
            dlg.Controls.Add(btnOk);
            dlg.Controls.Add(btnCopy);
            dlg.Icon = Icon.FromHandle(Resources.SymbolError1.GetHicon());
            dlg.Text = Resources.ErrorCaption;
            dlg.Width = DlgWidth;
            dlg.Height = pos.Y + 80;
            dlg.FormBorderStyle = FormBorderStyle.FixedSingle;
            dlg.ResumeLayout(false);
            dlg.PerformLayout();

            // ダイアログオープン
            dlg.ShowDialog();
        }

        /// <summary>
        /// Get error message from the exception.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <returns>The error message.</returns>
        private static string GetErrorMessage(Exception error)
        {
            string result = error.Message + "\n";
            if (error.InnerException != null)
            {
                result += "\n    " + GetErrorMessage(error.InnerException);
            }

            return result;
        }

        /// <summary>
        /// Get stack trace from the exception.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <returns>The call stack.</returns>
        private static string GetStackTrace(Exception error)
        {
            string result = error.StackTrace + "\n";
            if (error.InnerException != null)
            {
                return GetStackTrace(error.InnerException) + "    " + result;
            }

            return result;
        }

        #region OutputFatalErrorMessage

        /// <summary>
        /// 致命的なエラーが発生したときのメッセージをコンソールに出力します。
        /// </summary>
        /// <param name="message">メッセージ</param>
        private static void OutputFatalErrorMessage(string message)
        {
            // コンソール画面にアタッチ
            bool attached = AttachConsole(AttachParentProcess);

            ConsoleColor originalColor = ConsoleColor.White;

            // 文字色を変更
            try
            {
                originalColor = Console.ForegroundColor;

                Console.ForegroundColor = ConsoleColor.Red;
            }
            catch
            {
            }

            // メッセージを出力
            Console.WriteLine(message);

            // 文字色を復元
            try
            {
                Console.ForegroundColor = originalColor;
            }
            catch
            {
            }

            // アタッチを解除
            if (attached)
            {
                FreeConsole();
            }
        }

        /// <summary>
        /// Win32 API Constant value: ATTACH_PARENT_PROCESS
        /// </summary>
        private const int AttachParentProcess = -1;

        //// <summary>
        //// Win32 API Function: AttachConsole(DWORD)
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AttachConsole(int processId);

        //// <summary>
        //// Win32 API Function: AttachConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AllocConsole();

        //// <summary>
        //// Win32 API Function: FreeConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool FreeConsole();

        #endregion
    }
}
