﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor.OptionPanes
{
    /// <summary>
    /// プロジェクト設定ペインです。
    /// </summary>
    public partial class ProjectSettingsPane : UserControl, IOptionPane
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ProjectSettingsPane()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// オプションペインの識別名を取得します。
        /// </summary>
        public string Identifier { get { return "ProjectSettingsPane"; } }

        /// <summary>
        /// このオプションペインの親ノードになるペインの識別名を取得します。
        /// ルートペインのときはnullを返します。
        /// </summary>
        public string ChildOf { get { return null; } }

        /// <summary>
        /// オプションダイアログに表示するノード名を取得します。
        /// </summary>
        public string DisplayName { get { return Localization.Controls.OPTION_PANE_NAME_PROJECT_SETTINGS; } }

        /// <summary>
        /// オプションペインを生成したときの処理を行います。
        /// </summary>
        public void OnInitialize()
        {
            if (Globals.Options.EnvironmentSettings == null)
            {
                return;
            }

            this.txtProjectSettingFilePath.Text = Globals.Options.EnvironmentSettings.ProjectSettingFilePath;

            if (Globals.Options.ProjectSettings == null)
            {
                return;
            }

            if (Globals.Options.ProjectSettings.DefinitionPaths != null)
            {
                this.lstDefinitionPath.SetDefinitionPaths(Globals.Options.ProjectSettings.DefinitionPaths);
            }

            this.txtPreSaveCommandPath.Text = Globals.Options.ProjectSettings.PreSaveCommandPath;
            this.txtPostSaveCommandPath.Text = Globals.Options.ProjectSettings.PostSaveCommandPath;
            this.txtPreOpenCommandPath.Text = Globals.Options.ProjectSettings.PreOpenCommandPath;
        }

        /// <summary>
        /// オプションダイアログを閉じたときの処理を行います。
        /// </summary>
        public void OnTerminate()
        {
            foreach (var item in lstDefinitionPath.ListView.Items.Cast<PathInfoListViewItem>())
            {
                item.Dispose();
            }
        }

        /// <summary>
        /// オプションダイアログのOKボタンを押したときの処理を行います。
        /// </summary>
        public void OnAccept()
        {
            if (Globals.Options.EnvironmentSettings == null)
            {
                Globals.Options.EnvironmentSettings = new EnvironmentSettings();
            }

            Globals.Options.EnvironmentSettings.ProjectSettingFilePath = this.txtProjectSettingFilePath.Text;

            if (Globals.Options.ProjectSettings == null)
            {
                Globals.Options.ProjectSettings = new ProjectSettings();
            }

            Globals.Options.ProjectSettings.DefinitionPaths = this.lstDefinitionPath.GetDefinitionPaths();

            Globals.Options.ProjectSettings.PreSaveCommandPath = txtPreSaveCommandPath.Text;
            Globals.Options.ProjectSettings.PostSaveCommandPath = txtPostSaveCommandPath.Text;
            Globals.Options.ProjectSettings.PreOpenCommandPath = txtPreOpenCommandPath.Text;
        }

        /// <summary>
        /// オプションダイアログのキャンセルボタンを押したときの処理を行います。
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// プロジェクト設定ファイル設定ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowseProjectSettingFilePathClick(object sender, EventArgs e)
        {
            // ファイル名入力ダイアログを表示
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.xml|{1}|*.*", Localization.Messages.DLG_FILTER_XML_FILES, Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PROJECT_SETTING_FILE_TITLE,
            };

            // OKボタンが押されたらプロジェクト設定ファイルをロードする
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                // プロジェクト設定ファイルパスを設定
                txtProjectSettingFilePath.Text = dlg.FileName;

                // プロジェクト設定ファイルをロード
                ProjectSettings settings = OptionsManager.LoadProjectSetting(dlg.FileName);

                if (settings == null)
                {
                    settings = new ProjectSettings();
                }

                // UIを更新
                {
                    foreach (var item in lstDefinitionPath.ListView.Items.Cast<PathInfoListViewItem>())
                    {
                        item.Dispose();
                    }

                    lstDefinitionPath.ListView.Items.Clear();

                    if (settings.DefinitionPaths != null)
                    {
                        this.lstDefinitionPath.SetDefinitionPaths(settings.DefinitionPaths);
                    }

                    this.txtPreSaveCommandPath.Text = settings.PreSaveCommandPath;
                    this.txtPostSaveCommandPath.Text = settings.PostSaveCommandPath;
                    this.txtPreOpenCommandPath.Text = settings.PreOpenCommandPath;
                }
            }
        }

        /// <summary>
        /// プロジェクト設定のエクスポートボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void ExportProjectSettingsClick(object sender, EventArgs e)
        {
            // ファイル名入力ダイアログを表示
            var dlg = new SaveFileDialog
            {
                CheckFileExists = false,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.xml|{1}|*.*", Localization.Messages.DLG_FILTER_XML_FILES, Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PROJECT_SETTING_FILE_TITLE,
            };

            // プロジェクト設定ファイルを保存
            if (dlg.ShowDialog() == DialogResult.OK && string.IsNullOrEmpty(dlg.FileName) == false)
            {
                // 保存用のプロジェクト設定データを作成
                ProjectSettings settings = new ProjectSettings()
                {
                    DefinitionPaths = this.lstDefinitionPath.GetDefinitionPaths(),
                    PreSaveCommandPath = this.txtPreSaveCommandPath.Text,
                    PostSaveCommandPath = this.txtPostSaveCommandPath.Text,
                    PreOpenCommandPath = this.txtPreOpenCommandPath.Text,
                };

                // プロジェクト設定データを保存
                OptionsManager.SaveProjectSetting(dlg.FileName, settings);
            }
        }

        /// <summary>
        /// 定義パス追加ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void AddDefinitionsPathClick(object sender, EventArgs e)
        {
            this.lstDefinitionPath.AddOrEditDefinitionPath(true);
        }

        /// <summary>
        /// 保存前コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePreSaveCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PRE_SAVE_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPreSaveCommandPath.Text = dlg.FileName;
            }
        }

        /// <summary>
        /// 保存後コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePostSaveCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_POST_SAVE_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPostSaveCommandPath.Text = dlg.FileName;
            }
        }

        /// <summary>
        /// オープン前コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePreOpenCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PRE_OPEN_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPreOpenCommandPath.Text = dlg.FileName;
            }
        }
    }
}
