﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectCombiner.Editor.Controls
{
    public partial class TagEditorControl : UserControl
    {
        public TagEditorControl()
        {
            InitializeComponent();

            txtInput.KeyDown += InputKeyDown;
        }

        private static readonly Func<string, Control> defaultControlProducer =
            s =>
            {
                var label = new Label
                {
                    AutoSize = false,
                    Text = s,
                    TextAlign = ContentAlignment.MiddleLeft,
                };

                label.Size = TextRenderer.MeasureText(s, label.Font);
                label.Width += 4;
                label.Height += 6;

                return label;
            };

        public FixedPanel FixedPanel
        {
            get { return splitContainer1.FixedPanel; }
            set { splitContainer1.FixedPanel = value; }
        }

        public bool IsSplitterFixed
        {
            get { return splitContainer1.IsSplitterFixed; }
            set { splitContainer1.IsSplitterFixed = value; }
        }

        public int Panel1MinSize
        {
            get { return splitContainer1.Panel1MinSize; }
            set { splitContainer1.Panel1MinSize = value; }
        }

        public int Panel2MinSize
        {
            get { return splitContainer1.Panel2MinSize; }
            set { splitContainer1.Panel2MinSize = value; }
        }

        public int SplitterDistance
        {
            get { return splitContainer1.SplitterDistance; }
            set { splitContainer1.SplitterDistance = value; }
        }

        public event EventHandler<TagEventArgs> TagRemoved;

        public void ClearTags()
        {
            tagContainerControl.ClearControls();
        }

        private void InputKeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyValue == 13 && e.Modifiers == Keys.None &&
                string.IsNullOrWhiteSpace(txtInput.Text) == false)
            {
                AddTag(txtInput.Text);
                txtInput.Text = "";
            }
        }

        public void AddTag(string tag)
        {
            TagControl local = null;
            var control = defaultControlProducer(tag);
            Action remove = () =>
            {
                tagContainerControl.RemoveControl(local);
                var handler = TagRemoved;
                if (handler != null)
                    handler(this, new TagEventArgs(tag));
            };
            var btn = new TagControl(control, remove);
            local = btn;
            tagContainerControl.AddControl(btn);
        }

        public void SetTags(string[] tags)
        {
            ClearTags();
            foreach (var tag in tags)
                AddTag(tag);
        }

        public string[] GetTags()
        {
            return tagContainerControl.GetTags()
                .Cast<TagControl>()
                .Select(tc => tc.Content)
                .Select(lbl => lbl.Text)
                .ToArray();
        }
    }

    public class TagEventArgs : EventArgs
    {
        public string Tag { get; private set; }

        public TagEventArgs(string tag)
        {
            Tag = tag;
        }
    }
}
