﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// Globals RParameters LayerData

//=============================================================================
// include
//=============================================================================
#include "../NintendoFtx/NpsCommon.h"

#ifdef _MSC_VER
    #pragma warning(push)
    #pragma warning(disable: 4121) // メンバのアライメントは過剰にパッキングされています。
#endif

#include "PIFilter.h"

#ifdef _MSC_VER
    #pragma warning(pop)
#endif

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// plugin name & help
#define NPS_PLUGIN_NAME "NintendoNormalMapFilter"
#define NPS_PLUGIN_DESCRIPTION "Normal Map Filter Plugin"
#define NPS_PLUGIN_HELP_URL "html/NW4F_NormalMapFilter.html"

//=============================================================================
//! @brief パラメーターのクラスです。
//=============================================================================
class RParameters
{
public:
    //! @brief フィルター処理の種類を表す列挙型です。
    enum Operation
    {
        Operation_ConvertFromHeightMap, //!< 高さマップを法線マップに変換します。
        Operation_Normalize,            //!< 正規化します。
        Operation_ScaleSclope,          //!< 接空間法線マップの傾きをスケールします。
        Operation_CombineFile,          //!< 他の接空間法線マップのファイルと合成します。
        Operation_CombineLayer,         //!< 複数レイヤーの接空間法線マップを合成します。
        Operation_Count                 //!< フィルター処理の種類の数です。
    };

    //! @brief サンプリングする範囲を表す列挙型です。
    enum FilterType
    {
        FilterType_4,     //!< 4 ピクセルをサンプリングします。
        FilterType_3x3,   //!< 3x3 ピクセルをサンプリングします。
        FilterType_5x5,   //!< 5x5 ピクセルをサンプリングします。
        FilterType_7x7,   //!< 7x7 ピクセルをサンプリングします。
        FilterType_9x9,   //!< 9x9 ピクセルをサンプリングします。
        FilterType_13x13, //!< 13x13 ピクセルをサンプリングします。
        FilterType_17x17, //!< 17x17 ピクセルをサンプリングします。
        FilterType_21x21, //!< 21x21 ピクセルをサンプリングします。
        FilterType_25x25, //!< 25x25 ピクセルをサンプリングします。
        FilterType_Count  //!< サンプリングする範囲の種類の数です。
    };

    static const int CombineLayerCountMax = 5; //!< 合成するレイヤーの最大数です（ベースレイヤーも含みます）。
    static const int LayerNameLengthMax = 256; //!< レイヤー名の最大長です（終端の \0 を含みます）。
    static const std::string ActiveLayerName; //!< アクティブレイヤーを表す名前です。

public:
    // common
    Operation m_Operation; //!< フィルター処理の種類です。
    bool m_PositiveZ; //!< Z が 0 以上になるように調整して正規化するなら true です。

    // convert from height map
    int m_HeightScale; //!< 凹凸の高さを調整するスケール値です。
    FilterType m_FilterType; //!< サンプリングする範囲です。

    //! @brief 画像が繰り返しているとして端のピクセルを処理するなら true です。
    //!        画像の端のピクセルを引き伸ばして処理するなら false です。
    bool m_EdgeWrap;

    bool m_MultiplyAlpha; //!< アルファチャンネルを高さマップに乗算するなら true です。

    // scale slope
    int m_SlopeScale; //!< 傾きのスケール値です。

    // ファイル合成
    char m_OtherPath[MAX_PATH]; //!< 他の接空間法線マップのファイルパスです。
    int m_CombineScale; //!< 合成のスケール値です。

    // レイヤー合成
    char m_CombineLayerNames[CombineLayerCountMax][LayerNameLengthMax]; //!< 合成するレイヤーの名前配列です。
    int m_CombineLayerScales[CombineLayerCountMax]; //!< 合成するレイヤーのスケール値配列です（インデックス 0 の値は使用しません）。

public:
    //! @brief 初期化します。
    void Initialize();

    //! @brief 合成するレイヤーの名前をクリアします。
    //!
    //! @param[in] layerIdx レイヤーインデックスです。
    //!
    void ClearCombineLayerName(const int layerIdx)
    {
        memset(m_CombineLayerNames[layerIdx], 0x00, LayerNameLengthMax);
        if (layerIdx == 0)
        {
            strcpy_s(m_CombineLayerNames[layerIdx], ActiveLayerName.c_str());
        }
    }

    //! @brief 合成するレイヤーのスケール値をクリアします。
    //!
    //! @param[in] layerIdx レイヤーインデックスです。
    //!
    void ClearCombineLayerScale(const int layerIdx)
    {
        m_CombineLayerScales[layerIdx] = 100;
    }

    //! @brief レイヤー合成関連のパラメーターをクリアします。
    //!
    void ClearCombineLayerParams()
    {
        for (int layerIdx = 0; layerIdx < CombineLayerCountMax; ++layerIdx)
        {
            ClearCombineLayerName(layerIdx);
            ClearCombineLayerScale(layerIdx);
        }
    }

    //! 同値であれば true を返します。
    bool operator==(const RParameters& rhs) const
    {
        for (int layerIdx = 0; layerIdx < CombineLayerCountMax; ++layerIdx)
        {
            if (strncmp(m_CombineLayerNames[layerIdx], rhs.m_CombineLayerNames[layerIdx], LayerNameLengthMax) != 0)
            {
                return false;
            }
            if (layerIdx >= 1 && m_CombineLayerScales[layerIdx] != rhs.m_CombineLayerScales[layerIdx])
            {
                return false;
            }
        }
        return (
            m_Operation     == rhs.m_Operation     &&
            m_PositiveZ     == rhs.m_PositiveZ     &&
            m_HeightScale   == rhs.m_HeightScale   &&
            m_FilterType    == rhs.m_FilterType    &&
            m_EdgeWrap      == rhs.m_EdgeWrap      &&
            m_MultiplyAlpha == rhs.m_MultiplyAlpha &&
            m_SlopeScale    == rhs.m_SlopeScale    &&
            m_CombineScale  == rhs.m_CombineScale  &&
            strncmp(m_OtherPath, rhs.m_OtherPath, sizeof(m_OtherPath)) == 0);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RParameters& rhs) const
    {
        return !(*this == rhs);
    }
};

typedef RParameters* RParametersPtr;
typedef RParameters** RParametersHdl;

//=============================================================================
//! @brief レイヤーデータのクラスです。
//=============================================================================
class LayerData
{
public:
    std::string m_Name; //!< レイヤー名です。
    bool m_IsVisible; //!< 可視なら true です（親グループの可視性も反映した値です）。
    int m_Opacity; //!< 不透明度です（親グループの不透明度も反映した値です）。
    bool m_HasPriority; //!< 他の同名レイヤーより優先して使用するなら true です。
    std::vector<uint8_t> m_Pixels; //!< ピクセルデータです。
    const ReadLayerDesc* m_pLayerDesc; //!< リードレイヤー記述子へのポインターです。

public:
    //! @brief コンストラクタです。
    //!
    LayerData()
    : m_IsVisible(true),
      m_Opacity(0xff),
      m_HasPriority(true),
      m_pLayerDesc(nullptr)
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @param[in] pLayerDesc リードレイヤー記述子へのポインターです。
    //! @param[in] isVisible 可視なら true です（親グループの可視性も反映した値です）。
    //! @param[in] opacity 不透明度です（親グループの不透明度も反映した値です）。
    //!
    LayerData(
        const ReadLayerDesc* pLayerDesc,
        const bool isVisible,
        const int opacity
    )
    : m_Name(pLayerDesc->name),
      m_IsVisible(isVisible),
      m_Opacity(opacity),
      m_HasPriority(true),
      m_pLayerDesc(pLayerDesc)
    {
    }
};

//! @brief レイヤーデータ配列の定義です。
typedef std::vector<LayerData> LayerDataArray;

//=============================================================================
//! @brief グローバルデータのクラスです。
//=============================================================================
class Globals
{
public:
    // フィルター共通
    int16* result;                  // Must always be first in Globals.
    FilterRecord* filterParamBlock; // Must always be second in Globals.
    Boolean queryForParameters;

    // ドキュメント情報
    LayerDataArray* m_pLayerDataArray; //!< レイヤーデータ配列へのポインターです。
    nn::gfx::tool::nps::RStringArray* m_pLayerNameArray; //!< 合成可能なレイヤーのレイヤー名配列へのポインターです。
    int m_ActiveLayerIdx; //!< 合成可能なレイヤー名配列中のアクティブレイヤーのインデックスです（存在しなければ -1）。

    // ピクセルデータ
    VRect m_ImageBounds; //!< ピクセルデータの領域です。
    int m_ImageW; //!< ピクセルデータの幅です。
    int m_ImageH; //!< ピクセルデータの高さです。
    uint8_t* m_pSrcPixels; //!< フィルター処理前のピクセルデータです（RGBA 形式。A 成分はピクセルの不透明度）。
    uint8_t* m_pDstPixels; //!< フィルター処理後のピクセルデータです（RGBA 形式。A 成分はピクセルの不透明度）。
    uint8_t* m_pSrcAlphas; //!< アルファチャンネルのピクセルデータです。
    uint8_t* m_pGrayMatPixels; //!< グレーの背景（RGB = 128）とブレンドしたピクセルデータです（RGBA 形式。A 成分はピクセルの不透明度）。

    // プレビュー
    bool m_DisplaysPreview; //!< プレビューを表示するなら true です。
    bool m_LightingPreview; //!< ライティング結果をプレビューするなら true です。
    bool m_RotatesLightingPreview; //!< プレビューのライト方向を自動的に回転するなら true です。
    bool m_IsPreviewFiltered; //!< プレビューでフィルター処理済みなら true です。
    RParameters m_PreviewParams; //!< プレビューでフィルター処理したパラメーターです。
};

typedef Globals* GPtr;
typedef Globals** GHdl;
typedef void (*FProc)(GPtr globals);

// filter common
#define gResult (*(globals->result))       // SDK のマクロのため、かならず gResult という名前で定義します。
#define gStuff (globals->filterParamBlock) // SDK のマクロのため、かならず gStuff  という名前で定義します。
#define gDocInfo (gStuff->documentInfo)
#define gBigDoc (gStuff->bigDocumentData)
#define gParams ((RParametersPtr)*gStuff->parameters)
#define gQueryForParameters (globals->queryForParameters)

