﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ReadRegistryParameters WriteRegistryParameters

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
// constants
//=============================================================================
static const char* EXPORT_TEXTURE_REGISTRY_ID = "528544A-BCD4-4cd8-9A57-50B88273B4DB";
static const DescriptorKeyID EXPORT_TEXTURE_CONFIG_NAME_KEY = 'cfgN';

//=============================================================================
//! @brief Photoshop レジストリアクセスのクラスです。
//=============================================================================
class RRegistryAccess
{
protected:
    const SPBasicSuite* m_pBasicSuite; //!< ベーシックスートのポインタです。
    const PSDescriptorRegistryProcs* m_pRegistryProcs; //!< 記述子レジストリ関数群のポインタです。
    const PSActionDescriptorProcs* m_pDescProcs; //!< アクション記述子関数群のポインタです。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[out] pErr エラーコードを格納します。
    //! @param[in] basicSuite ベーシックスートのポインタです。
    //!
    RRegistryAccess(SPErr* pErr, const SPBasicSuite* basicSuite)
    : m_pBasicSuite(basicSuite),
      m_pRegistryProcs(NULL),
      m_pDescProcs(NULL)
    {
        if (basicSuite == NULL)
        {
            *pErr = errPlugInHostInsufficient;
            return;
        }

        *pErr = basicSuite->AcquireSuite(kPSDescriptorRegistrySuite,
            kPSDescriptorRegistrySuiteVersion, reinterpret_cast<const void**>(&m_pRegistryProcs));
        if (*pErr)
        {
            return;
        }

        *pErr = basicSuite->AcquireSuite(kPSActionDescriptorSuite,
            kPSActionDescriptorSuiteVersion, reinterpret_cast<const void**>(&m_pDescProcs));

        //RNoteTrace("RRegistryAccess() : %p %p", m_pRegistryProcs, m_pDescProcs);
    }

    //! デストラクタです。
    virtual ~RRegistryAccess()
    {
        //RNoteTrace("~RRegistryAccess(): %p %p", m_pRegistryProcs, m_pDescProcs);
        if (m_pRegistryProcs != NULL)
        {
            m_pBasicSuite->ReleaseSuite(kPSDescriptorRegistrySuite, kPSDescriptorRegistrySuiteVersion);
        }
        if (m_pDescProcs != NULL)
        {
            m_pBasicSuite->ReleaseSuite(kPSActionDescriptorSuite, kPSActionDescriptorSuiteVersion);
        }
    }

    //! レジストリを登録します。
    OSErr Register(const char* key, PIActionDescriptor descriptor, Boolean isPersistent) const
    {
        return m_pRegistryProcs->Register(key, descriptor, isPersistent);
    }

    //! レジストリを削除します。
    OSErr Erase(const char* key) const
    {
        return m_pRegistryProcs->Erase(key);
    }

    //! レジストリを取得します。
    OSErr Get(const char* key, PIActionDescriptor* descriptor) const
    {
        return m_pRegistryProcs->Get(key, descriptor);
    }

    //! アクション記述子関数群のポインタを返します。
    const PSActionDescriptorProcs* GetDescProcs() const
    {
        return m_pDescProcs;
    }
};

//-----------------------------------------------------------------------------
//! @brief アクション記述子の内容をダンプします（デバッグ用）。
//!
//! @param[in] desc アクション記述子です。
//! @param[in] pDescProcs アクション記述子関数群のポインタです。
//-----------------------------------------------------------------------------
inline void DumpActionDescriptor(const PIActionDescriptor desc, const PSActionDescriptorProcs* pDescProcs)
{
    uint32 keyCount1; // keyCount は PITerminology.h で define されているので注意
    pDescProcs->GetCount(desc, &keyCount1);
    for (int iKey = 0; iKey < static_cast<int>(keyCount1); ++iKey)
    {
        DescriptorKeyID keyId;
        pDescProcs->GetKey(desc, iKey, &keyId);
        //RNoteTrace("key%d: %08x", iKey, keyId);
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン（ライト時）のレジストリパラメータをリードします。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return エラーコードを返します。
//-----------------------------------------------------------------------------
SPErr ReadRegistryParameters(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // レジストリにアクセスするスートを取得します。
    SPErr error = noErr;
    RRegistryAccess access(&error, gStuff->sSPBasic);
    if (error)
    {
        return error;
    }
    const PSActionDescriptorProcs* pDescProcs = access.GetDescProcs();

    //-----------------------------------------------------------------------------
    // ファイル形式プラグイン共通のレジストリを取得します。
    PIActionDescriptor formatDesc = NULL;
    error = access.Get(plugInUniqueID, &formatDesc); // レジストリが存在しなくても返り値は noErr です。
    //RNoteTrace("format registry: %d: %p", error, formatDesc);
    if (!error && formatDesc != NULL)
    {
        //DumpActionDescriptor(formatDesc, pDescProcs);
        Boolean boolVal;
        error = pDescProcs->GetBoolean(formatDesc, keyPreviewFlag, &boolVal);
        if (!error)
        {
            globals->m_DisplaysPreview = (boolVal == TRUE);
            //RNoteTrace("display preview: %d", boolVal);
        }
    }
    if (formatDesc != NULL) pDescProcs->Free(formatDesc);

    //-----------------------------------------------------------------------------
    // テクスチャエクスポートプラグインのレジストリを取得します。
    PIActionDescriptor exportDesc = NULL;
    error = access.Get(EXPORT_TEXTURE_REGISTRY_ID, &exportDesc); // レジストリが存在しなくても返り値は noErr です。
    //RNoteTrace("export registry: %d: %p", error, exportDesc);
    if (!error && exportDesc != NULL)
    {
        //DumpActionDescriptor(exportDesc, pDescProcs);
        //uint32 strLen;
        //error = pDescProcs->GetStringLength(exportDesc, EXPORT_TEXTURE_CONFIG_NAME_KEY, &strLen);
        char strBuf[256];
        memset(strBuf, 0x00, sizeof(strBuf));
        error = pDescProcs->GetString(exportDesc, EXPORT_TEXTURE_CONFIG_NAME_KEY, strBuf, sizeof(strBuf));
            // GetString は ends も含めた長さが最後の引数で指定した長さ以下になるように取得します。
        //RNoteTrace("config name: %d: [%s]", error, strBuf);
        if (!error)
        {
            if (!g_AddInfo.m_IsConfigNameGot) // リソースデータからコンフィグ名を取得していなければ
            {
                g_AddInfo.m_ConfigName = strBuf;
            }
        }
    }
    if (exportDesc != NULL) pDescProcs->Free(exportDesc);

    return error;
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン（ライト時）のレジストリパラメータをライトします。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return エラーコードを返します。
//-----------------------------------------------------------------------------
SPErr WriteRegistryParameters(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // レジストリにアクセスするスートを取得します。
    SPErr error = noErr;
    RRegistryAccess access(&error, gStuff->sSPBasic);
    if (error)
    {
        return error;
    }
    const PSActionDescriptorProcs* pDescProcs = access.GetDescProcs();

    //-----------------------------------------------------------------------------
    // ファイル形式プラグイン共通のレジストリを登録します。
    PIActionDescriptor formatDesc = NULL;
    error = pDescProcs->Make(&formatDesc);
    if (!error)
    {
        error = pDescProcs->PutBoolean(formatDesc, keyPreviewFlag, globals->m_DisplaysPreview);
        if (!error)
        {
            error = access.Register(plugInUniqueID, formatDesc, TRUE);
        }
    }
    if (formatDesc != NULL) pDescProcs->Free(formatDesc);

    //-----------------------------------------------------------------------------
    // テクスチャエクスポートプラグインのレジストリを登録します。
    PIActionDescriptor exportDesc = NULL;
    error = access.Get(EXPORT_TEXTURE_REGISTRY_ID, &exportDesc); // レジストリが存在しなくても返り値は noErr です。
    if (!error && exportDesc == NULL) // レジストリが存在しなければ作成します。
    {
        error = pDescProcs->Make(&exportDesc);
    }
    if (!error && exportDesc != NULL)
    {
        char strBuf[256];
        strncpy_s(strBuf, sizeof(strBuf), g_AddInfo.m_ConfigName.c_str(), _TRUNCATE);
        error = pDescProcs->PutString(exportDesc, EXPORT_TEXTURE_CONFIG_NAME_KEY, strBuf);
        if (!error)
        {
            error = access.Register(EXPORT_TEXTURE_REGISTRY_ID, exportDesc, TRUE);
        }
    }
    if (exportDesc != NULL) pDescProcs->Free(exportDesc);

    return error;
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

