﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "Common.h"

//-----------------------------------------------------------------------------
//! @brief Maya 用のエクスポートオプションのクラスです。
//-----------------------------------------------------------------------------
class YExpOpt : public nn::gfx::tool::dcc::RExpOpt
{
public:
    //-----------------------------------------------------------------------------
    // project & scene
    std::string m_ProjectPath; //!< Maya のプロジェクトフォルダのパスです。
    std::string m_ProjectName; //!< Maya のプロジェクトフォルダの名前です。現在未使用です。
    std::string m_SceneName; //!< Maya のシーン名です。

    //-----------------------------------------------------------------------------
    // unit internal
    double m_InternalMagnify; //!< (cm から Maya 上の単位へのスケール) x m_Magnify です。

    //-----------------------------------------------------------------------------
    // anim internal
    double m_DoubleStartFrame; //!< Maya から取得した double 型の開始フレームです。
    double m_DoubleEndFrame; //!< Maya から取得した double 型の終了フレームです。
    MTime::Unit m_UIUnitTime; //!< Maya の UI の時間の単位です。
    float m_FramesPerSecond; //!< Maya 上の 1 秒間のフレーム数です。

    //-----------------------------------------------------------------------------
    // optimization internal
    bool m_FmdIsBindPose; //!< fmd ファイルをバインドポーズ状態で出力するなら true です。現在常に true です。
    bool m_ForceFullWeight; //!< // 一番ウェイトの大きいノードにウェイト 100% で出力するなら true です。現在常に false です。
    bool m_OutAllTex; //!< 使用されていないテクスチャも出力するなら true です。現在常に false です。

    //-----------------------------------------------------------------------------
    // user data
    nn::gfx::tool::dcc::RUserDataArray m_FileUserDatas;

    //-----------------------------------------------------------------------------
    // tmp folder

    //! @brief テンポラリフォルダ使用フラグです。
    //! 中間ファイルを一時的にテンポラリフォルダに出力し、
    //! 最後に本来のファイルとして移動させるなら true です。
    bool m_UseTmpFolderFlag;

    std::string m_TmpOutFolderPath; //!< モデルとアニメーションの中間ファイルを一時的に出力するテンポラリフォルダのパスです。
    std::string m_TmpTexFolderPath; //!< テクスチャ中間ファイルを一時的に出力するテンポラリフォルダのパスです。

    //-----------------------------------------------------------------------------
    // internal flag
    bool m_CancelledFlag; //!< 上書き確認ダイアログでキャンセルされたら true です。
    bool m_CheckElementFlag; //!< 実際に出力するのではなくシーンの要素が正しいかチェックするモードなら true です。

    //-----------------------------------------------------------------------------
    // for parse & check
    int m_SpecifiedStartFrame; //!< ユーザーによって指定された開始フレームです。
    int m_SpecifiedEndFrame; //!< ユーザーによって指定された終了フレームです。

public:
    //! エクスポートオプションに初期値を設定します。
    void Init();

    //! @brief テンポラリフォルダに出力する中間ファイルのパスを返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return テンポラリフォルダに出力する中間ファイルのパスを返します。
    //!
    std::string GetTmpFilePath(const FileType fileType) const
    {
        return m_TmpOutFolderPath + GetFileName(fileType);
    }

    //! @brief Export コマンドのオプション文字列を解析して、エクスポートオプションを設定します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //! @param[in] optionString オプション文字列です。
    //! @param[in] overwritesOption オプション上書きのための解析なら true、通常の解析なら false を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    MStatus Parse(
        nn::gfx::tool::dcc::RScene* pScene,
        const MString& optionString,
        const bool overwritesOption
    );

    //! @brief 出力ファイル（フォルダ）とマージファイル（フォルダ）のパスの
    //!        環境変数やマクロ文字を展開してフルパスに変換します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //! @param[in] checksSelectionError 出力ファイル名が @node でノードが選択されていない場合にエラーにするなら true です。
    //!
    //! @return 処理結果を返します。
    //!
    MStatus ExpandOutputAndMergePath(nn::gfx::tool::dcc::RScene* pScene, const bool checksSelectionError);

    //! @brief エクスポートオプションが有効か判定します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    //! @return オプションがすべて有効なら MS::kSuccess を返します。
    //!         無効なオプションが存在すればエラーを表示して MS::kFailure を返します。
    //!
    MStatus Check(nn::gfx::tool::dcc::RScene* pScene);

    //! @brief エクスポートオプションのマージファイルが有効か判定します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    //! @return 有効なら MS::kSuccess を返します。
    //!         無効ならエラーを表示して MS::kFailure を返します。
    //!
    MStatus CheckMergeFile(nn::gfx::tool::dcc::RScene* pScene);

protected:
    //! @brief Maya のアニメーションの開始／終了フレームを出力フレーム範囲として取得します。
    //!
    void GetMayaAnimationRange();

    //! @brief Maya のアニメーションの再生開始／再生終了フレームを出力フレーム範囲として取得します。
    //!
    void GetMayaPlaybackRange();
};

