﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// FindDeformerNodes
// GetBindPoseMtx

// GetBlendShapeData GetBlendShapeTargetFromNodeAttr
// GetBlendShapeUpdateFlag
// SetBlendShapeUpdateForShape
// GetBlendShapeUseVtxNrmFlags GetBlendShapeUseVtxTanFlags GetBlendShapeUseVtxColFlags
// GetBlendShapeFullAnimValue AnalyzeBlendShapeFullAnim GetBlendShapeKeyAnim
// OutputFshFile

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
// fixed
#define BS_ALLOW_DEL_TARGET_SW

//=============================================================================
// include
//=============================================================================
#include "Export.h"
#include "Deform.h"
#include <maya/MFnPointArrayData.h>
#include <maya/MFnComponentListData.h>

using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// blend shape

//!< ブレンドシェイプで MPoint 型の頂点座標を比較する際の許容値です。
const double BLEND_SHAPE_POS_SAME_TOL = 1.0e-5;

//!< ブレンドシェイプで MVector 型の法線を比較する際の許容値です。
const double BLEND_SHAPE_NRM_SAME_TOL = 1.0e-5;

//!< ブレンドシェイプで MFloatVector 型の接線（従法線）を比較する際の許容値です。
const float  BLEND_SHAPE_TAN_SAME_TOL = 1.0e-5f;

//!< ブレンドシェイプで MColor 型の頂点カラーを比較する際の許容値です。
const float  BLEND_SHAPE_COL_SAME_TOL = 1.0e-5f;

//-----------------------------------------------------------------------------
//! @brief シェイプのヒストリーに存在するデフォーマをすべて取得します。
//-----------------------------------------------------------------------------
bool FindDeformerNodes(CObjectArray& nodes, const MObject& shapeObj)
{
    MStatus status;
    bool isFound = false;

    // シェイプが mesh ノードなら inMesh アトリビュートを、
    // それ以外なら create アトリビュートの入力を調査します。
    const MString inShapeName = (shapeObj.apiType() == MFn::kMesh) ?
        "inMesh" : "create";

    // アトリビュートに接続されているオブジェクトを取得します。
    MFnDependencyNode depFn(shapeObj);
    MPlug inShapePlug = depFn.findPlug(inShapeName);
    MPlugArray plugArray;
    inShapePlug.connectedTo(plugArray, true, false);
    if (plugArray.length() == 0)
    {
        // 見つからなかった場合はそのまま関数を終了します。
        return isFound;
    }

    MObject inObj = plugArray[0].node();
    for (;;)
    {
        // 接続されていたオブジェクトのタイプを取得し、
        // そのオブジェクトの DependencyNode を作成します。
        MFn::Type type = inObj.apiType();
        depFn.setObject(inObj);

        // メッシュが接続されていたら検索を中止します。
        if (type == MFn::kMesh)
        {
            break;
        }

        // 接続されていたオブジェクトがスムーススキンかブレンドシェイプなら
        // デフォーマとして登録します。
        if (type == MFn::kSkinClusterFilter ||
        //  type == MFn::kJointCluster      ||
        //  type == MFn::kClusterFilter     ||
            type == MFn::kBlendShape)
        {
            isFound = true;
            nodes.push_back(inObj);
            //cerr << "find: " << depFn.name() << R_ENDL;
        }

        // さらに inputGeometry か inputPolymesh か inMesh(create)
        // に接続されているオブジェクトを探索します。
        MPlug inputPlug;
        inputPlug = FindPlugQuiet(depFn, "inputGeometry", &status);
        if (!status)
        {
            inputPlug = FindPlugQuiet(depFn, "inputPolymesh", &status);
            if (!status)
            {
                inputPlug = FindPlugQuiet(depFn, inShapeName, &status);
            }
        }
        // 接続されているオブジェクトがなければ探索を終了します。
        if (!status)
        {
            break;
        }

        // さらに接続されているオブジェクトを探索します。
        inputPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() == 0)
        {
            break;
        }
        inObj = plugArray[0].node();
    }

    return isFound;
}

//-----------------------------------------------------------------------------
//! @brief デフォーマオブジェクト群を無効にします。
//-----------------------------------------------------------------------------
void DisableDeformerStates(MIntArray& nodeStates, const CObjectArray& nodes)
{
    const int deformerCount = static_cast<int>(nodes.size());
    for (int iDeformer = 0; iDeformer < deformerCount; ++iDeformer)
    {
        MPlug plug = MFnDependencyNode(nodes[iDeformer]).findPlug("nodeState");
        int nodeState;
        plug.getValue(nodeState);
        nodeStates.append(nodeState);
        nodeState = 1; // HasNoEffect
        plug.setValue(nodeState);
    }
}

//-----------------------------------------------------------------------------
//! @brief デフォーマオブジェクト群の有効状態を元に戻します。
//-----------------------------------------------------------------------------
void RestoreDeformerStates(const CObjectArray& nodes, const MIntArray& nodeStates)
{
    const int deformerCount = static_cast<int>(nodes.size());
    for (int iDeformer = 0; iDeformer < deformerCount; ++iDeformer)
    {
        const int nodeState = nodeStates[iDeformer];
        MFnDependencyNode(nodes[iDeformer]).findPlug("nodeState").setValue(nodeState);
    }
}

//-----------------------------------------------------------------------------
//! @brief 各頂点がデフォーマのメンバーかどうかのフラグ配列を取得します。
//!
//! @param[out] isMembers 各頂点がデフォーマのメンバーかどうかのフラグ配列を格納します。
//!                       シェイプの頂点数と同じ長さの配列が格納されます。
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] deformerObj デフォーマオブジェクトです。
//!
//! @return デフォーマのメンバーである頂点数を返します。
//-----------------------------------------------------------------------------
static int GetDeformerMemberVertexFlags(
    MIntArray& isMembers,
    const MDagPath& shapePath,
    const MObject& deformerObj
)
{
    int memberCount = 0;

    //-----------------------------------------------------------------------------
    // メンバフラグを作成し、値を初期化します。
    int vtxCount = MFnMesh(shapePath).numVertices();
    isMembers = MIntArray(vtxCount, 0);

    //-----------------------------------------------------------------------------
    // デフォーマに接続されているセットオブジェクトを取得します。
    MObject setObj;
    MFnDependencyNode depFn(deformerObj);
    // deformer.message -> objectSet.usedBy[n]
    MPlugArray plugArray;
    depFn.findPlug("message").connectedTo(plugArray, false, true);
    for (int iplug = 0; iplug < static_cast<int>(plugArray.length()); ++iplug)
    {
        const MObject obj = plugArray[iplug].node();
        // プラグの接続先のオブジェクトタイプが kSet で verticesOnlySet アトリビュート
        // に 1 が設定されているものを探します。
        if (obj.apiType() == MFn::kSet)
        {
            MFnSet setFn(obj);
            bool vtxOnlyFlag;
            setFn.findPlug("verticesOnlySet").getValue(vtxOnlyFlag);
            if (vtxOnlyFlag)
            {
                setObj = obj;
                break;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // セットオブジェクトからブレンドシェイプが適用される頂点のメンバを取得します。
    if (!setObj.isNull())
    {
        // セットオブジェクトに含まれる全メンバの頂点コンポーネントを取得し、
        // ブレンドシェイプによる変改が与えられる頂点にフラグを設定します。
        MSelectionList slist;
        MFnSet setFn(setObj);
        setFn.getMembers(slist, true);
        MItSelectionList sIter(slist);
        for ( ; !sIter.isDone(); sIter.next())
        {
            // セットオブジェクトのメンバから頂点コンポーネントを取得します。
            MDagPath dagPath;
            MObject comp;
            sIter.getDagPath(dagPath, comp);
            if (dagPath == shapePath &&
                comp.apiType() == MFn::kMeshVertComponent)
            {
                // 頂点コンポーネントに含まれている頂点にフラグを設定します。
                MIntArray elements;
                MFnSingleIndexedComponent(comp).getElements(elements);
                int elementCount = elements.length();
                for (int ielm = 0; ielm < elementCount; ++ielm)
                {
                    int iVtx = elements[ielm];
                    if (iVtx < static_cast<int>(isMembers.length()))
                    {
                        isMembers[iVtx] = 1;
                    }
                }
                memberCount += elementCount;
                break;
            }
        }
    }

    return memberCount;
}

//-----------------------------------------------------------------------------
//! @brief 各フェースがブレンドシェイプのメンバーかどうかのフラグ配列を取得します。
//!
//! @param[in,out] bsData ブレンドシェイプデータです。
//-----------------------------------------------------------------------------
static void GetBlendShapeMemberFaceFlags(YBlendShapeData& bsData)
{
    MItMeshPolygon pIter(bsData.m_BasePath);
    bsData.m_IsMemberFaces = MIntArray(pIter.count(), 0);

    for ( ; !pIter.isDone(); pIter.next())
    {
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
        {
            // フェースの頂点が 1 つでもブレンドシェイプのメンバーなら
            // そのフェースはブレンドシェイプのメンバーとみなします。
            if (bsData.m_IsMemberVtxs[pIter.vertexIndex(iVtx)])
            {
                bsData.m_IsMemberFaces[pIter.index()] = 1;
                break;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! not used
//! @brief デフォーマの後のヒストリーでコンポーネントが削除されているか
//!        チェックします。
//-----------------------------------------------------------------------------
//bool CheckDeleteComponentAfterDeformer(const MObject& shapeObj,
//  const MObject& deformerObj)
//{
//  MStatus status;
//
//  MString inShapeName;
//  if (shapeObj.apiType() == MFn::kMesh)
//      inShapeName = "inMesh";
//  else
//      inShapeName = "create";
//  MFnDependencyNode depFn(shapeObj);
//  MPlug inShapePlug = depFn.findPlug(inShapeName);
//  MPlugArray plugArray;
//  inShapePlug.connectedTo(plugArray, true, false);
//  if (plugArray.length() == 0)
//      return false;
//
//  MObject inObj = plugArray[0].node();
//  while (inObj != deformerObj)
//  {
//      MFn::Type type = inObj.apiType();
//      if (type == MFn::kDeleteComponent)
//          return true;
//      depFn.setObject(inObj);
//      MPlug inputPlug;
//      inputPlug = FindPlugQuiet(depFn, "inputGeometry", &status);
//      if (!status)
//      {
//          inputPlug = FindPlugQuiet(depFn, "inputPolymesh", &status);
//          if (!status)
//              inputPlug = FindPlugQuiet(depFn, "inMesh", &status);
//      }
//      if (!status)
//          return false;
//      inputPlug.connectedTo(plugArray, true, false);
//      if (plugArray.length() == 0)
//          return false;
//      inObj = plugArray[0].node();
//  }
//  return false;
//}

//-----------------------------------------------------------------------------
//! @brief 指定したシェイプノードのスキニングに影響するオブジェクト（joint ノードも）
//!        をすべて取得します。
//-----------------------------------------------------------------------------
int GetSkinInfluencePathsForShape(CDagPathArray& infPaths, const MDagPath& shapePath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // get deformer
    CObjectArray deformerObjs;
    FindDeformerNodes(deformerObjs, shapePath.node());

    //-----------------------------------------------------------------------------
    // loop for deformer
    infPaths.clear();
    const int deformerCount = static_cast<int>(deformerObjs.size());
    for (int iDeformer = 0; iDeformer < deformerCount; ++iDeformer)
    {
        const MObject& deformerObj = deformerObjs[iDeformer];
        if (deformerObj.apiType() == MFn::kSkinClusterFilter &&
            IsNodeStateEffective(deformerObj))
        {
            MFnSkinCluster skinFn(deformerObj);
            MDagPathArray infs;
            skinFn.influenceObjects(infs, &status); // status は必ず必要
            const int infCount = infs.length();
            for (int iInf = 0; iInf < infCount; ++iInf)
            {
                infPaths.push_back(infs[iInf]);
            }
        }
    }

    return static_cast<int>(infPaths.size());
}

//-----------------------------------------------------------------------------
//! @brief シーン中のスキニングが設定されたオブジェクトをすべて取得します。
//-----------------------------------------------------------------------------
void GetSkinedObjectPaths(CDagPathArray& skinedObjPaths)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // skinCluster ノードを列挙するイテレータを作成します。
    MItDependencyNodes dnIter(MFn::kSkinClusterFilter, &status);
    if (status != MS::kSuccess)
    {
        // イテレータの作成に失敗した場合はシーン中に
        // スキニングが設定されたオブジェクトがないということなのでそのまま終了します。
        return;
    }

    for ( ; !dnIter.isDone(); dnIter.next())
    {
        MFnSkinCluster skinFn(dnIter.item());
        MObjectArray objs;
        if (skinFn.getOutputGeometry(objs) != MS::kSuccess)
        {
            // ジオメトリに接続されていない skinCluster ノードは除外します。
            continue;
        }
        // skinCluster ノードが接続されている全てのジオメトリを列挙し、同じオブジェクトが
        // 重複しないようにそのジオメトリの親の transform ノード の DAG パス取得します。
        const int objCount = objs.length();
        for (int iObj = 0; iObj < objCount; ++iObj)
        {
            // skinCluster ノードが接続されているジオメトリがインスタンス化
            // されている場合は、全てのパスを取得します。
            MDagPathArray allPaths;
            MFnDagNode(objs[iObj]).getAllPaths(allPaths);
            const int pathCount = allPaths.length();
            for (int iPath = 0; iPath < pathCount; ++iPath)
            {
                // ジオメトリの DAG パスの親(transformノード)を取得します。
                MDagPath dagPath = allPaths[iPath];
                dagPath.pop(); // shape -> transform
                if (std::find(skinedObjPaths.begin(), skinedObjPaths.end(), dagPath) == skinedObjPaths.end())
                {
                    skinedObjPaths.push_back(dagPath);
                    //cerr << "skined: " << dagPath.fullPathName() << R_ENDL;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シーン中のスキニングのインフルエンスオブジェクト（joint ノード以外）
//!        をすべて取得します。
//-----------------------------------------------------------------------------
void GetSkinInfluencePaths(CDagPathArray& infPaths)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // シーン中のスキニングが設定されたオブジェクトをすべて取得します。
    CDagPathArray skinedObjPaths;
    GetSkinedObjectPaths(skinedObjPaths);

    //-----------------------------------------------------------------------------
    // skinCluster ノードを列挙するイテレータを作成します。
    MItDependencyNodes dnIter(MFn::kSkinClusterFilter, &status);
    if (!status)
    {
        // イテレータの作成に失敗した場合はシーン中に
        // スキニングが設定されたオブジェクトがないということなのでそのまま終了します。
        return;
    }

    for ( ; !dnIter.isDone(); dnIter.next())
    {
        MFnSkinCluster skinFn(dnIter.item());
        MDagPathArray objs;
        // 一つの skinCluster ノードからインフルエンスオブジェクト配列を取得します。
        unsigned int objCount = skinFn.influenceObjects(objs, &status);
        //cerr << skinFn.name() << " " << objCount << R_ENDL;
        if (status != MS::kSuccess)
        {
            continue;
        }

        // 列挙されたインフルエンスオブジェクトを配列に登録します。
        for (int iObj = 0; iObj < static_cast<int>(objCount); ++iObj)
        {
            MDagPath& dagPath = objs[iObj];
            // インフルエンスオブジェクトが joint ノードなら登録しません。
            if (dagPath.node().hasFn(MFn::kJoint))
            {
                continue;
            }
            // スキンされたオブジェクトがインフルエンスオブジェクトでもある場合はパス
            if (std::find(skinedObjPaths.begin(), skinedObjPaths.end(),
                dagPath) != skinedObjPaths.end())
            {
                continue;
            }
            // 同じオブジェクトが重複しないように DAG パス を infPaths に追加します。
            if (std::find(infPaths.begin(), infPaths.end(), dagPath) == infPaths.end())
            {
                infPaths.push_back(dagPath);
                //cerr << "inf: " << dagPath.partialPathName() << R_ENDL;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定したクラスターオブジェクトの outputGeometry アトリビュートが
//!        他のアトリビュートに接続されていれば true を返します。
//!        つまりクラスターオブジェクトが使用されていれば true を返します。
//-----------------------------------------------------------------------------
static bool IsClusterOutputConnected(const MObject& clusterObj)
{
    MFnDependencyNode clusterFn(clusterObj);
    MPlug ogPlug = clusterFn.findPlug("outputGeometry");
    //cerr << "og: " << ogPlug.info() << ": " << ogPlug.numElements() << R_ENDL;
    const int ogCount = ogPlug.numElements();
    for (int iOg = 0; iOg < ogCount; ++iOg)
    {
        MPlug ognPlug = ogPlug.elementByPhysicalIndex(iOg);
        MPlugArray outs;
        ognPlug.connectedTo(outs, false, true);
        if (outs.length() > 0)
        {
            //cerr << "og: " << ognPlug.info() << " -> " << outs[0].info() << R_ENDL;
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 出力対象に影響する skinCluster オブジェクトなら true を返します。
//!
//! @param[in] effectiveSkinClusters 出力対象に影響する skinCluster オブジェクト配列です。
//! @param[in] obj 判定する skinCluster オブジェクトです。
//!
//! @return 出力対象に影響する skinCluster オブジェクトなら true を返します。
//!         effectiveSkinClusters が空の場合も true を返します。
//-----------------------------------------------------------------------------
static bool IsEffectiveSkinCluster(
    const MObjectArray& effectiveSkinClusters,
    const MObject& obj
)
{
    return (effectiveSkinClusters.length() == 0 ||
        FindObjectInArray(effectiveSkinClusters, obj) != -1);
}

//-----------------------------------------------------------------------------
//! @brief 複数のバインドポーズが使用されていれば true を返します。
//-----------------------------------------------------------------------------
bool IsMultipleBindPoseUsed(
    const MDagPath& xformPath,
    const MObjectArray& effectiveSkinClusters
)
{
    //-----------------------------------------------------------------------------
    // get world matrix [0] connection
    MFnTransform xformFn(xformPath);
    MPlug wmPlug = xformFn.findPlug("worldMatrix");
    MPlugArray plugArray;
    wmPlug.elementByLogicalIndex(0).connectedTo(plugArray, false, true);

    //-----------------------------------------------------------------------------
    // check all bind pre matrix of connected cluster nodes are same
    bool isFound = false;
    MMatrix bindPreMtx;
    for (int iplug = 0; iplug < static_cast<int>(plugArray.length()); ++iplug)
    {
        const MObject obj = plugArray[iplug].node();
        if (obj.apiType() == MFn::kSkinClusterFilter)
        {
            if (IsClusterOutputConnected(obj) &&
                IsEffectiveSkinCluster(effectiveSkinClusters, obj))
            {
                MPlug pmPlug = MFnDependencyNode(obj).findPlug("bindPreMatrix");
                const int plugIndex = plugArray[iplug].logicalIndex();
                pmPlug = pmPlug.elementByLogicalIndex(plugIndex);
                MObject pmObj;
                if (!pmPlug.getValue(pmObj))
                {
                    continue;
                }
                MMatrix pmMtx = MFnMatrixData(pmObj).matrix();
                //cerr << "imbp: " << xformPath.partialPathName() << ": " << MFnDependencyNode(obj).name() << ": " << pmMtx << R_ENDL;
                if (!isFound)
                {
                    isFound = true;
                    bindPreMtx = pmMtx;
                }
                else if (!pmMtx.isEquivalent(bindPreMtx, R_SAME_TOLERANCE))
                {
                    return true;
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief バインドポーズのノード行列を取得します。
//-----------------------------------------------------------------------------
MStatus GetBindPoseMtx(
    MMatrix& mtx,
    RScene& rscene,
    const MDagPath& xformPath,
    const MObjectArray& effectiveSkinClusters,
    const double magnify
)
{
    //-----------------------------------------------------------------------------
    // find cluster node
    //      joint.wm[0] -> skinCluster.matrix[n]
    MFnTransform xformFn(xformPath);
    MPlug wmPlug = xformFn.findPlug("worldMatrix");
    MPlugArray plugArray;
    wmPlug.elementByLogicalIndex(0).connectedTo(plugArray, false, true);

    MObject clusterObj;
    int plugIndex = 0;
    MObject firstClusterObj;
    int firstPlugIndex = 0;
    for (int iplug = 0; iplug < static_cast<int>(plugArray.length()); ++iplug)
    {
        const MObject obj = plugArray[iplug].node();
        if (obj.apiType() == MFn::kSkinClusterFilter)
        {
            if (IsClusterOutputConnected(obj))
            {
                if (IsEffectiveSkinCluster(effectiveSkinClusters, obj))
                {
                    clusterObj = obj;
                    plugIndex = plugArray[iplug].logicalIndex();
                    //cerr << "gbpm: " << xformPath.partialPathName() << ": " << MFnDependencyNode(clusterObj).name() << R_ENDL;
                    break;
                }
                else if (firstClusterObj.isNull())
                {
                    firstClusterObj = obj;
                    firstPlugIndex = plugArray[iplug].logicalIndex();
                }
            }
        }
    }
    if (clusterObj.isNull())
    {
        if (firstClusterObj.isNull())
        {
            return MS::kFailure;
        }
        else
        {
            // transform ノードに接続された skinCluster オブジェクトが
            // どれも出力対象に影響しない場合、plugArray 内で先頭のものから行列を取得します。
            clusterObj = firstClusterObj;
            plugIndex = firstPlugIndex;
        }
    }

    //-----------------------------------------------------------------------------
    // get bind pre matrix
    MPlug pmPlug = MFnDependencyNode(clusterObj).findPlug("bindPreMatrix");
    pmPlug = pmPlug.elementByLogicalIndex(plugIndex);
    MObject pmObj;
    if (pmPlug.getValue(pmObj) != MS::kSuccess)
    {
        YShowWarning(&rscene, // Get bind pre matrix failed: %s // 通常は発生しない
            "bindPreMatrix の取得に失敗しました: {0}",
            "Failed to get bindPreMatrix: {0}",
            pmPlug.info().asChar());
        return MS::kFailure;
    }
    MMatrix mtxPm = MFnMatrixData(pmObj).matrix();

    // return inverse
    mtx = mtxPm.inverse();

    // magnify
    mtx[3][0] *= magnify;
    mtx[3][1] *= magnify;
    mtx[3][2] *= magnify;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief バインドポーズのセグメントスケール補正を取得します（未完成）。
//-----------------------------------------------------------------------------
MStatus GetBindPoseScaleCompensate(bool& scaleCompensate, const MDagPath& xformPath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // find bind pose node (dagPose node)
    // joint.bindPose -> dagPose.worldMatrix[n] -> dagPose.xformMatrix[n]
    MFnTransform xformFn(xformPath);
    MPlug bpsPlug = xformFn.findPlug("bindPose");
    MPlugArray plugArray;
    bpsPlug.connectedTo(plugArray, false, true);

    MObject bpsObj = MObject::kNullObj;
    int plugIndex = 0;
    for (int iplug = 0; iplug < static_cast<int>(plugArray.length()); ++iplug)
    {
        MObject obj = plugArray[iplug].node();
        if (obj.apiType() == MFn::kDagPose)
        {
            bool bpsFlag;
            MFnDependencyNode(obj).findPlug("bindPose").getValue(bpsFlag);
            if (bpsFlag)
            {
                bpsObj = obj;
                plugIndex = plugArray[iplug].logicalIndex();
                break;
            }
        }
    }
    if (bpsObj.isNull())
    {
        return MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // get local xform matrix
    MPlug xmPlug = MFnDependencyNode(bpsObj).findPlug("xformMatrix");
    xmPlug = xmPlug.elementByLogicalIndex(plugIndex, &status);
    CheckStatus(status);
    MObject xmObj;
    status = xmPlug.getValue(xmObj);
    CheckStatus(status);
    MTransformationMatrix xformMtx = MFnMatrixData(xmObj).transformation();

    //cerr << xformPath.partialPathName() << ": " << xmPlug.name() << R_ENDL;

    // ssc を取得する方法は？
    // scaleCompensate = ?;
    R_UNUSED_VARIABLE(scaleCompensate);

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 指定した transform ノードの子のシェイプノードが
//!        ブレンドシェイプのターゲットシェイプとして参照されているなら true を返します。
//-----------------------------------------------------------------------------
bool IsBlendShapeTarget(const MDagPath& xformPath)
{
    const int childCount = xformPath.childCount();
    for (int iChild = 0; iChild < childCount; ++iChild)
    {
        MObject childObj = xformPath.child(iChild);
        if (childObj.apiType() == MFn::kMesh)
        {
            // worldMesh アトリビュートが blendShape ノードに接続されていれば
            // ターゲットシェイプとして参照されています。
            MFnDependencyNode depFn(childObj);
            MPlug worldMeshPlug = depFn.findPlug("worldMesh");
            MPlugArray plugArray;
            worldMeshPlug[0].connectedTo(plugArray, false, true);
            if (plugArray.length() > 0 &&
                plugArray[0].node().apiType() == MFn::kBlendShape)
            {
                //cerr << "blend shape target: " << xformPath.partialPathName() << R_ENDL;
                return true;
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプの頂点属性更新情報を blendShape ノードの
//!        カスタムアトリビュートから取得します。
//!
//! @param[in] blendShapeObj blendShape ノードです。
//!
//! @return シェイプアニメーションの頂点属性更新情報を返します。
//-----------------------------------------------------------------------------
static RShapeUpdate GetBlendShapeUpdateFlag(const MObject& blendShapeObj)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 更新フラグを初期化します。
    // デフォルトでは頂点座標と法線（接線・従法線）のみ更新します。
    RShapeUpdate shapeUpdate;
    shapeUpdate.m_UpdatesPos = true;
    shapeUpdate.m_UpdatesNrm = true;
    shapeUpdate.m_UpdatesTan = true;
    shapeUpdate.m_UpdatesBin = true;
    shapeUpdate.SetAllCols(false);

    MFnDependencyNode blendFn(blendShapeObj);

    //-----------------------------------------------------------------------------
    // 頂点座標の更新フラグを取得します。
    MPlug updatePosPlug = FindPlugQuiet(blendFn, "nw4fUpdatePos", &status);
    if (status)
    {
        updatePosPlug.getValue(shapeUpdate.m_UpdatesPos);
    }

    //-----------------------------------------------------------------------------
    // 法線の更新フラグを取得します。
    // 接線と従法線も法線の設定に従います。
    MPlug updateNrmPlug = FindPlugQuiet(blendFn, "nw4fUpdateNrm", &status);
    if (status)
    {
        updateNrmPlug.getValue(shapeUpdate.m_UpdatesNrm);
        shapeUpdate.m_UpdatesTan = shapeUpdate.m_UpdatesBin = shapeUpdate.m_UpdatesNrm;
    }

    //-----------------------------------------------------------------------------
    // 頂点カラーの更新フラグを取得します。
    MPlug updateColPlug = FindPlugQuiet(blendFn, "nw4fUpdateCol", &status);
    if (status)
    {
        bool updatesCol;
        updateColPlug.getValue(updatesCol);
        shapeUpdate.SetAllCols(updatesCol);
    }

    return shapeUpdate;
}

//-----------------------------------------------------------------------------
//! @brief 頂点単位の法線の使用フラグ配列を取得します。
//!
//! @param[in,out] bsData ブレンドシェイプデータです。
//-----------------------------------------------------------------------------
static void GetBlendShapeUseVtxNrmFlags(YBlendShapeData& bsData)
{
    MStatus status;

    // 頂点単位の法線を使用する状態で初期化します。
    bsData.m_UseVtxNrmFlags = MIntArray(bsData.m_VtxCount, YBlendShapeData::PER_VTX);

    const int shapeCount = 1 + static_cast<int>(bsData.m_Targets.size()); // 1:ベースシェイプ
    for (int iShape = 0; iShape < shapeCount; ++iShape)
    {
        const MDagPath shapePath = (iShape == 0) ?
            bsData.m_BasePath : bsData.m_Targets[iShape - 1].m_ShapePath;
        //cerr << "bs nrm: " << shapePath.partialPathName() << ": " << MFnMesh(shapePath).numNormals() << ", " << bsData.m_VtxCount << R_ENDL;
        if (MFnMesh(shapePath).numNormals() == bsData.m_VtxCount)
        {
            // 法線数と頂点数が一致していれば全てソフトエッジなので何もしません。
            continue;
        }

        MItMeshVertex vIter(shapePath);
        for ( ; !vIter.isDone(); vIter.next())
        {
            const int iVtxObj = vIter.index();
            int& flag = bsData.m_UseVtxNrmFlags[iVtxObj];
            if (flag == YBlendShapeData::PER_VTX_FACE) // すでに頂点フェース単位を使用
            {
                continue;
            }
            MIntArray faces;
            vIter.getConnectedFaces(faces);
            const int faceCount = faces.length();
            if (faceCount == 1) // 1 つのフェースにのみ属する
            {
                continue;
            }
            MVector firstNrm;
            for (int iFace = 0; iFace < faceCount; ++iFace)
            {
                const int iFaceObj = faces[iFace];
                MVector nrm;
                status = vIter.getNormal(nrm, iFaceObj, MSpace::kObject);
                //cerr << "nrm: " << iVtxObj << ", " << iFaceObj << ", " << nrm << ", " << status << R_ENDL;
                if (iFace == 0)
                {
                    firstNrm = nrm;
                }
                else if (!nrm.isEquivalent(firstNrm, BLEND_SHAPE_NRM_SAME_TOL))
                {
                    flag = YBlendShapeData::PER_VTX_FACE;
                    //cerr << "nrm vtx face: " << shapePath.partialPathName() << ".vtx[" << iVtxObj << "]" << endl;
                    //cerr << firstNrm << endl << nrm << endl;
                    break;
                }
            }
        }
    }
    //cerr << "use vtx nrm: " << R_ENDL << bsData.m_UseVtxNrmFlags << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 頂点単位の接線（従法線）の使用フラグ配列を取得します。
//!        高速化のためにベースシェイプの接線（従法線）は
//!        出力用に取得したものを再利用します。
//-----------------------------------------------------------------------------
void GetBlendShapeUseVtxTanFlags(
    MIntArray& useVtxTanFlags,
    const YBlendShapeData& bsData,
    const MStringArray& tanUvSetNames,
    const MFloatVectorArray& baseTans,
    const MFloatVectorArray& baseBins
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 頂点単位の接線（従法線）を使用する状態で初期化します。
    useVtxTanFlags = MIntArray(bsData.m_VtxCount, YBlendShapeData::PER_VTX);

    //-----------------------------------------------------------------------------
    // ベースシェイプと各ターゲットシェイプについて処理します。
    const int shapeCount = 1 + static_cast<int>(bsData.m_Targets.size()); // 1:ベースシェイプ
    for (int iShape = 0; iShape < shapeCount; ++iShape)
    {
        const MDagPath shapePath = (iShape == 0) ?
            bsData.m_BasePath : bsData.m_Targets[iShape - 1].m_ShapePath;

        MFnMesh meshFn(shapePath);
        const MString tanUvSetName = tanUvSetNames[iShape]; // C4238 回避のため一度コピー
        MFloatVectorArray tans;
        MFloatVectorArray bins;
        if (iShape == 0)
        {
            tans = baseTans;
            bins = baseBins;
        }
        else
        {
            meshFn.getTangents (tans, MSpace::kObject, &tanUvSetName);
            meshFn.getBinormals(bins, MSpace::kObject, &tanUvSetName);
        }

        MItMeshVertex vIter(shapePath);
        for ( ; !vIter.isDone(); vIter.next())
        {
            const int iVtxObj = vIter.index();
            int& flag = useVtxTanFlags[iVtxObj];
            if (flag == YBlendShapeData::PER_VTX_FACE) // すでに頂点フェース単位を使用
            {
                continue;
            }
            MIntArray faces;
            vIter.getConnectedFaces(faces);
            const int faceCount = faces.length();
            if (faceCount == 1) // 1 つのフェースにのみ属する
            {
                continue;
            }
            MFloatVector firstTan;
            MFloatVector firstBin;
            for (int iFace = 0; iFace < faceCount; ++iFace)
            {
                const int iFaceObj = faces[iFace];
                const int iTan = meshFn.getTangentId(iFaceObj, iVtxObj);
                const MFloatVector& tan = tans[iTan];
                const MFloatVector& bin = bins[iTan];
                if (iFace == 0)
                {
                    firstTan = tan;
                    firstBin = bin;
                }
                else if (!tan.isEquivalent(firstTan, BLEND_SHAPE_TAN_SAME_TOL) ||
                         !bin.isEquivalent(firstBin, BLEND_SHAPE_TAN_SAME_TOL))
                {
                    flag = YBlendShapeData::PER_VTX_FACE;
                    //cerr << "tan vtx face: " << shapePath.partialPathName() << ".vtx[" << iVtxObj << "]" << endl;
                    //cerr << firstTan << endl << tan << endl;
                    //cerr << firstBin << endl << bin << endl;
                    break;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点単位の頂点カラーの使用フラグ配列を取得します。
//!
//! @param[in,out] bsData ブレンドシェイプデータです。
//-----------------------------------------------------------------------------
static void GetBlendShapeUseVtxColFlags(YBlendShapeData& bsData)
{
    MStatus status;

    for (int iColSet = 0; iColSet < RPrimVtx::VTX_COL_MAX; ++iColSet)
    {
        //-----------------------------------------------------------------------------
        // ベースシェイプのカラーセット名が空文字なら処理しません。
        const MString& baseColSet = bsData.m_BaseColSets[iColSet];
        if (baseColSet.length() == 0)
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // 頂点単位の頂点カラーを使用する状態で初期化します。
        MIntArray& useVtxColFlags = bsData.m_UseVtxColFlagss[iColSet];
        useVtxColFlags = MIntArray(bsData.m_VtxCount, YBlendShapeData::PER_VTX);

        //-----------------------------------------------------------------------------
        // ベースシェイプと各ターゲットシェイプについて処理します。
        const int shapeCount = 1 + static_cast<int>(bsData.m_Targets.size()); // 1:ベースシェイプ
        for (int iShape = 0; iShape < shapeCount; ++iShape)
        {
            MDagPath shapePath = (iShape == 0) ?
                bsData.m_BasePath : bsData.m_Targets[iShape - 1].m_ShapePath;
            MFnMesh meshFn(shapePath);
            if (meshFn.numColorSets() == 0)
            {
                continue;
            }
            const MString colorSet = (iShape == 0) ?
                baseColSet : bsData.m_Targets[iShape - 1].m_ColSets[iColSet];

            MItMeshVertex vIter(shapePath);
            for ( ; !vIter.isDone(); vIter.next())
            {
                const int iVtxObj = vIter.index();
                int& flag = useVtxColFlags[iVtxObj];
                if (flag == YBlendShapeData::PER_VTX_FACE) // すでに頂点フェース単位を使用
                {
                    continue;
                }
                MIntArray faces;
                vIter.getConnectedFaces(faces);
                const int faceCount = faces.length();
                if (faceCount == 1) // 1 つのフェースにのみ属する
                {
                    continue;
                }

                MColorArray colors;
                status = vIter.getColors(colors, &colorSet);
                // vIter.hasColor(iFaceObj) はカラーセット非対応
                //cerr << "colors: " << iVtxObj << R_ENDL << colors << R_ENDL;
                MColor firstCol;
                for (int iFace = 0; iFace < faceCount; ++iFace)
                {
                    //const int iFaceObj = faces[iFace];
                    MColor col(0.0f, 0.0f, 0.0f, 1.0f);
                    if (iFace < static_cast<int>(colors.length()) && colors[iFace] != Y_MCOLOR_NULL)
                    {
                        col = colors[iFace];
                    }
                    //cerr << "col: " << iVtxObj << ", " << iFaceObj << ", " << col << ", " << colorSet << ", " << (colors[iFace] != Y_MCOLOR_NULL) << ", " << status << R_ENDL;
                    if (iFace == 0)
                    {
                        firstCol = col;
                    }
                    else if (!IsEquivalentColor(col, firstCol, BLEND_SHAPE_COL_SAME_TOL))
                    {
                        flag = YBlendShapeData::PER_VTX_FACE;
                        break;
                    }
                }
            }
        }
        //cerr << "use vtx col: " << iColSet << endl << useVtxColFlags << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief blendShape ノードのアトリビュートからターゲットシェイプの頂点座標を取得します。
//!        ターゲットシェイプが削除されていた場合に呼ばれます。
//!
//! @param[in,out] target ブレンドシェイプターゲットです。
//! @param[in,out] bsData ブレンドシェイプデータです。
//! @param[in,out] rscene シーンです。
//! @param[in] blendFn ブレンドシェイプのファンクションノードです。
//! @param[in] iTarget ターゲットシェイプのインデックスです。
//! @param[in] weightIndex ウェイトインデックスです。
//! @param[in] orgPosArray ベースシェイプのローカル座標系での頂点座標配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetBlendShapeTargetFromNodeAttr(
    YBlendShapeTarget& target,
    YBlendShapeData& bsData,
    RScene& rscene,
    const MFnBlendShapeDeformer& blendFn,
    const int iTarget,
    const int weightIndex,
    const MPointArray& orgPosArray,
    const YExpOpt& yopt
)
{
    //-----------------------------------------------------------------------------
    // ウェイトプラグにエイリアス名が設定されていなければ、
    // ターゲットの名前を targetN（N はターゲットシェイプのインデックス）とします。
    if (target.m_Name.empty())
    {
        target.m_Name = RGetNumberString(iTarget, "target%d");
    }

    //-----------------------------------------------------------------------------
    // get input target index
    MObject curBaseObj = bsData.m_BasePath.node();
    int inputTargetIndex = 0; // ベースシェイプのインデックス
    MObjectArray baseObjs;
    // ブレンドシェイプのベースとなるメッシュを探し、そのインデックスを
    // inputTargetIndex に格納します。
    blendFn.getBaseObjects(baseObjs);
    for (int iBase = 0; iBase < static_cast<int>(baseObjs.length()); ++iBase)
    {
        //cerr << "base: " << MFnDependencyNode(baseObjs[iBase]).name() << R_ENDL;
        if (baseObjs[iBase] == curBaseObj)
        {
            inputTargetIndex = iBase;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // get target item index
    MIntArray targetItemList;
    blendFn.targetItemIndexList(weightIndex, curBaseObj, targetItemList);
    if (targetItemList.length() != 1)
    {
        YShowError(&rscene, // Blend shape in-between target is not supported: %s (%s) (target index = %d)
            "ブレンドシェイプのシリアルなブレンド処理は使用できません。"
            "ブレンドシェイプ作成時にインビトウィーンオプションを OFF にしてください: {0} ({1}) (ターゲットインデックス = {2})",
            "Serial blend processing cannot be used for blend shapes. "
            "Make sure that you clear the In-Between option when creating blend shapes: {0} ({1}) (target index = {2})",
            bsData.m_Name, bsData.m_BaseName, RGetNumberString(iTarget));
        return MS::kFailure;
    }
    int targetItemIndex = targetItemList[0]; // 通常 6000

    //-----------------------------------------------------------------------------
    // get plug
    MPlug itPlug = blendFn.findPlug("inputTarget");
    MPlug itgPlug = FindChildPlugByName(itPlug.elementByLogicalIndex(inputTargetIndex), "inputTargetGroup");
    MPlug itiPlug = FindChildPlugByName(itgPlug.elementByLogicalIndex(weightIndex), "inputTargetItem");
    MPlug itinPlug = itiPlug.elementByLogicalIndex(targetItemIndex);

    //-----------------------------------------------------------------------------
    // プラグから形状データを取得します。
    MObject pntObj; // kPointArrayData
    FindChildPlugByName(itinPlug, "inputPointsTarget").getValue(pntObj);
    MFnPointArrayData targetPntFn(pntObj); // 頂点座標の差分データ

    MObject compListObj; // kComponentListData
    FindChildPlugByName(itinPlug, "inputComponentsTarget").getValue(compListObj);
    MFnComponentListData compListFn(compListObj);

    // オリジナルの頂点座標をターゲットシェイプにコピします。
    for (int iVtx = 0; iVtx < bsData.m_VtxCount; ++iVtx)
    {
        target.m_Poss.append(orgPosArray[iVtx]);
    }

    // ブレンドシェイプのアトリビュートには差分値が格納されているので、
    // コピーした頂点座標にその差分を足して変形後の形状データを取得します。

    // 頂点座標の変化なしとみなす閾値です。
    const double posSameTol = BLEND_SHAPE_POS_SAME_TOL * yopt.m_InternalMagnify;

    for (int iComp = 0; iComp < static_cast<int>(compListFn.length()); ++iComp)
    {
        MObject comp = compListFn[iComp];
        if (comp.apiType() == MFn::kMeshVertComponent)
        {
            // MeshVertComponent には頂点座標へのインデックスが格納されています。
            MIntArray compVtxs;
            MFnSingleIndexedComponent(comp).getElements(compVtxs);
            const int compVtxCount = compVtxs.length();
            for (int iCv = 0; iCv < compVtxCount; ++iCv)
            {
                const int iVtx = compVtxs[iCv];
                // 元の頂点座標を取得し、アトリビュートから取得した変形後頂点への
                // 差分を足して変形後の頂点座標を求めます。
                MPoint& pos = target.m_Poss[iVtx];
                pos += targetPntFn[iCv] * yopt.m_InternalMagnify;
                if (!bsData.m_MovePosFlags[iVtx] &&
                    !pos.isEquivalent(orgPosArray[iVtx], posSameTol))
                {
                    // 元の頂点座標から変化があれば頂点座標の変化フラグを 1 にします。
                    bsData.m_MovePosFlags[iVtx] = 1;
                }
            }
        }
    }

    //cerr << "tar: " << target.m_Name << ": " << itinPlug.name() << R_ENDL << target.m_Poss << R_ENDL;
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプのベースシェイプ名と各ターゲット名が
//!        重複していないかチェックします。
//!
//! @param[in,out] rscene シーンです。
//! @param[in] bsData ブレンドシェイプデータです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus CheckBlendShapeTargetDuplicate(RScene& rscene, const YBlendShapeData& bsData)
{
    bool isDuplicate = false;
    std::string name;
    const int targetCount = static_cast<int>(bsData.m_Targets.size());
    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
    {
        name = bsData.m_Targets[iTarget].m_Name;
        if (name == bsData.m_BaseName)
        {
            isDuplicate = true;
            break;
        }
        for (int iOther = 0; iOther < iTarget; ++iOther)
        {
            if (bsData.m_Targets[iOther].m_Name == name)
            {
                isDuplicate = true;
                break;
            }
        }
    }

    if (isDuplicate)
    {
        YShowError(&rscene, // Blend shape target is duplicate: %s (%s)
            "ブレンドシェイプで、同じ名前のターゲットシェイプが重複登録されています: {0} ({1}) \n"
            "ベースシェイプとすべてのターゲットシェイプはユニークな名前になるようにブレンドシェイプエディタで設定してください。",
            "More than one target object with the same name has been registered for shape blending: {0} ({1}) \n"
            "Make sure that you configure the base shape and all target shapes with unique names in the blend shape editor.",
            bsData.m_Name, name);
        return MS::kFailure;
    }

    return MS::kSuccess;
}

#if (MAYA_API_VERSION >= 201700)
//=============================================================================
//! @brief ブレンドシェイプグループの構造体です。
//=============================================================================
struct YBlendShapeGroup
{
    MString name; //!< 名前です。
    MIntArray childIdxs; //!< 子ターゲット（グループ）のインデックス配列です（グループはインデックスに -1 を掛けた値）。
    int parentIdx; //!< 親グループのインデックスです（最上位なら -1）。
    MPlug weightPlug; //!< ウェイトプラグです。
};

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプのグループ情報を取得します。
//!
//! @param[out] pBlendShapeGroups ブレンドシェイプグループ配列を格納します。
//! @param[in] blendFn ブレンドシェイプのファンクションノードです。
//-----------------------------------------------------------------------------
static void GetBlendShapeGroups(
    std::vector<YBlendShapeGroup>* pBlendShapeGroups,
    const MFnBlendShapeDeformer& blendFn
)
{
    // 各グループのアトリビュートを取得します。
    pBlendShapeGroups->clear();
    const MPlug targetDirPlug = blendFn.findPlug("targetDirectory");
    const int groupCount = static_cast<int>(targetDirPlug.numElements());
    for (int physicalIdx = 0; physicalIdx < groupCount; ++physicalIdx)
    {
        const MPlug tdnPlug = targetDirPlug[physicalIdx];
        const size_t logicalIndex = tdnPlug.logicalIndex();
        if (logicalIndex >= pBlendShapeGroups->size())
        {
            pBlendShapeGroups->resize(logicalIndex + 1);
        }
        YBlendShapeGroup& group = (*pBlendShapeGroups)[logicalIndex];
        group.name = FindChildPlugByName(tdnPlug, "directoryName").asString();
        const MPlug cidPlug = FindChildPlugByName(tdnPlug, "childIndices");
        MFnIntArrayData(cidPlug.asMObject()).copyTo(group.childIdxs);
        group.parentIdx = -1;
        group.weightPlug = (logicalIndex == 0) ? blendFn.findPlug("envelope") :
            FindChildPlugByName(tdnPlug, "directoryWeight");
            // 最上位グループのウェイトは envelope アトリビュートです。
    }

    // 各グループの親グループのインデックスを設定します。
    for (size_t logicalIndex = 0; logicalIndex < pBlendShapeGroups->size(); ++logicalIndex)
    {
        const YBlendShapeGroup& group = (*pBlendShapeGroups)[logicalIndex];
        for (unsigned int localChildIdx = 0; localChildIdx < group.childIdxs.length(); ++localChildIdx)
        {
            const int childIdx = -group.childIdxs[localChildIdx];
                // グループの場合はインデックスに -1 を掛けた値なので、正の値に戻します。
            if (1 <= childIdx && childIdx < static_cast<int>(pBlendShapeGroups->size()))
            {
                (*pBlendShapeGroups)[childIdx].parentIdx = static_cast<int>(logicalIndex);
            }
        }
    }

    // 各グループの情報を表示します（動作確認用）。
    //for (size_t logicalIndex = 0; logicalIndex < pBlendShapeGroups->size(); ++logicalIndex)
    //{
    //  const YBlendShapeGroup& group = (*pBlendShapeGroups)[logicalIndex];
    //  cerr << "blend shape group[" << logicalIndex << "]: " << group.name << ": " << group.weightPlug.info() << ": " << group.parentIdx << ": " << group.childIdxs << endl;
    //}
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプのターゲットに親グループのウェイトプラグ配列を設定します。
//!
//! @param[in,out] pTarget ブレンドシェイプターゲットへのポインタです。
//! @param[in] weightIndex ブレンドシェイプターゲットのウェイトのインデックスです。
//! @param[in] blendShapeGroups ブレンドシェイプグループ配列です。
//-----------------------------------------------------------------------------
static void SetBlendShapeGroupWeightPlugs(
    YBlendShapeTarget* pTarget,
    const int weightIndex,
    const std::vector<YBlendShapeGroup>& blendShapeGroups
)
{
    MPlugArray& groupWeightPlugs = pTarget->m_GroupWeightPlugs;
    groupWeightPlugs.clear();
    for (size_t logicalIndex = 0; logicalIndex < blendShapeGroups.size(); ++logicalIndex)
    {
        const YBlendShapeGroup& group = blendShapeGroups[logicalIndex];
        if (YFindValueInArray(group.childIdxs, weightIndex) != -1)
        {
            groupWeightPlugs.append(group.weightPlug);
            int parentIdx = group.parentIdx;
            while (parentIdx != -1)
            {
                const YBlendShapeGroup& parent = blendShapeGroups[parentIdx];
                if (parent.weightPlug.isNull())
                {
                    break;
                }
                groupWeightPlugs.append(parent.weightPlug);
                parentIdx = parent.parentIdx;
            }
            break;
        }
    }

    //cerr << "group weight plugs: " << pTarget->m_Name << ": " << groupWeightPlugs.length() << endl;
    //for (unsigned int i = 0; i < groupWeightPlugs.length(); ++i) cerr << groupWeightPlugs[i].info() << endl;
}
#endif

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプデータを取得します。
//-----------------------------------------------------------------------------
MStatus GetBlendShapeData(
    YBlendShapeData& bsData,
    bool& isValid,
    RScene& rscene,
    const std::string& baseName,
    const MDagPath& baseShapePath,
    const MObject& blendShapeObj,
    const MPointArray& orgPosArray,
    const MVector& pivot,
    const MString* baseColSets,
    const YExpOpt& yopt
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 有効フラグを false で初期化します。
    isValid = false;

    //-----------------------------------------------------------------------------
    // blendShape ノードの名前を取得します。
    MFnBlendShapeDeformer blendFn(blendShapeObj);
    bsData.m_Name = blendFn.name().asChar();
    bsData.m_BlendShapeObj = blendShapeObj;

    //-----------------------------------------------------------------------------
    // シェイプアニメーションの頂点属性更新情報を取得します。
    bsData.m_ShapeUpdate = GetBlendShapeUpdateFlag(blendShapeObj);
    if (!bsData.m_ShapeUpdate.UpdatesSome())
    {
        // どの頂点属性も更新されない場合はブレンドシェイプデータを取得しません。
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // ベースシェイプの名前、DAG パス、カラーセット名配列を設定します。
    bsData.m_BaseName = baseName;
    bsData.m_BasePath = baseShapePath;

    MDagPath baseXformPath = baseShapePath;
    baseXformPath.pop();
    const std::string baseOrgName = MFnDagNode(baseXformPath).name().asChar();

    for (int iColSet = 0; iColSet < RPrimVtx::VTX_COL_MAX; ++iColSet)
    {
        bsData.m_BaseColSets[iColSet] = baseColSets[iColSet];
    }

    //-----------------------------------------------------------------------------
    // デフォーマにより変形されるメンバー（頂点とフェース）を取得します。
    if (GetDeformerMemberVertexFlags(bsData.m_IsMemberVtxs,
        baseShapePath, blendShapeObj) == 0)
    {
        YShowWarning(&rscene, // Blend shape has no member: %s (%s)
            "ブレンドシェイプのデフォーマセットのメンバが 1 つもありません。シェイプアニメーションは出力されません: {0} ({1})",
            "There are no members in the deformer set for the blend shape. The shape animation is not output: {0} ({1})",
            bsData.m_Name, baseOrgName);
        return MS::kSuccess;
    }
    GetBlendShapeMemberFaceFlags(bsData);
    //crr << "memberVtx : " << bsData.m_IsMemberVtxs << R_ENDL;
    //crr << "memberFace: " << bsData.m_IsMemberFaces << R_ENDL;

    //-----------------------------------------------------------------------------
    // ベースシェイプのトポロジを取得します。
    MFnMesh baseMeshFn(baseShapePath);
    bsData.m_VtxCount  = orgPosArray.length();
    bsData.m_FaceCount = baseMeshFn.numPolygons();
    MIntArray polyVtxCounts; // 各フェースごとの頂点数です。
    MItMeshPolygon pIter(baseShapePath);
    for ( ; !pIter.isDone(); pIter.next())
    {
        polyVtxCounts.append(pIter.polygonVertexCount());
    }

    //-----------------------------------------------------------------------------
    // ターゲット群を取得します。
    const int allTargetCount = blendFn.numWeights(); // ウェイトの要素数がターゲットの数となります。
    if (allTargetCount == 0)
    {
        YShowWarning(&rscene, // Blend shape has no target: %s (%s)
            "ブレンドシェイプのターゲットシェイプが設定されていません。シェイプアニメーションは出力されません: {0} ({1})",
            "A target shape for the blend shape has not been set. The shape animation is not output: {0} ({1})",
            bsData.m_Name, baseOrgName);
        return MS::kSuccess;
    }

    // 頂点座標の変化なしとみなす閾値です。
    const double posSameTol = BLEND_SHAPE_POS_SAME_TOL * yopt.m_InternalMagnify;
    bsData.m_MovePosFlags = MIntArray(bsData.m_VtxCount, 0);

    bool isTargetDeleted = false;

    MPlug weightPlug = blendFn.findPlug("weight");
    MIntArray weightIndexes; // 各ターゲットのウェイトインデックス配列です。
    blendFn.weightIndexList(weightIndexes);
    #if (MAYA_API_VERSION >= 201700)
    const MPlug targetVisPlug = blendFn.findPlug("targetVisibility");
    const MPlug targetParentVisPlug = blendFn.findPlug("targetParentVisibility");
    std::vector<YBlendShapeGroup> blendShapeGroups;
    GetBlendShapeGroups(&blendShapeGroups, blendFn);
    #endif

    for (int iTarget = 0; iTarget < allTargetCount; ++iTarget)
    {
        const int weightIndex = weightIndexes[iTarget];

        //-----------------------------------------------------------------------------
        // ウェイトプラグからターゲット名を取得します。
        YBlendShapeTarget target;
        target.m_AnimPlug = weightPlug.elementByLogicalIndex(weightIndex, &status);
        target.m_Name = target.m_AnimPlug.partialName(false, false, false, true).asChar();
            // 第 4 引数 useAlias を true にすることで
            // ブレンドシェイプエディタで入力したエイリアス名を取得できます。
            // エイリアス名が設定されていなければ "w[0]" "w[1]" といったプラグ名が返るので、
            // ターゲット名を一度空文字（未取得状態）にします。
        target.m_Name = (target.m_Name.find('[') == std::string::npos) ?
            GetOutElementName(target.m_Name, yopt.m_RemoveNamespace) : std::string();

        //-----------------------------------------------------------------------------
        // ターゲットの親グループのウェイトプラグ配列を設定し、可視性を取得します。
        #if (MAYA_API_VERSION >= 201700)
        SetBlendShapeGroupWeightPlugs(&target, weightIndex, blendShapeGroups);
        target.m_Visibility = targetVisPlug.elementByLogicalIndex(weightIndex).asBool() &&
            targetParentVisPlug.elementByLogicalIndex(weightIndex).asBool();
        #else
        target.m_Visibility = true;
        #endif

        //-----------------------------------------------------------------------------
        // ターゲットシェイプのシェイプノードを取得します。
        MObjectArray targetObjs;
        status = blendFn.getTargets(baseShapePath.node(), weightIndex, targetObjs);
        //cerr << "tar: " << target.m_AnimPlug.info() << ", " << targetObjs.length() << R_ENDL;
        // 一つのウェイトに二つ以上のオブジェクトが関連付けられているデータはサポートしない
        if (targetObjs.length() >= 2)
        {
            YShowError(&rscene, // Blend shape in-between target is not supported: %s (%s) (target index = %d)
                "ブレンドシェイプのシリアルなブレンド処理は使用できません。"
                "ブレンドシェイプ作成時にインビトウィーンオプションを OFF にしてください: {0} ({1}) (ターゲットインデックス = {2})",
                "Serial blend processing cannot be used for blend shapes. "
                "Make sure that you clear the In-Between option when creating blend shapes: {0} ({1}) (target index = {2})",
                bsData.m_Name, baseOrgName, RGetNumberString(iTarget));
            return MS::kFailure;
        }
        #ifdef BS_ALLOW_DEL_TARGET_SW
        // ウェイトに関連付けられているオブジェクトがない場合は、ブレンドシェイプの
        // アトリビュートに設定されている頂点座標の差分データからターゲットシェイプを
        // 取得し、このターゲットの処理を終了させます。
        if (targetObjs.length() == 0 || isTargetDeleted)
        {
            // ブレンドシェイプのアトリビュートには頂点座標しか格納されていないので、
            // その場合は頂点属性の更新フラグを変更する必要があります。そのためにターゲットが
            // 削除されていることを記録します。
            isTargetDeleted = true;

            status = GetBlendShapeTargetFromNodeAttr(target,
                bsData, rscene, blendFn, iTarget, weightIndex, orgPosArray, yopt);
            CheckStatus(status);
            bsData.m_Targets.push_back(target);
            continue;
        }
        #else
        // ウェイトに関連付けられているオブジェクトが存在しない場合はエラーとします。
        if (targetObjs.length() == 0)
        {
            YShowError(&rscene, // Blend shape target is not found: %s (%s) (target index = %d) // 現在は発生しない
                "ブレンドシェイプのターゲットが見つかりません: {0} ({1}) (ターゲットインデックス = {2})",
                "Blend shape target cannot be found: {0} ({1}) (target index = {2})",
                bsData.m_Name, baseOrgName, RGetNumberString(iTarget));
            return MS::kFailure;
        }
        #endif

        //-----------------------------------------------------------------------------
        // ウェイトプラグにエイリアス名が設定されていなければ、
        // ターゲットシェイプの親の transform ノード名をターゲット名とします。
        MFnDagNode(targetObjs[0]).getPath(target.m_ShapePath);
        MDagPath targetXformPath = target.m_ShapePath;
        targetXformPath.pop();
        const std::string targetOrgName = MFnDagNode(targetXformPath).name().asChar();

        if (target.m_Name.empty())
        {
            target.m_Name = GetOutElementName(targetOrgName, yopt.m_RemoveNamespace);
        }
        //cerr << "target" << iTarget << ": " << target.m_Name << ": " << targetOrgName << endl;

        //-----------------------------------------------------------------------------
        // ターゲットのトポロジの正当性をチェックします。
        bool isValidTopology = true;
        MFnMesh targetMeshFn(target.m_ShapePath);
        if (targetMeshFn.numVertices() != bsData.m_VtxCount ||
            targetMeshFn.numPolygons() != bsData.m_FaceCount)
        {
            // ベースシェイプとターゲットシェイプの頂点数またはフェース数が異なる場合は
            // 不正とみなします。
            isValidTopology = false;
        }
        else
        {
            // 各フェースの頂点数がベースシェイプとターゲットシェイプで一致しているか
            // 確認します。
            int iFace = 0;
            MItMeshPolygon targetPolyIter(target.m_ShapePath);
            for ( ; !targetPolyIter.isDone(); targetPolyIter.next())
            {
                if (static_cast<int>(targetPolyIter.polygonVertexCount()) != polyVtxCounts[iFace])
                {
                    isValidTopology = false;
                    //cerr << "topology: " << targetOrgName.c_str() << ".f[" << iFace << "]" << R_ENDL;
                    break;
                }
                ++iFace;
            }
        }
        if (!isValidTopology)
        {
            YShowError(&rscene, // Blend shape target topology is different from original mesh: %s (%s)
                "ブレンドシェイプのターゲットシェイプのトポロジ（頂点数 / フェース数 / 各フェースの頂点数）が、ベースシェイプと異なっています: {0} ({1})",
                "The target shape topology (numbers of vertices, surfaces, vertices per surface) of a blend shape has different from the base shape: {0} ({1})",
                bsData.m_Name, targetOrgName);
            return MS::kFailure;
        }

        //-----------------------------------------------------------------------------
        // ターゲットシェイプの頂点座標を取得します。
        MPointArray posArray;
        targetMeshFn.getPoints(posArray, MSpace::kObject);
        for (int iVtx = 0; iVtx < bsData.m_VtxCount; ++iVtx)
        {
            MPoint pos;
            // ブレンドシェイプにより更新される頂点ならターゲットシェイプから取得した
            // 座標を設定し、そうでなければベースシェイプの頂点座標を設定します。
            if (bsData.m_IsMemberVtxs[iVtx])
            {
                // ターゲットのメッシュから取得した頂点座標は InternalMagnify をかけて
                // ピボット位置を引いて中間ファイルに出力する座標系に変換します。
                pos = posArray[iVtx] * yopt.m_InternalMagnify - pivot;
                if (!bsData.m_MovePosFlags[iVtx] &&
                    !pos.isEquivalent(orgPosArray[iVtx], posSameTol))
                {
                    // 頂点座標がベースのメッシュの頂点座標と異なっていたら
                    // 頂点座標の変化フラグを 1 にします。
                    bsData.m_MovePosFlags[iVtx] = 1;
                }
            }
            else
            {
                pos = orgPosArray[iVtx];
            }
            target.m_Poss.append(pos);
        }

        //-----------------------------------------------------------------------------
        // ターゲットシェイプのカラーセット名配列を取得します。
        status = GetMeshVertexColorAttrs(target.m_ColSets, rscene, target.m_ShapePath);
        CheckStatus(status);

        //-----------------------------------------------------------------------------
        // ターゲットをブレンドシェイプデータに追加します。
        bsData.m_Targets.push_back(target);
    }
    //cerr << "movePos: " << bsData.m_MovePosFlags << R_ENDL;

    //-----------------------------------------------------------------------------
    // ベースシェイプ名と各ターゲット名が重複していないかチェックします。
    status = CheckBlendShapeTargetDuplicate(rscene, bsData);
    CheckStatus(status);

    //-----------------------------------------------------------------------------
    // ターゲットシェイプが削除されている場合は頂点座標しか更新できないので、
    // 法線と頂点カラーの更新フラグを false にします。
    if (isTargetDeleted)
    {
        if (bsData.m_ShapeUpdate.m_UpdatesNrm ||
            bsData.m_ShapeUpdate.UpdatesSomeCol())
        {
            YShowWarning(&rscene, // Blend shape normal and color cannot be animated when target is deleted: %s (%s)
                "ブレンドシェイプのターゲットのオブジェクトが存在しないため、法線と頂点カラーはアニメーションできません: {0} ({1})",
                "Normal and vertex color animation is impossible because the target object for the blend shape does not exist: {0} ({1})",
                bsData.m_Name, baseOrgName);
            bsData.m_ShapeUpdate.m_UpdatesNrm = false;
            bsData.m_ShapeUpdate.SetAllCols(false);
        }
    }

    //-----------------------------------------------------------------------------
    // ベースシェイプのカラーセットに対応するカラーセットが
    // 全ターゲットシェイプに存在しなければ
    // ベースシェイプのカラーセット名を空文字にし、
    // 頂点カラーの更新フラグを false にします。
    if (bsData.m_ShapeUpdate.UpdatesSomeCol())
    {
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iColSet = 0; iColSet < RPrimVtx::VTX_COL_MAX; ++iColSet)
        {
            if (bsData.m_BaseColSets[iColSet].length() != 0)
            {
                for (int iTarget = 0; iTarget < targetCount; ++iTarget)
                {
                    if (bsData.m_Targets[iTarget].m_ColSets[iColSet].length() == 0)
                    {
                        bsData.m_BaseColSets[iColSet].clear();
                        break;
                    }
                }
            }
            bsData.m_ShapeUpdate.m_UpdatesCols[iColSet] =
                (bsData.m_BaseColSets[iColSet].length() != 0);
        }
    }

    //-----------------------------------------------------------------------------
    // すべてのターゲットシェイプの頂点座標がベースシェイプと同じなら
    // 頂点座標の更新フラグを false にします。
    if (YFindValueInArray(bsData.m_MovePosFlags, 1) == -1)
    {
        bsData.m_ShapeUpdate.m_UpdatesPos = false;
    }

    //-----------------------------------------------------------------------------
    // 頂点単位の法線の使用フラグ配列を設定します。
    if (bsData.m_ShapeUpdate.m_UpdatesNrm)
    {
        GetBlendShapeUseVtxNrmFlags(bsData);
    }
    if (bsData.m_ShapeUpdate.UpdatesSomeCol())
    {
        GetBlendShapeUseVtxColFlags(bsData);
    }

    //-----------------------------------------------------------------------------
    // 有効フラグを設定します。
    isValid = bsData.m_ShapeUpdate.UpdatesSome();

    return MS::kSuccess;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の頂点座標がブレンドシェイプで変化するなら
//!        true を返します。
//!
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//!
//! @return 頂点座標がブレンドシェイプで変化するなら true を返します。
//-----------------------------------------------------------------------------
static bool CheckBlendShapePositionMove(const YBlendShapeData& bsData, MObject& comp)
{
    MItMeshPolygon pIter(bsData.m_BasePath, comp);
    for ( ; !pIter.isDone(); pIter.next())
    {
        const int vtxCount = pIter.polygonVertexCount();
        for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
        {
            if (bsData.m_MovePosFlags[pIter.vertexIndex(iVtx)])
            {
                return true;
            }
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の法線がブレンドシェイプで変化するなら
//!        true を返します。
//!
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//!
//! @return 法線がブレンドシェイプで変化するなら true を返します。
//-----------------------------------------------------------------------------
static bool CheckBlendShapeNormalMove(const YBlendShapeData& bsData, MObject& comp)
{
    //-----------------------------------------------------------------------------
    // ベースシェイプの法線を取得します。
    MVectorArray baseNrms;
    MItMeshPolygon pIter(bsData.m_BasePath, comp);
    for ( ; !pIter.isDone(); pIter.next())
    {
        if (bsData.m_IsMemberFaces[pIter.index()])
        {
            const int vtxCount = pIter.polygonVertexCount();
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
            {
                MVector nrm;
                pIter.getNormal(iVtx, nrm, MSpace::kObject);
                baseNrms.append(nrm);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 各ターゲットシェイプの法線をベースシェイプの法線と比較します。
    const int targetCount = static_cast<int>(bsData.m_Targets.size());
    for (int iTarget = 0; iTarget < targetCount; ++iTarget)
    {
        const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
        int iVtxTotal = 0;
        MItMeshPolygon qIter(target.m_ShapePath, comp);
        for ( ; !qIter.isDone(); qIter.next())
        {
            if (bsData.m_IsMemberFaces[qIter.index()])
            {
                const int vtxCount = qIter.polygonVertexCount();
                for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal)
                {
                    MVector nrm;
                    qIter.getNormal(iVtx, nrm, MSpace::kObject);
                    if (!nrm.isEquivalent(baseNrms[iVtxTotal], BLEND_SHAPE_NRM_SAME_TOL))
                    {
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の接線がブレンドシェイプで変化するなら
//!        true を返します。
//!
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] tanUvSetNamess 接線を取得する UV セット名の配列の配列です。
//!                           UV セット名の配列はベースシェイプが先頭で、ベースシェイプ以外の
//!                           キーシェイプが存在すればその後に格納します。
//!                           接線を出力しない場合、UV セット名の配列は空です。
//!
//! @return 接線がブレンドシェイプで変化するなら true を返します。
//-----------------------------------------------------------------------------
static bool CheckBlendShapeTangentMove(
    const YBlendShapeData& bsData,
    MObject& comp,
    const MStringArray* tanUvSetNamess
)
{
    const int targetCount = static_cast<int>(bsData.m_Targets.size());
    for (int iTanSet = 0; iTanSet < RPrimVtx::VTX_TAN_MAX; ++iTanSet)
    {
        //-----------------------------------------------------------------------------
        // UV セット名の配列の長さをチェックします。
        const MStringArray& tanUvSetNames = tanUvSetNamess[iTanSet];
        if (static_cast<int>(tanUvSetNames.length()) != 1 + targetCount)
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // ベースシェイプの接線を取得します。
        MFnMesh baseMeshFn(bsData.m_BasePath);
        const MString baseUvSetName = tanUvSetNames[0]; // C4238 回避のため一度コピー
        MFloatVectorArray baseTans;
        baseMeshFn.getTangents(baseTans, MSpace::kObject, &baseUvSetName);

        //-----------------------------------------------------------------------------
        // 各ターゲットシェイプの接線をベースシェイプの接線と比較します。
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            const MString targetUvSetName = tanUvSetNames[1 + iTarget]; // C4238 回避のため一度コピー
            MFloatVectorArray targetTans;
            MFnMesh(target.m_ShapePath).getTangents(targetTans, MSpace::kObject, &targetUvSetName);
            MItMeshPolygon qIter(target.m_ShapePath, comp);
            for ( ; !qIter.isDone(); qIter.next())
            {
                if (bsData.m_IsMemberFaces[qIter.index()])
                {
                    const int vtxCount = qIter.polygonVertexCount();
                    for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                    {
                        if (!targetTans[qIter.tangentIndex(iVtx)].isEquivalent(
                            baseTans[baseMeshFn.getTangentId(qIter.index(), qIter.vertexIndex(iVtx))],
                            BLEND_SHAPE_TAN_SAME_TOL))
                        {
                            return true;
                        }
                    }
                }
            }
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の従法線がブレンドシェイプで変化するなら
//!        true を返します。
//!
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] tanUvSetNamess 接線を取得する UV セット名の配列の配列です。
//!                           UV セット名の配列はベースシェイプが先頭で、ベースシェイプ以外の
//!                           キーシェイプが存在すればその後に格納します。
//!                           接線を出力しない場合、UV セット名の配列は空です。
//!
//! @return 従法線がブレンドシェイプで変化するなら true を返します。
//-----------------------------------------------------------------------------
static bool CheckBlendShapeBinormalMove(
    const YBlendShapeData& bsData,
    MObject& comp,
    const MStringArray* tanUvSetNamess
)
{
    const int targetCount = static_cast<int>(bsData.m_Targets.size());
    for (int iTanSet = 0; iTanSet < RPrimVtx::VTX_TAN_MAX; ++iTanSet)
    {
        //-----------------------------------------------------------------------------
        // UV セット名の配列の長さをチェックします。
        const MStringArray& tanUvSetNames = tanUvSetNamess[iTanSet];
        if (static_cast<int>(tanUvSetNames.length()) != 1 + targetCount)
        {
            return false;
        }

        //-----------------------------------------------------------------------------
        // ベースシェイプの従法線を取得します。
        MFnMesh baseMeshFn(bsData.m_BasePath);
        const MString baseUvSetName = tanUvSetNames[0]; // C4238 回避のため一度コピー
        MFloatVectorArray baseBins;
        baseMeshFn.getBinormals(baseBins, MSpace::kObject, &baseUvSetName);

        //-----------------------------------------------------------------------------
        // 各ターゲットシェイプの従法線をベースシェイプの従法線と比較します。
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            const MString targetUvSetName = tanUvSetNames[1 + iTarget]; // C4238 回避のため一度コピー
            MFloatVectorArray targetBins;
            MFnMesh(target.m_ShapePath).getBinormals(targetBins, MSpace::kObject, &targetUvSetName);
            MItMeshPolygon qIter(target.m_ShapePath, comp);
            for ( ; !qIter.isDone(); qIter.next())
            {
                if (bsData.m_IsMemberFaces[qIter.index()])
                {
                    const int vtxCount = qIter.polygonVertexCount();
                    for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                    {
                        if (!targetBins[qIter.tangentIndex(iVtx)].isEquivalent(
                            baseBins[baseMeshFn.getTangentId(qIter.index(), qIter.vertexIndex(iVtx))],
                            BLEND_SHAPE_TAN_SAME_TOL))
                        {
                            return true;
                        }
                    }
                }
            }
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の頂点カラーがブレンドシェイプで変化するなら
//!        頂点カラー更新フラグに true、変化しないなら false を設定します。
//!
//! @param[in,out] shapeUpdate シェイプアニメーションの頂点属性更新情報です。
//! @param[in] bsData ブレンドシェイプデータです。
//! @param[in] comp フェース群のコンポーネントです。
//! @param[in] forcesVtxColor true なら頂点カラーが設定されていない場合に
//!                           (1, 1, 1, 1) が設定されているとして処理します。
//!                           false なら (0, 0, 0, 1) が設定されているとして処理します。
//-----------------------------------------------------------------------------
static void CheckBlendShapeColorMove(
    RShapeUpdate& shapeUpdate,
    const YBlendShapeData& bsData,
    MObject& comp,
    const bool forcesVtxColor
)
{
    //-----------------------------------------------------------------------------
    // 頂点カラーが設定されていない場合のカラーを決定します。
    const MColor defaultMC = (forcesVtxColor) ?
        MColor(1.0f, 1.0f, 1.0f, 1.0f) : // white
        MColor(0.0f, 0.0f, 0.0f, 1.0f);  // black

    for (int iColSet = 0; iColSet < RPrimVtx::VTX_COL_MAX; ++iColSet)
    {
        //-----------------------------------------------------------------------------
        // すでに更新フラグが false の場合と
        // ベースシェイプのカラーセット名が空文字の場合は処理しません。
        bool& updatesCol = shapeUpdate.m_UpdatesCols[iColSet];
        const MString& baseColSet = bsData.m_BaseColSets[iColSet];
        if (!updatesCol || baseColSet.length() == 0)
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // ベースシェイプの頂点カラーを取得します。
        MColorArray baseCols;
        MItMeshPolygon pIter(bsData.m_BasePath, comp);
        for ( ; !pIter.isDone(); pIter.next())
        {
            if (bsData.m_IsMemberFaces[pIter.index()])
            {
                int colorCount = 0;
                pIter.numColors(colorCount, &baseColSet);

                MColorArray vtxColors;
                if (colorCount > 0)
                {
                    pIter.getColors(vtxColors, &baseColSet);
                }

                const int vtxCount = pIter.polygonVertexCount();
                for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                {
                    const MColor mc = (colorCount > 0 && vtxColors[iVtx] != Y_MCOLOR_NULL) ?
                        vtxColors[iVtx] : defaultMC;
                    baseCols.append(mc);
                }
            }
        }

        //-----------------------------------------------------------------------------
        // 各ターゲットシェイプの頂点カラーをベースシェイプの頂点カラーと比較します。
        updatesCol = false; // 一度 false にします。
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            const MString& targetColSet = target.m_ColSets[iColSet];
            int iVtxTotal = 0;
            MItMeshPolygon qIter(target.m_ShapePath, comp);
            for ( ; !qIter.isDone(); qIter.next())
            {
                if (bsData.m_IsMemberFaces[qIter.index()])
                {
                    int colorCount = 0;
                    qIter.numColors(colorCount, &targetColSet);

                    MColorArray vtxColors;
                    if (colorCount > 0)
                    {
                        qIter.getColors(vtxColors, &targetColSet);
                    }

                    int vtxCount = qIter.polygonVertexCount();
                    for (int iVtx = 0; iVtx < vtxCount; ++iVtx, ++iVtxTotal)
                    {
                        const MColor mc = (colorCount > 0 && vtxColors[iVtx] != Y_MCOLOR_NULL) ?
                            vtxColors[iVtx] : defaultMC;
                        if (!IsEquivalentColor(mc, baseCols[iVtxTotal], BLEND_SHAPE_COL_SAME_TOL))
                        {
                            updatesCol = true;
                            break;
                        }
                    }
                }
                if (updatesCol)
                {
                    break;
                }
            }
            if (updatesCol)
            {
                break;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定したフェース群の頂点属性がブレンドシェイプで変化するか判定して
//!        シェイプアニメーションの頂点属性更新情報を設定します。
//-----------------------------------------------------------------------------
void SetBlendShapeUpdateForShape(
    RShapeUpdate& shapeUpdate,
    const YBlendShapeData& bsData,
    MObject& comp,
    const MStringArray* tanUvSetNamess,
    const bool forcesVtxColor
)
{
    //-----------------------------------------------------------------------------
    // 頂点座標が変化するか判定します。
    if (shapeUpdate.m_UpdatesPos)
    {
        shapeUpdate.m_UpdatesPos = CheckBlendShapePositionMove(bsData, comp);
    }

    //-----------------------------------------------------------------------------
    // 法線が変化するか判定します。
    if (shapeUpdate.m_UpdatesNrm)
    {
        shapeUpdate.m_UpdatesNrm = CheckBlendShapeNormalMove(bsData, comp);
    }

    //-----------------------------------------------------------------------------
    // 接線が変化するか判定します。
    if (shapeUpdate.m_UpdatesTan)
    {
        shapeUpdate.m_UpdatesTan = CheckBlendShapeTangentMove(bsData, comp, tanUvSetNamess);
    }

    //-----------------------------------------------------------------------------
    // 従法線が変化するか判定します。
    if (shapeUpdate.m_UpdatesBin)
    {
        shapeUpdate.m_UpdatesBin = CheckBlendShapeBinormalMove(bsData, comp, tanUvSetNamess);
    }

    //-----------------------------------------------------------------------------
    // 頂点カラーが変化するか判定します。
    if (shapeUpdate.UpdatesSomeCol())
    {
        CheckBlendShapeColorMove(shapeUpdate, bsData, comp, forcesVtxColor);
    }
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプターゲットの現在のウェイト値を取得します。
//-----------------------------------------------------------------------------
float YBlendShapeTarget::GetWeight() const
{
    float weight = 0.0f;
    if (m_Visibility)
    {
        m_AnimPlug.getValue(weight);
        for (unsigned int parentIdx = 0; parentIdx < m_GroupWeightPlugs.length(); ++parentIdx)
        {
            weight *= m_GroupWeightPlugs[parentIdx].asFloat();
        }
    }
    return weight;
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるブレンドシェイプアニメーション値を取得します。
//-----------------------------------------------------------------------------
void GetBlendShapeFullAnimValue(YBlendShapeDataArray& bsDatas)
{
    const int dataCount = static_cast<int>(bsDatas.size());
    for (int iData = 0; iData < dataCount; ++iData)
    {
        YBlendShapeData& bsData = bsDatas[iData];
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            const float weight = target.GetWeight();
            target.m_Anim.m_FullValues.push_back(RSnapToZero(weight));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるブレンドシェイプアニメーション値を分析します。
//-----------------------------------------------------------------------------
void AnalyzeBlendShapeFullAnim(YBlendShapeDataArray& bsDatas, const YExpOpt& yopt)
{
    const int dataCount = static_cast<int>(bsDatas.size());
    for (int iData = 0; iData < dataCount; ++iData)
    {
        YBlendShapeData& bsData = bsDatas[iData];
        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            YAnimCurve& curve = target.m_Anim;
            curve.m_Name      = bsData.m_Name + "." + target.m_Name;
            curve.m_LoopFlag  = yopt.m_LoopAnim;
            curve.m_AngleFlag = false;
            curve.m_Tolerance = R_MAKE_KEY_TOL_SHAPE_ANIM;
            curve.UpdateConstantFlag();
            //cerr << "bs: " << curve.m_Name << ": " << curve.m_ConstantFlag << R_ENDL;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ブレンドシェイプのキーアニメーションを取得します。
//-----------------------------------------------------------------------------
MStatus GetBlendShapeKeyAnim(YBlendShapeDataArray& bsDatas, const YExpOpt& yopt)
{
    const int dataCount = static_cast<int>(bsDatas.size());
    for (int iData = 0; iData < dataCount; ++iData)
    {
        YBlendShapeData& bsData = bsDatas[iData];
        bsData.m_OutTargetAnimCount = 0;

        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            //-----------------------------------------------------------------------------
            // get channel input
            YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            YAnimCurve& curve = target.m_Anim;
            ChannelInput chan(target.m_AnimPlug);

            // アニメーションが設定されているか、値が 0 でないウェイト値のみカーブを出力します。
            curve.m_UseFlag =
                !curve.m_ConstantFlag                       ||
                (target.m_Visibility && chan.IsEffective()) ||
                curve.m_FullValues[0] != 0.0f;
            if (curve.m_UseFlag)
            {
                ++bsData.m_OutTargetAnimCount;
            }

            if (curve.m_ConstantFlag)
            {
                continue;
            }

            //-----------------------------------------------------------------------------
            // check get key OK
            bool forceMakeKey = yopt.m_BakeAllAnim;
            float valueScale = 1.0f;
            float valueOfs = 0.0f;
            bool getKeyFlag = (
                !forceMakeKey &&
                chan.m_CurveFlag &&
                !chan.m_CurveWeighted &&
                IsValidAnimCurveValue(chan.m_CurveObj,
                    curve.m_FullValues, valueScale, valueOfs, yopt));

            if (getKeyFlag)
            {
                //-----------------------------------------------------------------------------
                // get keys (blend shape)
                curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
            }
            else
            {
                //-----------------------------------------------------------------------------
                // make keys (blend shape)
                //cerr << "make keys: " << curve.m_Name << R_ENDL;
                curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, false);
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief fsh ファイル（シェイプアニメーション）を出力します。
//-----------------------------------------------------------------------------
MStatus OutputFshFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YVertexShapeAnimArray& vertexShapeAnims,
    const YBlendShapeDataArray& bsDatas
)
{
    //-----------------------------------------------------------------------------
    // check size
    const int vertexShapeAnimCount = static_cast<int>(vertexShapeAnims.size());
    if (vertexShapeAnimCount == 0)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // begin vertex shape anim array
    const int tc = 0;
    os << RTab(tc) << "<vertex_shape_anim_array length=\"" << vertexShapeAnimCount << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for vertex shape anim
    for (int iAnim = 0; iAnim < vertexShapeAnimCount; ++iAnim)
    {
        const YVertexShapeAnim& vertexShapeAnim = vertexShapeAnims[iAnim];
        const YBlendShapeData& bsData =
            bsDatas[vertexShapeAnim.m_BlendShapeDataIndex];

        os << RTab(tc + 1) << "<vertex_shape_anim index=\"" << iAnim
           << "\" shape_name=\"" << RGetUtf8FromShiftJis(vertexShapeAnim.m_ShapeName)
           << "\" base_name=\"" << RGetUtf8FromShiftJis(bsData.m_BaseName)
           << "\">" << R_ENDL;

        const int targetCount = static_cast<int>(bsData.m_Targets.size());
        for (int iTarget = 0; iTarget < targetCount; ++iTarget)
        {
            const YBlendShapeTarget& target = bsData.m_Targets[iTarget];
            if (target.m_Anim.m_UseFlag)
            {
                const YAnimCurve& curve = target.m_Anim;
                os << RTab(tc + 2) << "<shape_anim_target"
                   << " key_shape_name=\"" << RGetUtf8FromShiftJis(target.m_Name)
                   << "\" base_value=\"" << curve.GetBaseValue() << "\"";
                if (curve.m_ConstantFlag)
                {
                    os << " />" << R_ENDL;
                }
                else
                {
                    os << ">" << R_ENDL;

                    curve.Out(os, dataStreams, tc + 3, false);

                    os << RTab(tc + 2) << "</shape_anim_target>" << R_ENDL;
                }
            }
        }

        os << RTab(tc + 1) << "</vertex_shape_anim>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end vertex shape anim array
    os << RTab(tc) << "</vertex_shape_anim_array>" << R_ENDL;

    return MS::kSuccess;
}

