﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <maya/MStatus.h>
#include <maya/MPxCommand.h>
#include <maya/MArgList.h>
#include <maya/MSyntax.h>
#include <maya/MPlug.h>
#include <maya/MSelectionList.h>
#include <maya/MStringArray.h>
#include <maya/MObjectArray.h>
#include <maya/MDagPath.h>

class MFnPlugin;

namespace lod
{
    class Model;
    struct Shape;
}

//=============================================================================
//! @brief CheckLod コマンドのクラスです。
//!
//! Usage:
//!    NintendoCheckLodCmd base_root lod_root option_string
//! Result:
//!    0: error
//!    1: no error
//!    2: warning
//=============================================================================
class NintendoCheckLodCmd : public MPxCommand
{
    lod::Model* m_pBaseModel; //!< ベースモデルです。
    lod::Model* m_pLodModel; //!< LOD モデルです。
    bool m_treatQuantizeTypeMismatchAsError; //!< 量子化タイプの相違をエラーとみなすなら true です。
    int m_WarningCount; //!< 警告数です。
    MSelectionList m_errorSelection; //!< エラー原因の選択リストです。

public:
    //! コンストラクタです。
    NintendoCheckLodCmd()
    : m_treatQuantizeTypeMismatchAsError(false),
      m_WarningCount(0)
    {
    }

    //! デストラクタです。
    virtual ~NintendoCheckLodCmd()
    {
    }

    bool isUndoable() const{ return false; }

    MStatus doIt(const MArgList& args);

    static void* creator(){ return new NintendoCheckLodCmd(); }

    static MStatus Register(MFnPlugin& plugin);

    static MStatus Deregister(MFnPlugin& plugin);

private:
    bool ParseOptions(const MString& options);

    bool CheckBones();

    bool CheckMaterials();

    //! @brief シェイプ情報群の一致をチェックします。
    //!
    //! @return ベースモデルと LOD モデルのシェイプ情報がすべて一致するなら true を返します。
    //!
    bool CheckShapes();

    //! @brief 2 つのシェイプ情報の一致をチェックします。
    //!
    //! @param[in] lodModelName LOD モデル名です。
    //! @param[in] pBaseShape ベースモデルのシェイプ情報へのポインタです。
    //! @param[in] pLodShape LOD モデルのシェイプ情報へのポインタです。
    //! @param[in] shapeIndex シェイプインデックスです。
    //! @param[in] maxVertexSkinningCount スキニングのウェイト値を分配可能な最大ノード数です。
    //!
    //! @return 2 つのシェイプ情報が一致するなら true を返します。
    //!
    bool CheckShape(
        const MString& lodModelName,
        const lod::Shape* pBaseShape,
        const lod::Shape* pLodShape,
        const int shapeIndex,
        const int maxVertexSkinningCount
    );
};

