﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RShapeCT AppendPolygon AppendLines AppendPoints TriangulateRPrim
// SetAndOptimize makePrimSets OptimizePrimSets
// MergeSingleRPrim
// OutRShape OutVtxAttrData OutVtxAttrFromVec4s OutVtxColorData

// RShape RPrimSet RPrimitive RPrimVtx
// RVtxBoneRef RBoneAndWeight

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityShape.h"
#include "DccUtilityLogger.h"


#define STRING2(x) #x
#define STRING(x) STRING2(x)
#define TODO(x) __FILE__ "(" STRING(__LINE__) "): TODO: "x

using namespace std;
namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

FOutShapeInfo* FOutShapeInfo::duplicate(void) const
{
    FOutShapeInfo* dup = new FOutShapeInfo;

    dup->m_BoneName = m_BoneName;
    dup->m_Index = m_Index;
    dup->m_KeyNames = m_KeyNames;
    dup->m_MatName = m_MatName;
    dup->m_Name = m_Name;
    dup->m_OrientedBB = m_OrientedBB;

    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        // 要素数と量子化フラグをコピー
        dup->m_VtxInfos[a].m_CompCount = m_VtxInfos[a].m_CompCount;
        dup->m_VtxInfos[a].m_ValueType = m_VtxInfos[a].m_ValueType;

        // 頂点要素のデータ配列を複製
        if(m_VtxInfos[a].m_pArrays.size() > 0)
        {
            dup->m_VtxInfos[a].m_pArrays.clear();

            for(UINT iShape = 0; iShape < m_VtxInfos[a].m_pArrays.size(); ++iShape)
            {
                switch(Dcc::RPrimVtx::VtxAttr(a))
                {
                case Dcc::RPrimVtx::POS0:
                case Dcc::RPrimVtx::NRM0:
                    {
                        const Dcc::RVec3Array* va =
                            reinterpret_cast<const Dcc::RVec3Array*>(m_VtxInfos[a].m_pArrays[iShape]);
                        if(va)
                        {
                            Dcc::RVec3Array* d = new Dcc::RVec3Array(*va);
                            dup->m_VtxInfos[a].m_pArrays.push_back(d);
                        }
                        break;
                    }
                case Dcc::RPrimVtx::TAN0:
                case Dcc::RPrimVtx::TAN1:
                case Dcc::RPrimVtx::TAN2:
                case Dcc::RPrimVtx::TAN3:
                case Dcc::RPrimVtx::BIN0:
                case Dcc::RPrimVtx::BIN1:
                case Dcc::RPrimVtx::BIN2:
                case Dcc::RPrimVtx::BIN3:
                case Dcc::RPrimVtx::COL0:
                case Dcc::RPrimVtx::COL1:
                case Dcc::RPrimVtx::COL2:
                case Dcc::RPrimVtx::COL3:
                case Dcc::RPrimVtx::COL4:
                case Dcc::RPrimVtx::COL5:
                case Dcc::RPrimVtx::COL6:
                case Dcc::RPrimVtx::COL7:
                    {
                        const Dcc::RVec4Array* va =
                            reinterpret_cast<const Dcc::RVec4Array*>(m_VtxInfos[a].m_pArrays[iShape]);
                        if(va)
                        {
                            Dcc::RVec4Array* d = new Dcc::RVec4Array(*va);
                            dup->m_VtxInfos[a].m_pArrays.push_back(d);
                        }
                        break;
                    }
                case Dcc::RPrimVtx::TEX0:
                case Dcc::RPrimVtx::TEX1:
                case Dcc::RPrimVtx::TEX2:
                case Dcc::RPrimVtx::TEX3:
                case Dcc::RPrimVtx::TEX4:
                case Dcc::RPrimVtx::TEX5:
                case Dcc::RPrimVtx::TEX6:
                case Dcc::RPrimVtx::TEX7:
                    {
                        const Dcc::RVec2Array* va =
                            reinterpret_cast<const Dcc::RVec2Array*>(m_VtxInfos[a].m_pArrays[iShape]);
                        if(va)
                        {
                            Dcc::RVec2Array* d = new Dcc::RVec2Array(*va);
                            dup->m_VtxInfos[a].m_pArrays.push_back(d);
                        }
                        break;
                    }
                default:
                    break;
                }

            }
        }
        else
        {
            dup->m_VtxInfos[a].m_pArrays.clear();
        }

        //	m_pArrays は初期化時に NULL を一つ追加されている。
        //	FDC はその状態を前提として書かれているため
        //	配列が完全に空になるとクラッシュする箇所がある。
        //	ここでは一旦クリアしているため、配列が空になる可能性があるので回避する。
        if( dup->m_VtxInfos[a].m_pArrays.size() == 0 )
        {
            dup->m_VtxInfos[a].m_pArrays.push_back(nullptr);
        }
    }

    dup->m_uvHintIdxs = m_uvHintIdxs;
    dup->m_uvAttribNames = m_uvAttribNames;

    return dup;
}

//----------------------------------------------------------------------------
// 頂点属性毎の、データ配列の要素数を取得
size_t FOutShapeInfo::getNumOfVertexInfo(Dcc::RPrimVtx::VtxAttr attr) const
{
    // データ配列が空ならば0を返す。
    if(m_VtxInfos[attr].m_pArrays.size() == 0) return 0;

    // 頂点属性毎にmVtxInfos[attr].m_pArraysをキャストしてサイズを取得する。
    size_t numData = 0;
    // とりあえず最初の頂点情報を使用する
    void* arrayPtr = const_cast<void*>(m_VtxInfos[attr].m_pArrays[0]);
    if(arrayPtr == nullptr) return 0;

    switch(attr)
    {
        case Dcc::RPrimVtx::POS0:
        case Dcc::RPrimVtx::NRM0:
        {
            numData = reinterpret_cast<Dcc::RVec3Array*>(arrayPtr)->size();
            break;
        }
        case Dcc::RPrimVtx::TAN0:
        case Dcc::RPrimVtx::BIN0:
        case Dcc::RPrimVtx::COL0:
        case Dcc::RPrimVtx::COL1:
        case Dcc::RPrimVtx::COL2:
        case Dcc::RPrimVtx::COL3:
        case Dcc::RPrimVtx::COL4:
        case Dcc::RPrimVtx::COL5:
        case Dcc::RPrimVtx::COL6:
        case Dcc::RPrimVtx::COL7:
        {
            numData = reinterpret_cast<Dcc::RVec4Array*>(arrayPtr)->size();
            break;
        }
        case Dcc::RPrimVtx::TEX0:
        case Dcc::RPrimVtx::TEX1:
        case Dcc::RPrimVtx::TEX2:
        case Dcc::RPrimVtx::TEX3:
        case Dcc::RPrimVtx::TEX4:
        case Dcc::RPrimVtx::TEX5:
        case Dcc::RPrimVtx::TEX6:
        case Dcc::RPrimVtx::TEX7:
        {
            numData = reinterpret_cast<Dcc::RVec2Array*>(arrayPtr)->size();
            break;
        }
        default:
            break;
    }

    return numData;
}



//----------------------------------------------------------------------------
// 頂点属性データを関連付け
void FShape::setVertexData(const FOutShapeInfo &shapeInfo)
{
    // 直前に設定されているものがあれば削除する。
    if(m_ShapeInfo)
    {
        delete m_ShapeInfo;
        m_ShapeInfo = nullptr;
    }

    // 頂点属性データの複製を作成する。
    m_ShapeInfo = shapeInfo.duplicate();
}

//----------------------------------------------------------------------------
// インデックスマップを参照して頂点データ配列の並べ替えを行う。
template<typename T>
void copyVertexDataByIndexMap(vector<T*>& dataArrays, int* indexMap)
//void copyVertexDataByIndexMap(vector<Dcc::RVec3Array*>& dataArrays, int* indexMap)
{
    assert(dataArrays.size() > 0);
    for (size_t iKey = 1; iKey < dataArrays.size(); iKey++)
    {
        assert(dataArrays[0]->size() == dataArrays[iKey]->size());
        if(dataArrays[0]->size() != dataArrays[iKey]->size()) return;
    }

    // すべてのキーシェイプで並べ替えを行う
    for (size_t iKey = 0; iKey < dataArrays.size(); iKey++)
    {
        T& dataArray = *(dataArrays[iKey]);
        //Dcc::RVec3Array* dataArray = dataArrays[iKey];
        int newDataSize = 0;

        for(size_t v = 0; v < dataArray.size(); v++)
        {
            // v番目の頂点をindexMap[v]番目にコピーする。
            if(0 <= indexMap[v])
            {
                dataArray[indexMap[v]] = dataArray[v];
                newDataSize = max(newDataSize, indexMap[v]+1);
            }
            // (indexMap[v] < 0)の場合、その頂点は使われないことを意味する。
            // よってコピーしなくても大丈夫。
        }
        // データ配列のサイズを修正する。
        dataArray.resize(newDataSize);
    }
}


//----------------------------------------------------------------------------
// 参照されていない頂点属性データを削除
void FShape::deleteUnusedVertexData(void)
{
    // setVertexData() が呼び出されている必要あり。
    assert(m_ShapeInfo);

#if 0
    // シェープアニメ対応に悪影響の可能性があるので、無効化
    #pragma message(TODO("シェイプアニメーション用キーシェープのコードは未実装"))
    return;
#else
    // 頂点データの数
    size_t numData[Dcc::RPrimVtx::VA_COUNT];

    // 使用されている頂点属性に対して参照フラグをセットするための配列を作成
    bool* usedFlags[Dcc::RPrimVtx::VA_COUNT];
    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        numData[a] = m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::VtxAttr(a));
        if(0 < numData[a])
        {
            usedFlags[a] = new bool [numData[a]];
            memset(usedFlags[a], 0, sizeof(bool)*numData[a]);
        }
        else
        {
            usedFlags[a] = nullptr;
        }
    }

    // 次にプリミティブから参照されている頂点データにフラグをセットする
    Dcc::ItRPrimPtr it = m_pPrimitives.begin();
    while(it != m_pPrimitives.end())
    {
        // プリミティブ内の全頂点データから、
        Dcc::RPrimitive* prim = *it;
        for(int v = 0; v < prim->m_VtxCount; v++)
        {
            // 頂点属性毎に使用している頂点属性データのインデックスを取得し、
            // 参照フラグをセットする。
            const Dcc::RPrimVtx &pv = prim->m_Vtxs[v];
            for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
            {
                if(0 < numData[a])
                {
                    usedFlags[a][pv[a]] = true;
                }
            }
        }
        ++it;
    }

    // 頂点属性毎に、使われていない頂点データの削除を行う。
    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        if(numData[a] == 0) continue;

        // インデックスマップ
        // 元々の頂点(v)が、何番に移動したか(indexMap[v])を記録する。
        int* indexMap = new int [numData[a]];

        int s = 0;
        for(size_t v = 0; v < numData[a]; v++)
        {
            if(usedFlags[a][v])
            {
                // v番の頂点は参照されているので、s番に移動。
                indexMap[v] = s++;
            }
            else
            {
                // v番の頂点は参照されていないので破棄。
                indexMap[v] = -1;
            }
        }

        // 頂点データ配列の並べ替えを行う。
        //void* arrayPtr = const_cast<void*>(m_ShapeInfo->m_VtxInfos[a].m_ArrayPtr);
        Dcc::ROutShapeVtxInfo &attrData = m_ShapeInfo->m_VtxInfos[a];
        switch(Dcc::RPrimVtx::VtxAttr(a))
        {
        case Dcc::RPrimVtx::POS0:
        case Dcc::RPrimVtx::NRM0:
            {
                vector<Dcc::RVec3Array*> dataArrays;
                for(UINT ai = 0; ai < attrData.m_pArrays.size(); ai++)
                {
                    dataArrays.push_back(reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(attrData.m_pArrays[ai])));
                }
                copyVertexDataByIndexMap(dataArrays, indexMap);
                break;
            }
        case Dcc::RPrimVtx::TAN0:
        case Dcc::RPrimVtx::BIN0:
        case Dcc::RPrimVtx::COL0:
        case Dcc::RPrimVtx::COL1:
        case Dcc::RPrimVtx::COL2:
        case Dcc::RPrimVtx::COL3:
        case Dcc::RPrimVtx::COL4:
        case Dcc::RPrimVtx::COL5:
        case Dcc::RPrimVtx::COL6:
        case Dcc::RPrimVtx::COL7:
            {
                vector<Dcc::RVec4Array*> dataArrays;
                for(UINT ai = 0; ai < attrData.m_pArrays.size(); ai++)
                {
                    dataArrays.push_back(reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(attrData.m_pArrays[ai])));
                }
                copyVertexDataByIndexMap(dataArrays, indexMap);
                break;
            }
        case Dcc::RPrimVtx::TEX0:
        case Dcc::RPrimVtx::TEX1:
        case Dcc::RPrimVtx::TEX2:
        case Dcc::RPrimVtx::TEX3:
        case Dcc::RPrimVtx::TEX4:
        case Dcc::RPrimVtx::TEX5:
        case Dcc::RPrimVtx::TEX6:
        case Dcc::RPrimVtx::TEX7:
            {
                vector<Dcc::RVec2Array*> dataArrays;
                for(UINT ai = 0; ai < attrData.m_pArrays.size(); ai++)
                {
                    dataArrays.push_back(reinterpret_cast<Dcc::RVec2Array*>(const_cast<void*>(attrData.m_pArrays[ai])));
                }
                //Dcc::RVec2Array* dataArray = reinterpret_cast<Dcc::RVec2Array*>(arrayPtr);
                copyVertexDataByIndexMap(dataArrays, indexMap);
                break;
            }
        default:
            break;
        }

        // プリミティブのインデックス配列を修正する。
        Dcc::ItRPrimPtr it = m_pPrimitives.begin();
        while(it != m_pPrimitives.end())
        {
            bool res = true;
            for(int v = 0; v < (*it)->m_VtxCount; v++)
            {
                // i番のインデックスをindexMap[i]番に変換する。
                UINT i = (*it)->m_Vtxs[v][a];
                if(i < numData[a])
                {
                    (*it)->m_Vtxs[v][a] = indexMap[i];
                }
                else
                {
                    // 置き換え後のインデックス値が頂点属性データの総数(mapSize)を
                    // 超えることは本来ありえない。とりあえずインデックス値0を代入
                    // するが、ここが実行されるのはバグが原因。
                    (*it)->m_Vtxs[v][a] = 0;
                    res = false;
                }
            }

            assert(res);

            ++it;
        }

        // インデックスマップを削除
        delete [] indexMap;
    }

    // 参照フラグ配列を削除
    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        delete [] usedFlags[a];
    }
#endif
}

//----------------------------------------------------------------------------
template<typename T>
bool mergeSameDataAndMakeIndexMap(T &dataArray, int* indexMap)
{
#if	1
    // 頂点が未処理であることを示すためにインデックスマップを-1にリセットする。
    size_t indexMapSize = dataArray.size();
    for(size_t v = 0; v < indexMapSize; v++) indexMap[v] = -1;

    T::value_type**	pDataArray = new T::value_type*[dataArray.size()];
    memset( pDataArray, 0, sizeof(T*) * dataArray.size() );

    for( size_t v = 0; v < dataArray.size();v++ )
    {
        pDataArray[v] = &dataArray[v];
    }

    // 値が同じ頂点をマージする
    // !!!! 最適化すること
    unsigned int s = 0;
    for(size_t v = 0; v < dataArray.size(); v++)
    {
        // すでに他の頂点とマージ済みならば何もしない。
        if(indexMap[v] != -1) continue;

        // この頂点より前でマージされた分をつめる。
        pDataArray[s] = pDataArray[v];
        indexMap[v] = s;

        // 後ろに向かって同じ値の頂点を検索してマージする。
        for(size_t w = v+1; w < dataArray.size(); w++)
        {
            if( indexMap[w] != -1 )	continue;

            if(*pDataArray[w] == *pDataArray[s])
            {
                indexMap[w] = s;
            }
        }

        s++;
    }

    for( size_t v = 0; v < s;v++ )
    {
        dataArray[v] = *pDataArray[v];
    }

    // データ配列のサイズを修正する。
    dataArray.resize(s);

    delete[] pDataArray;
#else
    // 頂点が未処理であることを示すためにインデックスマップを-1にリセットする。
    size_t indexMapSize = dataArray.size();
    for(size_t v = 0; v < indexMapSize; v++) indexMap[v] = -1;

    // 値が同じ頂点をマージする
    // !!!! 最適化すること

    typedef std::vector<Dcc::RVec3> Dcc::RVec3Array;

    int s = 0;
    for(size_t v = 0; v < dataArray.size(); v++)
    {
        // すでに他の頂点とマージ済みならば何もしない。
        if(indexMap[v] != -1) continue;

        // この頂点より前でマージされた分をつめる。
        dataArray[s] = dataArray[v];
        indexMap[v] = s;

        // 後ろに向かって同じ値の頂点を検索してマージする。
        for(size_t w = v+1; w < dataArray.size(); w++)
        {
            if(dataArray[w] == dataArray[s])
            {
                indexMap[w] = s;
            }
        }

        s++;
    }

    // データ配列のサイズを修正する。
    dataArray.resize(s);
#endif

    return true;
}

//----------------------------------------------------------------------------
template<typename T>
bool isSameAttributeInAllKeyShapes(vector<T*> &dataArrays)
//bool isSameAttributeInAllKeyShapes(vector<Dcc::RVec3Array*>& dataArrays)
{
    assert(dataArrays.size() > 1);
    //Dcc::RVec3Array* baseArray = dataArrays[0];
    T* baseArray = dataArrays[0];

    bool isSame = true;

    // すべてのキーシェイプで値を比較
    for(UINT iKey = 1; iKey < dataArrays.size(); iKey++)
    {
        //Dcc::RVec3Array* targetArray = dataArrays[iKey];
        T* targetArray = dataArrays[iKey];
        assert(baseArray->size() == targetArray->size()) ;

        for(UINT iVtx = 0; iVtx < baseArray->size(); iVtx++)
        {
            isSame = baseArray->at(iVtx).IsEquivalent(targetArray->at(iVtx));
            if(!isSame)
            {
                break;
            }
        }

        if (!isSame)
        {
            break;
        }
    }

    return isSame;
}


//----------------------------------------------------------------------------
// すべてのキーシェイプが同じ値の頂点属性データはキーシェイプを削除
bool FShape::deleteSameKeyShapes(void)
{
    // setVertexData() が呼び出されている必要あり。
    assert(m_ShapeInfo);

    bool isAllKeyShapesDeleted = true;

    // 頂点属性毎に独立して、同じ値を持つデータのマージを行う。
    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        Dcc::ROutShapeVtxInfo &attrData = m_ShapeInfo->m_VtxInfos[a];
        // データ配列が一つ以下ならば何もしない。
        if(attrData.m_pArrays.size() <= 1) continue;

        switch(Dcc::RPrimVtx::VtxAttr(a))
        {
        case Dcc::RPrimVtx::POS0:
        case Dcc::RPrimVtx::NRM0:
            {
                vector<Dcc::RVec3Array*> dataArrays;
                for(UINT ai = 0; ai < attrData.m_pArrays.size(); ai++)
                {
                    dataArrays.push_back(reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(attrData.m_pArrays[ai])));
                }
                // すべてのキーシェイプですべての頂点値を比較し、同じならキーシェイプを削除
                if(isSameAttributeInAllKeyShapes(dataArrays))
                {
                    attrData.m_pArrays.resize(1);
                }
                else
                {
                    isAllKeyShapesDeleted = false;
                }
                break;
            }

        case Dcc::RPrimVtx::TAN0:
        case Dcc::RPrimVtx::BIN0:
        case Dcc::RPrimVtx::COL0:
        case Dcc::RPrimVtx::COL1:
        case Dcc::RPrimVtx::COL2:
        case Dcc::RPrimVtx::COL3:
        case Dcc::RPrimVtx::COL4:
        case Dcc::RPrimVtx::COL5:
        case Dcc::RPrimVtx::COL6:
        case Dcc::RPrimVtx::COL7:
            {
                vector<Dcc::RVec4Array*> dataArrays;
                for(UINT ai = 0; ai < attrData.m_pArrays.size(); ai++)
                {
                    dataArrays.push_back(reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(attrData.m_pArrays[ai])));
                }

                // すべてのキーシェイプですべての頂点値を比較し、同じならキーシェイプを削除
                if(isSameAttributeInAllKeyShapes(dataArrays))
                {
                    attrData.m_pArrays.resize(1);
                }
                else
                {
                    isAllKeyShapesDeleted = false;
                }
                break;
            }
        default:
            break;
        }
    }

    if (isAllKeyShapesDeleted)
    {
        m_ShapeInfo->m_KeyNames.clear();
    }

    return true;
}


//----------------------------------------------------------------------------
// 同じ値を持つ頂点属性データをマージし、プリミティブのインデックス配列を修正
bool FShape::mergeSameVertexData(void)
{
    // setVertexData() が呼び出されている必要あり。
    assert(m_ShapeInfo);

    // 頂点属性毎に独立して、同じ値を持つデータのマージを行う。
    for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
    {
        // #pragma message(TODO("シェイプアニメーション用キーシェープのコードは未実装"))
        Dcc::ROutShapeVtxInfo &attrData = m_ShapeInfo->m_VtxInfos[a];
        // データ配列が一つ以外はならば何もしない。
        // データがない場合、シェープアニメーションがある場合はマージを行わない。
        if(attrData.m_pArrays.size() != 1 || attrData.m_pArrays[0] == nullptr) continue;

        // インデックスマップを作成。
        // 元々の頂点(v)が、何番に移動したか(indexMap[v])を記録する。
        int* indexMap = nullptr;
        size_t indexMapSize = 0;

        // 属性毎にデータ配列の型をキャストして同じ値を持つ要素のマージを行う。
        void* arrayPtr = const_cast<void*>(attrData.m_pArrays[0]);
        switch(Dcc::RPrimVtx::VtxAttr(a))
        {
        case Dcc::RPrimVtx::POS0:
        case Dcc::RPrimVtx::NRM0:
        {
            Dcc::RVec3Array* dataArray = reinterpret_cast<Dcc::RVec3Array*>(arrayPtr);
            indexMapSize = dataArray->size();
            indexMap = new int [indexMapSize];
            mergeSameDataAndMakeIndexMap(*dataArray, indexMap);
            break;
        }
        case Dcc::RPrimVtx::TAN0:
        case Dcc::RPrimVtx::BIN0:
        case Dcc::RPrimVtx::COL0:
        case Dcc::RPrimVtx::COL1:
        case Dcc::RPrimVtx::COL2:
        case Dcc::RPrimVtx::COL3:
        case Dcc::RPrimVtx::COL4:
        case Dcc::RPrimVtx::COL5:
        case Dcc::RPrimVtx::COL6:
        case Dcc::RPrimVtx::COL7:
        {
            Dcc::RVec4Array* dataArray = reinterpret_cast<Dcc::RVec4Array*>(arrayPtr);
            indexMapSize = dataArray->size();
            indexMap = new int [indexMapSize];
            mergeSameDataAndMakeIndexMap(*dataArray, indexMap);
            break;
        }
        case Dcc::RPrimVtx::TEX0:
        case Dcc::RPrimVtx::TEX1:
        case Dcc::RPrimVtx::TEX2:
        case Dcc::RPrimVtx::TEX3:
        case Dcc::RPrimVtx::TEX4:
        case Dcc::RPrimVtx::TEX5:
        case Dcc::RPrimVtx::TEX6:
        case Dcc::RPrimVtx::TEX7:
        {
            Dcc::RVec2Array* dataArray = reinterpret_cast<Dcc::RVec2Array*>(arrayPtr);
            indexMapSize = dataArray->size();
            indexMap = new int [indexMapSize];
            mergeSameDataAndMakeIndexMap(*dataArray, indexMap);
            break;
        }
        default:
            break;
        }
        // 未定義の頂点属性セマンティックだった(ありえない)
        if(indexMap == nullptr) continue;

        // プリミティブのインデックス配列を修正する。
        Dcc::ItRPrimPtr it = m_pPrimitives.begin();
        while(it != m_pPrimitives.end())
        {
            bool res = true;
            for(int v = 0; v < (*it)->m_VtxCount; v++)
            {
                // i番のインデックスをindexMap[i]番に変換する。
                int i = (*it)->m_Vtxs[v][a];
                if(i < static_cast<int>(indexMapSize))
                {
                    (*it)->m_Vtxs[v][a] = indexMap[i];
                }
                else
                {
                    // 置き換え後のインデックス値が頂点属性データの総数(mapSize)を
                    // 超えることは本来ありえない。とりあえずインデックス値0を代入
                    // するが、ここが実行されるのはバグが原因。
                    (*it)->m_Vtxs[v][a] = 0;
                    res = false;
                }
            }

            assert(res);

            ++it;
        }

        // インデックスマップを削除
        delete [] indexMap;
    }
    return true;
}


//----------------------------------------------------------------------------
// モデルの頂点情報を拡大縮小する
bool FShape::magnifyVertexData( float magnify )
{
    // setVertexData() が呼び出されている必要あり。
    assert(m_ShapeInfo);
    Dcc::ROutShapeVtxInfo &attrData = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::POS0];
    if(attrData.m_pArrays.size() == 0) return false;

    size_t shapeNum = attrData.m_pArrays.size();

    for(size_t shapeIdx = 0; shapeIdx < shapeNum; ++shapeIdx)
    {
        //Dcc::RVec3Array* dataArray = reinterpret_cast<Dcc::RVec3Array*>(arrayPtr);
        // 美しくないconst外し…
        Dcc::RVec3Array* dataArray = reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(attrData.m_pArrays[shapeIdx]));
        if(dataArray == nullptr) continue;
        size_t vtxNum = dataArray->size();
        for(UINT v = 0; v < vtxNum; ++v)
        {
            dataArray->at(v) = dataArray->at(v) * magnify;
        }
    }
    return true;
}

//----------------------------------------------------------------------------
// シェイプと頂点情報から法線マップ用のタンジェントを計算する。
Dcc::RStatus FShape::calcTangent( Dcc::RPrimVtx::VtxAttr texCh, bool isUVBackward /*= false*/ )
{

    //#pragma message(TODO("シェイプアニメーション用キーシェープのコードは未実装"))
    // setVertexData() より前に呼び出す必要が有る。
    if(!m_ShapeInfo)
    {
        return Dcc::RStatus::FAILURE;
    }

    // タンジェントがすでにセットされていたら何もしない
    if(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::TAN0) > 0)
    { return Dcc::RStatus::FAILURE; }

    // 対象のUV座標の範囲が正しくなければ何もしない
    if(texCh < Dcc::RPrimVtx::TEX0 || Dcc::RPrimVtx::TEX7 < texCh)
    { return Dcc::RStatus::FAILURE; }

    // 頂点座標がセットされていなければ何もしない
    if(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::POS0) == 0)
    { return Dcc::RStatus::FAILURE; }

    // 法線がセットされていなければ何もしない
    if(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0) == 0)
    { return Dcc::RStatus::FAILURE; }

    // 対象のUV座標がセットされていなければ何もしない
    if(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::VtxAttr(texCh)) == 0)
    { return Dcc::RStatus::FAILURE; }

    Dcc::RStatus ret = Dcc::RStatus::SUCCESS;

    // 頂点データの数
    //size_t numData[RPrimVtx::VA_MAX];
    Dcc::RPrimVtx::VtxAttr texAttr(texCh);

    // 座標、法線、UVデータへのポインタ
    vector<const void*> posArrays = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::POS0].m_pArrays;
    vector<const void*> nrmArrays = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays;
    //vector<const void*> texArrays = m_ShapeInfo->m_VtxInfos[texAttr].m_pArrays;
    Dcc::RVec2Array* texArray = reinterpret_cast<Dcc::RVec2Array*>(
        const_cast<void*>(m_ShapeInfo->m_VtxInfos[texAttr].m_pArrays[0]));

    assert(posArrays.size() == nrmArrays.size());

    // 長さが 0 の接線または従法線が存在
    bool hasZeroTangent = false;

    // keyshapeの数だけタンジェント作成を行う
    for(size_t shapei = 0; shapei < posArrays.size(); shapei++)
    {
        //各keyshapeの 座標、法線、UVデータへのポインタ
        Dcc::RVec3Array* posArray = reinterpret_cast<Dcc::RVec3Array*>(
            const_cast<void*>(m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::POS0].m_pArrays[shapei]));
        Dcc::RVec3Array* nrmArray = reinterpret_cast<Dcc::RVec3Array*>(
            const_cast<void*>(m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays[shapei]));

        // タンジェントを法線と同じ数だけ作る()
        Dcc::RVec3Array* tanArray = new Dcc::RVec3Array(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        Dcc::RVec3Array* binArray = new Dcc::RVec3Array(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        //Dcc::RVec3Array tmpTanArray(m_ShapeInfo->getNumOfVertexInfo(RPrimVtx::NRM));
        Dcc::RVec3Array tmpPosArray(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        Dcc::RVec2Array tmpTexArray(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        Dcc::RVec3Array& tmpTanArray = (*tanArray);
        Dcc::RVec3Array& tmpBinArray = (*binArray);

        Dcc::ItRPrimPtr it = m_pPrimitives.begin();


        // 計算する頂点と前後の頂点のインデックス
        int id0, id1,id2;

        // プリミティブ
        it = m_pPrimitives.begin();
        while(it != m_pPrimitives.end())
        {
            // プリミティブ内の全頂点データから、
            Dcc::RPrimitive* prim = *it;
            int numVtxIdx = prim->m_VtxCount;

            for(int v = 0; v < numVtxIdx; v++)
            {
                // 前後の頂点からタンジェントを計算する。
                id0 = v;
                id1 = (v - 1 + numVtxIdx) % numVtxIdx;
                id2 = (v + 1) % numVtxIdx;
                Dcc::RPrimVtx &pv = const_cast<Dcc::RPrimVtx &>(prim->m_Vtxs[id0]);
                Dcc::RPrimVtx &pv_prev = const_cast<Dcc::RPrimVtx &>(prim->m_Vtxs[id1]); //前の頂点
                Dcc::RPrimVtx &pv_next = const_cast<Dcc::RPrimVtx &>(prim->m_Vtxs[id2]); //次の頂点
                //int idx = pv[RPrimVtx::POS];
                Dcc::RVec3 pos0 = posArray->at(pv[Dcc::RPrimVtx::POS0]);
                Dcc::RVec3 pos1 = posArray->at(pv_prev[Dcc::RPrimVtx::POS0]);
                Dcc::RVec3 pos2 = posArray->at(pv_next[Dcc::RPrimVtx::POS0]);
                Dcc::RVec2 uv0 = texArray->at(pv[texAttr]);
                Dcc::RVec2 uv1 = texArray->at(pv_prev[texAttr]);
                Dcc::RVec2 uv2 = texArray->at(pv_next[texAttr]);
                Dcc::RVec3 oNrm = nrmArray->at(pv[Dcc::RPrimVtx::NRM0]);
                Dcc::RVec3 oTan, oBin;

                if(isUVBackward) // maxはv座標がフリップされるため、戻して計算する
                {
                    uv0.y *= -1.0f;
                    uv1.y *= -1.0f;
                    uv2.y *= -1.0f;
                }

                //	TODO:	utility -> FDC へ変更すること
                Dcc::RCalcTangent(oTan, oBin, pos0, pos1, pos2, uv0, uv1, uv2, oNrm);

                if (!oTan.IsFinite())
                {
                    oTan = Dcc::RVec3::kZero;
                }
                //-----------------------------------------------------------------------------
                // 接線の長さが 0 ならエッジ方向を接線とします。
                if (oTan == Dcc::RVec3::kZero || !oTan.IsFinite())
                {
                    oTan = pos1 - pos0;
                    oTan.Normalize();
                    oTan.SnapToZero();
                    if (oTan == Dcc::RVec3::kZero)
                    {
                        oTan = (!oNrm.IsEquivalent(Dcc::RVec3::kXAxis) && !oNrm.IsEquivalent(Dcc::RVec3::kXNegAxis)) ?
                            Dcc::RVec3::kXAxis : Dcc::RVec3::kZAxis;
                    }
                    hasZeroTangent = true;
                }

                //-----------------------------------------------------------------------------
                // 従法線の長さが 0 なら法線と接線から従法線を計算します。
                if (oBin == Dcc::RVec3::kZero || !oBin.IsFinite())
                {
                    oBin = oNrm ^ oTan;
                    oBin.Normalize();
                    oBin.SnapToZero();
                    if (oBin == Dcc::RVec3::kZero)
                    {
                        oBin = (!oNrm.IsEquivalent(Dcc::RVec3::kZAxis) && !oNrm.IsEquivalent(Dcc::RVec3::kZNegAxis)) ?
                            Dcc::RVec3::kZNegAxis : Dcc::RVec3::kXAxis;
                    }
                    hasZeroTangent = true;
                }

                pv[Dcc::RPrimVtx::TAN0] = pv[Dcc::RPrimVtx::NRM0];
                pv[Dcc::RPrimVtx::BIN0] = pv[Dcc::RPrimVtx::NRM0];

                tmpTanArray.at(pv[Dcc::RPrimVtx::TAN0]) = oTan;
                tmpBinArray.at(pv[Dcc::RPrimVtx::TAN0]) = oBin;
                tmpPosArray.at(pv[Dcc::RPrimVtx::TAN0]) = pos0;
                tmpTexArray.at(pv[Dcc::RPrimVtx::TAN0]) = uv0;
            }
            ++it;
        }

        // 全ての頂点から位置、UVが同じ頂点を探し、タンジェントを平均化する。
        //vector<Dcc::RIntArray> sameTanList;
        Dcc::RBoolArray smoothFlag(tanArray->size());
        for(UINT i = 0; i < tanArray->size(); i++) smoothFlag[i] = false;

#if 1
        for(UINT i = 0; i < (tanArray->size() - 1); i++)
        {
            if(smoothFlag[i]) continue;
            // 同じ値を持つ頂点リスト
            Dcc::RIntArray sameTanList;
            for(UINT j = (i + 1); j < tanArray->size(); j++)
            {
                //float rad = (tmpTanArray[i] * tmpTanArray[j]);
                //float ang = (float)cos(R_M_PI * 0.25);

                if( !smoothFlag[j] &&
                    (tmpPosArray[i] - tmpPosArray[j]).Length() < Dcc::R_SAME_TOLERANCE_F &&
                    (tmpTexArray[i] - tmpTexArray[j]).Length() < Dcc::R_SAME_TOLERANCE_F  &&
                    (tmpTanArray[i] * tmpTanArray[j]) > static_cast<float>(cos(Dcc::R_M_PI * 0.5)) //二つのタンジェントが90度以内ならスムース
                  )
                {
                    sameTanList.push_back(j);
                    smoothFlag[j] = true;
                }
            }

            // 同じ値の頂点が見つかれば全て足してノーマライズ
            if(sameTanList.size() > 0)
            {
                sameTanList.push_back(i);
                Dcc::RVec3 smoothTan(0.0f, 0.0f, 0.0f);
                Dcc::RVec3 smoothBin(0.0f, 0.0f, 0.0f);
                for(UINT k = 0; k < sameTanList.size(); k++)
                {
                    smoothTan += tanArray->at(sameTanList[k]);
                    smoothBin += binArray->at(sameTanList[k]);
                }
                smoothTan.Normalize();
                smoothBin.Normalize();
                for(UINT k = 0; k < sameTanList.size(); k++)
                {
                    tanArray->at(sameTanList[k]) = smoothTan;
                    binArray->at(sameTanList[k]) = smoothBin;
                }
            }
        }
#endif

        // CafeではTangent,binormalはVec4なので、ここでコピー
        // また法線などの座標系を判定してw成分にフラグを入れる
        Dcc::RVec4Array* tan4Array = new Dcc::RVec4Array(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        Dcc::RVec4Array* bin4Array = new Dcc::RVec4Array(m_ShapeInfo->getNumOfVertexInfo(Dcc::RPrimVtx::NRM0));
        for(UINT i = 0; i < tanArray->size(); ++i)
        {
            Dcc::RVec3 tan = tanArray->at(i);
            Dcc::RVec3 bin = binArray->at(i);
            Dcc::RVec3 nrm = nrmArray->at(i);

            tan.Normalize();
            bin.Normalize();
            nrm.Normalize();

            const float w = (RIsRightHandTangent(tan, bin, nrm)) ? 1.0f : -1.0f;

            tan4Array->at(i) = Dcc::RVec4(tan.x, tan.y, tan.z, w);
            bin4Array->at(i) = Dcc::RVec4(bin.x, bin.y, bin.z, w);
        }
        delete tanArray;
        delete binArray;

        // 最初のBaseShapeはROutShapeVtxInfoを作成
        // それ以降は追加する
        if(shapei == 0)
        {
            m_VtxAttrFlag[Dcc::RPrimVtx::TAN0] = true;
            m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::TAN0] = Dcc::ROutShapeVtxInfo(4, Dcc::RPrimVtx::VALUE_FLOAT, tan4Array);
            m_VtxAttrFlag[Dcc::RPrimVtx::BIN0] = true;
            m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::BIN0] = Dcc::ROutShapeVtxInfo(4, Dcc::RPrimVtx::VALUE_FLOAT, bin4Array);
        }
        else
        {
            m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::TAN0].m_pArrays.push_back(tan4Array);
            m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::BIN0].m_pArrays.push_back(bin4Array);
        }
    }

    if(hasZeroTangent)
    {
        ret =  Dcc::RStatus(Dcc::RStatus::SUCCESS, "hasZeroTangent");
    }

    return ret;
}

//----------------------------------------------------------------------------
// 頂点座標、法線ベクトルを変換する
void FShape::transformVertexData(const Dcc::RMtx44& mtx)
{
    // setVertexData() により頂点属性データが複製されている必要あり。
    assert(m_ShapeInfo);

    // 座標を変換
    Dcc::ROutShapeVtxInfo& vtxPositionInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::POS0];
    for(UINT iInfo = 0; iInfo < vtxPositionInfo.m_pArrays.size(); ++iInfo)
    {
        //	m_ArrayPtr が const void * のため、無理やり const はずし
        //	Maya では OutShapeVtxInfo へ入れる前に変換しているので const になっているようだ。
        Dcc::RVec3Array* pPosArray = reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(vtxPositionInfo.m_pArrays[iInfo]));
        if(pPosArray)
        {
            Dcc::RVec3Array& posArray = *pPosArray;
            for(size_t i = 0; i < posArray.size(); i++)
            {
                Dcc::RVec4 v = Dcc::RVec4(posArray[i]) * mtx;
                posArray[i] = Dcc::RVec3(v);
            }
        }
    }

    // 法線ベクトルを変換
    Dcc::ROutShapeVtxInfo& vtxNormalInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::NRM0];
    for(UINT iInfo = 0; iInfo < vtxNormalInfo.m_pArrays.size(); ++iInfo)
    {
        //	m_ArrayPtr が const void * のため、無理やり const はずし
        //	Maya では OutShapeVtxInfo へ入れる前に変換しているので const になっているようだ。
        Dcc::RVec3Array* pNorArray = reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(vtxNormalInfo.m_pArrays[iInfo]));
        if(pNorArray)
        {
            Dcc::RVec3Array& norArray = *pNorArray;
            for(size_t i = 0; i < norArray.size(); i++)
            {
                norArray[i] *= mtx;
                norArray[i].Normalize();
            }
        }
    }

    // 接線ベクトルを変換

    Dcc::ROutShapeVtxInfo& vtxTangentInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::TAN0];
    for(UINT iInfo = 0; iInfo < vtxTangentInfo.m_pArrays.size(); ++iInfo)
    {
        //	m_ArrayPtr が const void * のため、無理やり const はずし
        //	Maya では OutShapeVtxInfo へ入れる前に変換しているので const になっているようだ。
        Dcc::RVec4Array* pTanArray = reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(vtxTangentInfo.m_pArrays[iInfo]));
        if(pTanArray)
        {
            Dcc::RVec4Array& tanArray = *pTanArray;
            for(size_t i = 0; i < tanArray.size(); i++)
            {
                Dcc::RVec4& tan = tanArray[i];
                Dcc::RVec3 tmp(tan.x, tan.y, tan.z);
                tmp *= mtx;
                tmp.Normalize();

                tan.x = tmp.x;
                tan.y = tmp.y;
                tan.z = tmp.z;

                //tanArray[i] *= mtx;
                //tanArray[i].Normalize();
            }
        }
    }

    // 従法線ベクトルを変換
    Dcc::ROutShapeVtxInfo& vtxBinormalInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::BIN0];
    for(UINT iInfo = 0; iInfo < vtxBinormalInfo.m_pArrays.size(); ++iInfo)
    {
        //	m_ArrayPtr が const void * のため、無理やり const はずし
        //	Maya では OutShapeVtxInfo へ入れる前に変換しているので const になっているようだ。
        Dcc::RVec4Array* pBinArray = reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(vtxBinormalInfo.m_pArrays[iInfo]));
        if(pBinArray)
        {
            Dcc::RVec4Array& binArray = *pBinArray;
            for(size_t i = 0; i < binArray.size(); i++)
            {
                Dcc::RVec4& bin = binArray[i];
                Dcc::RVec3 tmp(bin.x, bin.y, bin.z);
                tmp *= mtx;
                tmp.Normalize();

                bin.x = tmp.x;
                bin.y = tmp.y;
                bin.z = tmp.z;

                //tanArray[i] *= mtx;
                //tanArray[i].Normalize();
            }
        }
    }
}

//----------------------------------------------------------------------------
// 頂点毎に頂点座標、法線ベクトルを参照するボーンの座標系へ変換する
void FShape::transformVertexDataToBone(const Dcc::RMtx44* mtxToBone, const Dcc::RVtxMtxArray& vtxMtxs)
{
    // setVertexData() により頂点属性データが複製されている必要あり。
    assert(m_ShapeInfo);

    // 頂点属性データの配列と、それを変換済みかを示すフラグ配列を作成。
    bool* transformPos = nullptr;
    bool* transformNor = nullptr;
    bool* transformTan = nullptr;
    bool* transformBin = nullptr;
    Dcc::RVec3Array* posArray = nullptr;
    Dcc::RVec3Array* norArray = nullptr;
    Dcc::RVec4Array* tanArray = nullptr;
    Dcc::RVec4Array* binArray = nullptr;

    vector<Dcc::RVec3Array*> posArrays;
    vector<Dcc::RVec3Array*> norArrays;
    vector<Dcc::RVec4Array*> tanArrays;
    vector<Dcc::RVec4Array*> binArrays;

    //#pragma message(TODO("シェイプアニメーション用キーシェープのコードは未実装"))
    {
        // 座標
        Dcc::ROutShapeVtxInfo& vtxPositionInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::POS0];
        for(UINT iInfo = 0; iInfo < vtxPositionInfo.m_pArrays.size(); ++iInfo)
        {
            posArray = reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(vtxPositionInfo.m_pArrays[iInfo]));
            if(posArray)
            {
                posArrays.push_back(posArray);
                transformPos = new bool [posArray->size()];
                memset(transformPos, 0, sizeof(bool) * posArray->size());
            }
        }
        // 法線
        Dcc::ROutShapeVtxInfo& vtxNormalInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::NRM0];
        for(UINT iInfo = 0; iInfo < vtxNormalInfo.m_pArrays.size(); ++iInfo)
        {
            norArray = reinterpret_cast<Dcc::RVec3Array*>(const_cast<void*>(vtxNormalInfo.m_pArrays[iInfo]));
            if(norArray)
            {
                norArrays.push_back(norArray);
                transformNor = new bool [norArray->size()];
                memset(transformNor, 0, sizeof(bool)*norArray->size());
            }
        }
        // 接線
        Dcc::ROutShapeVtxInfo& vtxTangentInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::TAN0];
        for(UINT iInfo = 0; iInfo < vtxTangentInfo.m_pArrays.size(); ++iInfo)
        {
            tanArray = reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(vtxTangentInfo.m_pArrays[iInfo]));
            if(tanArray)
            {
                tanArrays.push_back(tanArray);
                transformTan = new bool [tanArray->size()];
                memset(transformTan, 0, sizeof(bool)*tanArray->size());
            }
        }
        // 従法線
        Dcc::ROutShapeVtxInfo& vtxBinomalInfo = m_ShapeInfo->m_VtxInfos[Dcc::RPrimVtx::TAN0];
        for(UINT iInfo = 0; iInfo < vtxTangentInfo.m_pArrays.size(); ++iInfo)
        {
            binArray = reinterpret_cast<Dcc::RVec4Array*>(const_cast<void*>(vtxBinomalInfo.m_pArrays[iInfo]));
            if(binArray)
            {
                binArrays.push_back(binArray);
                transformBin = new bool [binArray->size()];
                memset(transformBin, 0, sizeof(bool)*binArray->size());
            }
        }
    }

    // シェイプ内の各プリミティブについて、
    Dcc::RPrimPtrArray::const_iterator itp = m_pPrimitives.begin();
    while(itp != m_pPrimitives.end())
    {
        const Dcc::RPrimitive &rprim = **itp;

        // プリミティブを構成する各頂点について、
        for(int v = 0; v < rprim.m_VtxCount; v++)
        {
            // 参照するボーンを取得して、
            const Dcc::RPrimVtx& vtx = rprim.m_Vtxs[v];
            int boneIdx = vtx.m_Mtx;
            if(boneIdx < 0) continue;


            const Dcc::RVtxMtx& vmt = vtxMtxs[boneIdx];

            {
                int boneIndex = vmt.GetBoneIndex( 0 );

                const Dcc::RMtx44 &mtx = mtxToBone[boneIndex];

                // 座標変換を行う。
                int i = vtx[Dcc::RPrimVtx::POS0];
                if(posArray && (0 <= i) && !transformPos[i])
                {
                    for(size_t ishape = 0; ishape < posArrays.size(); ishape++)
                    {
                        Dcc::RVec4 vec = Dcc::RVec4((*posArrays[ishape])[i]) * mtx;
                        (*posArray)[i] = Dcc::RVec3(vec);
                    }
                    transformPos[i] = true;
                }

                i = vtx[Dcc::RPrimVtx::NRM0];
                if(norArray && (0 <= i) && !transformNor[i])
                {
                    for(size_t ishape = 0; ishape < norArrays.size(); ishape++)
                    {
                        (*norArrays[ishape])[i] *= mtx;
                        (*norArrays[ishape])[i].Normalize();
                    }
                    transformNor[i] = true;
                }

                i = vtx[Dcc::RPrimVtx::TAN0];
                if(tanArray && (0 <= i) && !transformTan[i])
                {
                    for(size_t ishape = 0; ishape < tanArrays.size(); ishape++)
                    {
                        Dcc::RVec4& tan = (*tanArrays[ishape])[i];
                        Dcc::RVec3 tmp(tan.x, tan.y, tan.z);
                        tmp *= mtx;
                        tmp.Normalize();
                        tan.x = tmp.x;
                        tan.y = tmp.y;
                        tan.z = tmp.z;
                    }
                    transformTan[i] = true;
                }

                i = vtx[Dcc::RPrimVtx::BIN0];
                if(binArray && (0 <= i) && !transformBin[i])
                {
                    for(size_t ishape = 0; ishape < binArrays.size(); ishape++)
                    {
                        Dcc::RVec4& bin = (*binArrays[ishape])[i];
                        Dcc::RVec3 tmp(bin.x, bin.y, bin.z);
                        tmp *= mtx;
                        tmp.Normalize();
                        bin.x = tmp.x;
                        bin.y = tmp.y;
                        bin.z = tmp.z;
                    }

                    transformBin[i] = true;
                }
            }
        }

        ++itp;
    }

    // フラグ配列を削除
    delete [] transformPos;
    delete [] transformNor;
    delete [] transformTan;
}

//----------------------------------------------------------------------------
//	このシェイプ内で参照しているRVtxBoneRefへのポインタのリストを取得します。
void FShape::GetRefVmtIndexList( std::list<int>&	list )
{
    // シェイプ内の各プリミティブについて、
    Dcc::RPrimPtrArray::const_iterator itp = m_pPrimitives.begin();
    while(itp != m_pPrimitives.end())
    {
        const Dcc::RPrimitive &rprim = **itp;

        // プリミティブを構成する各頂点について、
        for(int v = 0; v < rprim.m_VtxCount; v++)
        {
            // 参照するボーンを取得する
            const Dcc::RPrimVtx& vtx = rprim.m_Vtxs[v];
            int vmtIdx = vtx.m_Mtx;
            if(vmtIdx < 0) continue;	//	未設定
            list.push_back( vmtIdx );
        }

        ++itp;
    }

    //	重複要素を削除
    list.unique();
    //	順番にアクセスするのでソートもしておく
    list.sort();
}



// マテリアルから参照されなかったUV座標を削除する
void FShape::deleteUnusedUV( FMaterial* fmat )
{
    // 削除されたかどうかを表すフラグ
    Dcc::RBoolArray removedUVFlags;
    removedUVFlags.resize(Dcc::RPrimVtx::VTX_TEX_MAX);

    // 削除前と後の対応リスト
    Dcc::RIntArray uvChOrders;
    uvChOrders.resize(Dcc::RPrimVtx::VTX_TEX_MAX);

    for(UINT i = 0; i < Dcc::RPrimVtx::VTX_TEX_MAX; i++)
    {
        removedUVFlags[i] = false;
        uvChOrders[i] = i;
    }

    uvChOrders.resize(m_ShapeInfo->m_uvHintIdxs.size());

    for(UINT i = 0; i < Dcc::RPrimVtx::VTX_TEX_MAX; i++)
    {
        Dcc::RPrimVtx::VtxAttr curTexID = Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + i);
        Dcc::ROutShapeVtxInfo &curAttrData = m_ShapeInfo->m_VtxInfos[curTexID];
        // 何もない場合はスキップ
        if(curAttrData.m_pArrays.size() == 0 || curAttrData.m_pArrays[0] == nullptr)
        {
            continue;
        }

        if(fmat->m_UseCafeUV[i])
        {
            // このシェイプのマテリアルでこのUV座標を使っている場合、
            // 前に空きがあれば詰める
            for(UINT j = 0; j < i; j++)
            {
                Dcc::RPrimVtx::VtxAttr texID = Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + j);
                Dcc::ROutShapeVtxInfo &attr = m_ShapeInfo->m_VtxInfos[texID];
                // このUV座標が空ならコピー
                if(attr.m_pArrays.size() == 0 || attr.m_pArrays[0] == nullptr)
                {
                    attr.m_pArrays.clear();
                    for(UINT k = 0; k < curAttrData.m_pArrays.size(); k++)
                    {
                        attr.m_pArrays.push_back(curAttrData.m_pArrays[k]);
                    }
                    m_VtxAttrFlag[texID] = true;

                    // 元のUV座標を削除
                    curAttrData.m_pArrays.resize(1);
                    curAttrData.m_pArrays[0] = nullptr;
                    m_VtxAttrFlag[curTexID] = false;

                    break;
                }
            }
        }
        else
        {
            // このシェイプの使うマテリアルでこのUV座標を使っていなければ削除
            curAttrData.m_pArrays.resize(1);
            curAttrData.m_pArrays[0] = nullptr;
            m_VtxAttrFlag[curTexID] = false;
            removedUVFlags[i] = true;
            /*
            if(m_ShapeInfo->m_uvHintIdxs.size() > i)
            {
                m_ShapeInfo->m_uvHintIdxs[i] = -1;
            }
            if(m_ShapeInfo->m_uvAttribNames.size() > i)
            {
                m_ShapeInfo->m_uvAttribNames[i] = "";
            }
            */
        }
    }

    // hintIdxとNamesが矛盾
    assert(m_ShapeInfo->m_uvHintIdxs.size() == m_ShapeInfo->m_uvAttribNames.size());

    // remove unused uv hints
    for(int i = static_cast<int>(m_ShapeInfo->m_uvHintIdxs.size() - 1); i >= 0; i--)
    {
        if(removedUVFlags[i])
        {
            m_ShapeInfo->m_uvHintIdxs.erase(m_ShapeInfo->m_uvHintIdxs.begin() + i);
            m_ShapeInfo->m_uvAttribNames.erase(m_ShapeInfo->m_uvAttribNames.begin() + i);
            uvChOrders.erase(uvChOrders.begin() + i);
        }
    }

    // auto number hint idx

    Dcc::RIntArray usedIdxs;
    for(int i = 0; i < m_ShapeInfo->m_uvHintIdxs.size(); i++)
    {
        int id = m_ShapeInfo->m_uvHintIdxs[i];
        if(id >= 0) usedIdxs.push_back(id);
    }

    int lastid = 0;
    for(int i = 0; i < m_ShapeInfo->m_uvHintIdxs.size(); i++)
    {
        int id = m_ShapeInfo->m_uvHintIdxs[i];
        if(id < 0)
        {
            for(int j = lastid; j <= Dcc::RVtxAttrib::HINT_INDEX_MAX; j++)
            {
                Dcc::RIntArray::iterator it = std::find(usedIdxs.begin(), usedIdxs.end(), j);
                if(it == usedIdxs.end())
                {
                    id = j;
                    lastid = j;
                    usedIdxs.push_back(j);
                    break;
                }
            }
            m_ShapeInfo->m_uvHintIdxs[i] = id;
        }
    }

    // reorder by hint
    for(UINT i = 0; i < m_ShapeInfo->m_uvHintIdxs.size(); i++)
    {
        Dcc::RPrimVtx::VtxAttr curTexID = Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + i);
        Dcc::ROutShapeVtxInfo &curAttrData = m_ShapeInfo->m_VtxInfos[curTexID];
        // 何もない場合はスキップ
        if(curAttrData.m_pArrays.size() == 0 || curAttrData.m_pArrays[0] == nullptr)
        {
            continue;
        }
        int lesser = i;
        int lesserHint = m_ShapeInfo->m_uvHintIdxs[lesser];

        for(UINT j = i; j < m_ShapeInfo->m_uvHintIdxs.size(); j++)
        {
            Dcc::RPrimVtx::VtxAttr texID = Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + j);
            Dcc::ROutShapeVtxInfo &attr = m_ShapeInfo->m_VtxInfos[texID];
            // このUV座標が空ならコピー
            if(attr.m_pArrays.size() == 0 || attr.m_pArrays[0] == nullptr)
            {
                continue;
            }
            if(m_ShapeInfo->m_uvHintIdxs[j] < lesserHint)
            {
                lesser = j;
                lesserHint = m_ShapeInfo->m_uvHintIdxs[lesser];
            }
        }

        // より低いUVヒントが有る場合はSwap
        if(lesser != i)
        {
            Dcc::RPrimVtx::VtxAttr texID = Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + lesser);
            Dcc::ROutShapeVtxInfo &attr = m_ShapeInfo->m_VtxInfos[texID];
            Dcc::ROutShapeVtxInfo temp(attr);

            attr.m_pArrays.clear();
            for(UINT k = 0; k < curAttrData.m_pArrays.size(); k++)
            {
                attr.m_pArrays.push_back(curAttrData.m_pArrays[k]);
            }

            curAttrData.m_pArrays.clear();
            for(UINT k = 0; k < temp.m_pArrays.size(); k++)
            {
                curAttrData.m_pArrays.push_back(temp.m_pArrays[k]);
            }

            swap(m_VtxAttrFlag[texID], m_VtxAttrFlag[curTexID]);
            swap(uvChOrders[i], uvChOrders[lesser]);
            swap(m_ShapeInfo->m_uvHintIdxs[i], m_ShapeInfo->m_uvHintIdxs[lesser]);
            swap(m_ShapeInfo->m_uvAttribNames[i], m_ShapeInfo->m_uvAttribNames[lesser]);
        }
    }

    // set material sampler hint
    if(fmat->m_Samplers.size() == fmat->m_SamplerReferUVs.size())
    {
        for(int i = 0; i < fmat->m_Samplers.size(); i++)
        {
            int cafeuv = fmat->m_SamplerReferUVs[i];
            int uvHint = 0;
            Dcc::RIntArray::iterator it = std::find(uvChOrders.begin(), uvChOrders.end(), cafeuv);
            if(it != uvChOrders.end())
            {
                unsigned int id = static_cast<unsigned int>(it - uvChOrders.begin());
                uvHint = m_ShapeInfo->m_uvHintIdxs[id];
            }
            else
            {
                std::stringstream ss;
                ss << "Can't find valid uv hint. " << m_OrgName << ": " << fmat->m_Samplers[i].m_TexName << ": " << i;
                RLogger::LogMessage(ss.str(), RLogger::kWarning);
            }
            fmat->m_Samplers[i].m_OriginalTexsrt.m_UvHintIdx = uvHint;
        }


        /*
        for(int i = 0; i < m_ShapeInfo->m_uvHintIdxs.size(); i++)
        {
            int id = m_ShapeInfo->m_uvHintIdxs[i];
            fmat->m_Samplers[i].m_OriginalTexsrt.m_UvHintIdx = id;
        }
        */
    }
    else
    {
        RLogger::LogMessage("SamplerCount is not match uvHint", RLogger::kWarning);
    }

}



/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility

/******************************************************************************
-------------------------------------------------------------------------------
                end of file
-------------------------------------------------------------------------------
******************************************************************************/
