﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityModel.h"
#include "DccUtilityNode.h"
#include "DccUtilityMaterial.h"
#include "DccUtilitySceneMaterials.h"
#include "DccUtilityLogger.h"

#include "DccShape.h"

using namespace std;
namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

//----------------------------------------------------------------------------
// コンストラクタ
RModel::RModel(
    const int mtxPalSize,
    const char* orgName,
    Dcc::RProgShape* progShapePtr)
    : mMtxPalSize(mtxPalSize)
    , mProgShapePtr(progShapePtr)
    , m_SmoothSkinningShapeCount(0)
    , m_RigidSkinningShapeCount(0)
{
    // デバッグ目的のシェイプ名を代入
    if(orgName)
    {
        mOrgName = orgName;
    }

    // 頂点属性が使われるかどうかのフラグをリセットしておく
    for(int i = 0; i < Dcc::RPrimVtx::VA_COUNT; i++)
    {
        mVtxAttrFlag[i] = false;
    }
}

//----------------------------------------------------------------------------
// デストラクタ
RModel::~RModel(void)
{
    // 作成されたRShapeインスタンスを削除する。
    ShapeDataList::iterator it = mShapeDataList.begin();
    while(it != mShapeDataList.end())
    {
        delete *it;
        ++it;
    }
}

//----------------------------------------------------------------------------
// 頂点属性が使われているかどうかのフラグを設定
void RModel::setVertexAttributeUsed(Dcc::RPrimVtx::VtxAttr attr, bool used)
{
    mVtxAttrFlag[attr] = used;
}

//----------------------------------------------------------------------------
// ポリゴンを追加
Dcc::RStatus RModel::appendPolygon(
    unsigned int materialID,
    const Dcc::RPrimitive& rprim,
    const Dcc::RVtxMtxArray& vtxMtxs,
    const bool forceTriangulate)
{
    // マテリアルIDに対応した RShape を取得もしくは作成
    FShape	*rshape = getRShape(materialID);

    // RShapeにポリゴンを追加する。
    Dcc::RStatus res = rshape->AppendPolygon(rprim, vtxMtxs, forceTriangulate, 0);

    return res;
}

//----------------------------------------------------------------------------
// マテリアルIDに対応した RShape を取得もしくは作成
FShape* RModel::getRShape(int materialID)
{
    // マテリアルIDに対応するRShapeがすでに作成済みか調べ、なければ新たに作成
    // する。
    FShape*	rshape = nullptr;
    ShapeDataList::iterator it = mShapeDataList.begin();

    while( it != mShapeDataList.end() )
    {
        if( (*it)->GetMaterialId() == materialID )
        {
            rshape = (*it);
            break;
        }
        ++it;
    }

    if( rshape == nullptr )
    {
        // 新たに作成
        rshape = new FShape(
            Dcc::RShape::RIGID,
            mMtxPalSize,
            "",	//	シェイプの名前は後で決定するため、ここではダミーの文字列を入れておく
            mProgShapePtr);
        rshape->SetMaterialId( materialID );
        mShapeDataList.push_back( rshape );

        // 頂点の属性をコピー
        for(int a = 0; a < Dcc::RPrimVtx::VA_COUNT; a++)
        {
            // ただし RPrimVtx::IDX と RPrimVtx::WGT は setVertexBoneRefArray()
            // で頂点ボーンウェイトを設定したかによって決まるので、ここでは必ず
            // falseになっている。
            rshape->m_VtxAttrFlag[a] = mVtxAttrFlag[a];
        }

        // マテリアル毎(RShape毎)に頂点属性データの複製を作成する。
        rshape->setVertexData(mShapeInfo);
    }
    else
    {
        // 作成済みのRShapeを参照
        rshape = *it;
    }

    return rshape;
}

//----------------------------------------------------------------------------
// スキニングに関するモードおよび座標系の変換処理を行う
void RModel::convertSkinningData(RNode* node, const std::vector<RNode*>&	NodeList, const Dcc::RVtxMtxArray& vtxMtxs)
{
    // モデルで参照するボーンが1つのみならばスキンを無効にして静的なモデルへ変換
    //if(convertOneBoneSkinningToStaticModel(node)) return;

    // 各頂点毎に参照するボーンが1つのみのスキニングモデルを、リジッドスキニングに変換
    //convertOneBoneSkinningToRigid(node);
    // 複数のボーンが影響するスキニングならば頂点座標をワールド座標系へ変換
    //convertSmoothSkinningToWorldSpace(node);


    ShapeDataList::const_iterator its = mShapeDataList.begin();
    while(its != mShapeDataList.end())
    {
        FShape &rshape = *(*its);

        switch(rshape.m_SkinningMode)
        {
            case	Dcc::RShape::NO_SKINNING:
            break;
            case	Dcc::RShape::RIGID:
            {
                std::list<int>	refVmtIndexList;

                rshape.GetRefVmtIndexList( refVmtIndexList );

                int maxBoneIndex = 0;
                std::list<int>::iterator	curIter = refVmtIndexList.begin();
                std::list<int>::iterator	endIter = refVmtIndexList.end();

                while( curIter != endIter )
                {
                    const Dcc::RVtxMtx& vmt = vtxMtxs[(*curIter)];
                    const int boneCount = vmt.GetBoneCount();

                    for(int w = 0; w < boneCount; w++)
                    {
                        // この後ボーン毎の作業用変数の配列を作るので、ボーンインデックス
                        // の最大値を求めておく。
                        int boneIndex = vmt.GetBoneIndex(w);
                        if(maxBoneIndex < boneIndex) maxBoneIndex = boneIndex;
                    }
                    ++curIter;
                }

#if	0
                //	TODO:	必要？
                // 各頂点のボーンへの参照を一つにするために、ウェイト値が0の参照をつぶす。
                curIter = refVmtIndexList.begin();
                endIter = refVmtIndexList.end();

                while( curIter != endIter )
                {
                    int s = 0;
                    RVtxBoneRef &boneRef = (*mVertexBoneRefs)[(*curIter)];
                    for(size_t w = 0; w < boneRef.size(); w++)
                    {
                        if(0 < boneRef[w].mWeight)
                        {
                            boneRef[s] = boneRef[w];
                            s++;
                        }
                    }
                    boneRef.resize(1);
                    curIter++;
                }
#endif

                // 頂点を一旦ワールド座標系へ変換し、それから各ボーンの座標系へ変換する。
                // 各ボーン毎にこの変換のための作業用変数(変換行列)を作成。
                // シェイプから参照されているボーンのインデックスの最大値になるが
                // 同じシェイプの変換に使用するので問題ない
                bool* makeMatrix = new bool [maxBoneIndex+1];
                memset(makeMatrix, 0, sizeof(bool)*(maxBoneIndex+1));
                Dcc::RMtx44* mtxToBone = new Dcc::RMtx44 [maxBoneIndex+1];

                //!!!! スキンモデルを持つノードでは、RNode::setBindMatrix() にバインドポーズ時
                //!!!! の姿勢行列が設定されていることが必要。
                const Dcc::RMtx44 mtxModelWorld = node->getBindMatrix(WorldSpace);

                // 参照しているボーンの変換だけ処理する
                curIter = refVmtIndexList.begin();
                endIter = refVmtIndexList.end();

                while( curIter != endIter )
                {
                    const Dcc::RVtxMtx& vmt = vtxMtxs[(*curIter)];
                    ++curIter;

                    // 各ボーンに付き一度だけ変換行列を計算
                    int boneIndex = vmt.GetBoneIndex( 0 );
                    if(makeMatrix[boneIndex]) continue;
                    makeMatrix[boneIndex] = true;

                    Dcc::RMtx44 mtxBone = NodeList[boneIndex]->getBindMatrix(WorldSpace);//wt.mBone->getBindMatrix(WorldSpace);
                    mtxToBone[boneIndex] = mtxModelWorld * mtxBone.Inverse();
                }

                // 各頂点の座標を変換する
                rshape.transformVertexDataToBone(mtxToBone, vtxMtxs);

                // 作業変数を削除して終わり
                delete [] makeMatrix;
                delete [] mtxToBone;
            }
            break;
            case	Dcc::RShape::SMOOTH:
            {
                const Dcc::RMtx44 mtxModelWorld = node->getBindMatrix(WorldSpace);
                // スキニングが有効な場合、頂点座標、法線ベクトルをワールド座標系へ
                // 変換する。
                rshape.transformVertexData(mtxModelWorld);
            }
            break;
        }
        ++its;
    }

}

//----------------------------------------------------------------------------
// データの整理を行う
// モデルの拡大縮小もここで行う
bool RModel::optimize( const Dcc::RVtxMtxArray& vtxMtxs,
                       const Dcc::RIntArray& smoothMtxIdxs,
                       const Dcc::RIntArray& rigidMtxIdxs,
                       const RSceneMaterials& sceneMaterials,
                       float magnify /*= 1.0f*/ )
{
    bool res = true;

    // データの整理
    ShapeDataList::const_iterator it = mShapeDataList.begin();
    while(it != mShapeDataList.end())
    {
        FShape* rshape = *it;

        // 使われていないUV座標を削除して前に詰める
        // このシェイプに割り当てられるマテリアル
        FMaterial* fmat = sceneMaterials.getFMaterialByIndex(rshape->GetMaterialId());
        if(fmat)
        {
            rshape->deleteUnusedUV(fmat);
        }

        // 同じ値を持つ頂点属性データをマージし、プリミティブのインデックス
        // 配列を修正する。
        if(!rshape->mergeSameVertexData())
        {
            return false;
        }

        // モデルの拡大縮小
        rshape->magnifyVertexData(magnify);

        // 参照されていない頂点属性データを削除する。
        rshape->deleteUnusedVertexData();

        // プリミティブセットの作成、スキニング関連のデータ作成を行う。
        const Dcc::RIntArray* pBoneMtxIdxs = (rshape->m_SkinningMode == Dcc::RShape::SMOOTH) ?
            &smoothMtxIdxs : &rigidMtxIdxs;
        if(rshape->SetAndOptimize( vtxMtxs, *pBoneMtxIdxs ) != Dcc::RStatus::SUCCESS)
        {
            res = false;
        }



        // すべてのキーシェイプで同じ頂点属性は削除
        rshape->deleteSameKeyShapes();

#if	0
        if(rshape->GetVtxSize() > 65535)
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_TheNumberOfVerticesExceedsTheLimitation, getOrgName());
            return false;
        }
#endif
        ++it;
    }

    return res;
}

//----------------------------------------------------------------------------
//	シェイプのスキニングモードを調べる。
void RModel::getShapeSkinningMode( const int NodeIndex, const Dcc::RVtxMtxArray& vtxMtxs, std::vector<RNode*>&	bones, const bool adjustsSmoothSkinning )
{
    // スムーススキニング強制フラグを設定します。
    bool forcesSmoothSkinning = false;
    if ( adjustsSmoothSkinning )
    {
        ShapeDataList::iterator it = mShapeDataList.begin();
        while( it != mShapeDataList.end() && forcesSmoothSkinning == false )
        {
            //	シェイプ中の全て頂点を調べる。
            Dcc::RPrimPtrArray::iterator	curPrimIter = (*it)->m_pPrimitives.begin();
            Dcc::RPrimPtrArray::iterator	endPrimIter = (*it)->m_pPrimitives.end();

            while( curPrimIter != endPrimIter && forcesSmoothSkinning == false )
            {
                Dcc::RPrimVtxArray::iterator	curVtxIter = (*curPrimIter)->m_Vtxs.begin();
                Dcc::RPrimVtxArray::iterator	endVtxIter = (*curPrimIter)->m_Vtxs.end();

                while( curVtxIter != endVtxIter && forcesSmoothSkinning == false )
                {
                    const Dcc::RVtxMtx&	vmt = vtxMtxs[(*curVtxIter).m_Mtx];
                    const int	localBoneCount = vmt.GetBoneCount();

                    // 頂点に影響するボーンが2個以上あれば
                    if ( localBoneCount >= 2 )
                    {
                        forcesSmoothSkinning = true;
                    }

                    ++curVtxIter;
                }

                ++curPrimIter;
            }

            ++it;
        }
    }

    // 作成されたRShapeインスタンスを削除する。
    ShapeDataList::iterator it = mShapeDataList.begin();
    while(it != mShapeDataList.end())
    {
        int	MaxBoneCountPerVtx = 0;
        int	MainBoneIndex = -1;
        bool	RigidSkinFlag = false;
        Dcc::RIntArray allBoneIdxs; // シェイプで使用されている全ボーンインデックスの配列（重複あり）です。

        //	TODO: RShape に実装したほうがいいと思う。

        //	シェイプ中の全て頂点を調べる。
        Dcc::RPrimPtrArray::iterator	curPrimIter = (*it)->m_pPrimitives.begin();
        Dcc::RPrimPtrArray::iterator	endPrimIter = (*it)->m_pPrimitives.end();

        while( curPrimIter != endPrimIter )
        {
            Dcc::RPrimVtxArray::iterator	curVtxIter = (*curPrimIter)->m_Vtxs.begin();
            Dcc::RPrimVtxArray::iterator	endVtxIter = (*curPrimIter)->m_Vtxs.end();

            while( curVtxIter != endVtxIter )
            {
                const Dcc::RVtxMtx&	vmt = vtxMtxs[(*curVtxIter).m_Mtx];
                const int	BoneCount = vmt.GetBoneCount();
                if( MaxBoneCountPerVtx < BoneCount )
                {
                    MaxBoneCountPerVtx = BoneCount;
                }

                // 影響するボーン数が 1 つの時にはリジッドスキンかスキン無しかの判定を行います。
                if( BoneCount == 1 )
                {
                    const int BoneIndex = vmt.GetBoneIndex(0);
                    if( MainBoneIndex == -1 )
                    {
                        MainBoneIndex = BoneIndex;
                    }
                    else if( MainBoneIndex != BoneIndex )
                    {
                        // 抽出したボーンと異なるボーンを使用する頂点があった場合は
                        // 1 つのシェイプ内で複数のボーンにより変形されるということなので
                        // スキン無しにはできません。そのためリジッドスキンのフラグを ON にします。
                        RigidSkinFlag = true;
                    }
                }

                // 全ボーンインデックスの配列に追加します。
                for (int iLocalBone = 0; iLocalBone < BoneCount; ++iLocalBone)
                {
                    const int boneIdx = vmt.GetBoneIndex(iLocalBone);
                    allBoneIdxs.push_back(boneIdx);
                }

                ++curVtxIter;
            }

            ++curPrimIter;
        }

        // シェイプ内で 1 つのボーンだけ使用する場合でも
        // シェイプの属するボーンでなければリジッドスキンとします。
        if (MainBoneIndex != -1 &&
            MainBoneIndex != NodeIndex)
        {
            RigidSkinFlag = true;
        }

        //-----------------------------------------------------------------------------
        // スキニングモードを決定します。
        if( MaxBoneCountPerVtx >= 2 || ( forcesSmoothSkinning && RigidSkinFlag ) )
        {
            // 1 頂点につき 2 つ以上のボーンを使用する場合はスムーススキンです。
            (*it)->m_SkinningMode = Dcc::RShape::SMOOTH;
            ++m_SmoothSkinningShapeCount;
        }
        else if( RigidSkinFlag )
        {
            // 1 頂点について 1 つしかボーンを使用せず、
            // シェイプ内で 2 つ以上のボーンを使用する場合リジッドスキンです。
            (*it)->m_SkinningMode = Dcc::RShape::RIGID;
            ++m_RigidSkinningShapeCount;
        }
        else
        {
            // それ以外の場合はスキン無しです。
            (*it)->m_SkinningMode = Dcc::RShape::NO_SKINNING;
        }

        //-----------------------------------------------------------------------------
        // スキニングに使用するボーンの行列パレットインデックスに 0 を設定します。
        // 実際の行列パレットインデックスは後で設定します。
        const int allBoneCount = static_cast<int>(allBoneIdxs.size());
        if ((*it)->m_SkinningMode == Dcc::RShape::RIGID)
        {
            for (int iAllBone = 0; iAllBone < allBoneCount; ++iAllBone)
            {
                bones[allBoneIdxs[iAllBone]]->m_RigidMtxIdx = 0;
            }
        }
        else if ((*it)->m_SkinningMode == Dcc::RShape::SMOOTH)
        {
            for (int iAllBone = 0; iAllBone < allBoneCount; ++iAllBone)
            {
                bones[allBoneIdxs[iAllBone]]->m_SmoothMtxIdx = 0;
            }
        }

        //	スキニングモードに基づいて頂点データ使用フラグを設定する。
        setVertexAttributeUsed(Dcc::RPrimVtx::IDX0, ((*it)->m_SkinningMode != Dcc::RShape::NO_SKINNING));
        setVertexAttributeUsed(Dcc::RPrimVtx::WGT0, ((*it)->m_SkinningMode == Dcc::RShape::SMOOTH));
        (*it)->m_VtxAttrFlag[Dcc::RPrimVtx::IDX0] = mVtxAttrFlag[Dcc::RPrimVtx::IDX0];
        (*it)->m_VtxAttrFlag[Dcc::RPrimVtx::WGT0] = mVtxAttrFlag[Dcc::RPrimVtx::WGT0];

        ++it;
    }
}

//----------------------------------------------------------------------------
// 頂点属性が使われているかどうかのフラグを頂点属性データから設定
void RModel::setVtxAttrUsedFromShapeInfo(void)
{
    // 頂点座標は常にtrue
    setVertexAttributeUsed(Dcc::RPrimVtx::POS0, true);
    // 法線座標が有ればtrue
    setVertexAttributeUsed(Dcc::RPrimVtx::NRM0,
        (mShapeInfo.m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays.size() > 0)
        && (mShapeInfo.m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays[0] != nullptr)
        );
    // 頂点カラーが有ればtrue
    for(int i = Dcc::RPrimVtx::COL0; i <= Dcc::RPrimVtx::COL7; i++)
    {
        setVertexAttributeUsed(Dcc::RPrimVtx::VtxAttr(i),
            (mShapeInfo.m_VtxInfos[i].m_pArrays.size() > 0)
            && (mShapeInfo.m_VtxInfos[i].m_pArrays[0] != nullptr)
            );
    }
    // UV座標が有ればtrue
    for(int i = Dcc::RPrimVtx::TEX0; i <= Dcc::RPrimVtx::TEX7; i++)
    {
        setVertexAttributeUsed(Dcc::RPrimVtx::VtxAttr(i),
            (mShapeInfo.m_VtxInfos[i].m_pArrays.size() > 0)
            && (mShapeInfo.m_VtxInfos[i].m_pArrays[0] != nullptr)
            );
    }
    // ユーザー定義頂点データが有ればtrue
    setVertexAttributeUsed(Dcc::RPrimVtx::USR0,
        (mShapeInfo.m_VtxInfos[Dcc::RPrimVtx::USR0].m_pArrays.size() > 0)
        && (mShapeInfo.m_VtxInfos[Dcc::RPrimVtx::USR0].m_pArrays[0] != nullptr)
        );
}

//----------------------------------------------------------------------------
// シェイプと頂点情報から法線マップ用のタンジェントを計算する。
Dcc::RStatus RModel::calcTangent( Dcc::RPrimVtx::VtxAttr texCh, bool isUVBackward /*= false*/ )
{
    ShapeDataList::const_iterator its = mShapeDataList.begin();
    while(its != mShapeDataList.end())
    {
        FShape &rshape = *(*its);
        ++its;

        Dcc::RStatus stat = rshape.calcTangent(texCh, isUVBackward);
        if(stat != Dcc::RStatus::SUCCESS)
        {
            return Dcc::RStatus::FAILURE;
        }
        else if(stat.GetMessage().length() > 0)
        {
            // 長さが 0 の接線または従法線が存在
            RLogger::LogMessagebyID(RLogger::kLogWRN_ZeroTangentExists, getOrgName());
        }
    }

    return Dcc::RStatus::SUCCESS;
}

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility

/******************************************************************************
-------------------------------------------------------------------------------
                end of file
-------------------------------------------------------------------------------
******************************************************************************/
