﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCamera.h"
#include "DccUtilityShape.h"

namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

float RCamera::mMagnify = 1.0f;

//----------------------------------------------------------------------------
// コンストラクタです。
RCamera::RCamera()
: mIsBranchVisible(true)
{
}

//----------------------------------------------------------------------------
//	カメラアニメーションの出力準備をします。
void RCamera::PrepareAnimations( const Dcc::RExpOpt& rOpt )
{
    /*
    const float baseValues[RCamera::kAttrMax] =
    {
        (float)RRoundToZero(mScale.x),
        (float)RRoundToZero(mScale.y),
        (float)RRoundToZero(mScale.z),
        (float)RRoundToZero(mRotate.x),
        (float)RRoundToZero(mRotate.y),
        (float)RRoundToZero(mRotate.z),
        (float)RRoundToZero(mTranslate.x),
        (float)RRoundToZero(mTranslate.y),
        (float)RRoundToZero(mTranslate.z),
        (float)RRoundToZero(mTargetPosition.x),
        (float)RRoundToZero(mTargetPosition.y),
        (float)RRoundToZero(mTargetPosition.z),
        (float)RRoundToZero(mUpwardVector.x),
        (float)RRoundToZero(mUpwardVector.y),
        (float)RRoundToZero(mUpwardVector.z),
        (float)RRoundToZero(mTwist),
        (float)RRoundToZero(mNearClip),
        (float)RRoundToZero(mFarClip),
        (float)RRoundToZero(mAspect),
        (float)RRoundToZero(mFovy),
        (float)RRoundToZero(mOrthoHeight)
    };
    */



    //=============================================================================
    //	AnalyzeCameraFullAnim
    //=============================================================================

    //-----------------------------------------------------------------------------
    // make rotate continuous
    Dcc::RMakeContinuousZxyAngleArray( // ZXY であることに注意！
        m_Anims[Dcc::RCamera::ROTATE_X].m_FullValues,
        m_Anims[Dcc::RCamera::ROTATE_Y].m_FullValues,
        m_Anims[Dcc::RCamera::ROTATE_Z].m_FullValues);


    for (int iattr = 0; iattr < Dcc::RCamera::PARAM_COUNT; ++iattr)
    {
        //-----------------------------------------------------------------------------
        // set curve name & loop flag & angle flag
        Dcc::RAnimCurve& curve = m_Anims[iattr];

        curve.m_Name = m_Name + "." + Dcc::RCamera::GetParamName(iattr);
        curve.m_LoopFlag = rOpt.m_LoopAnim;
        curve.m_AngleFlag = Dcc::RCamera::IsRotate(iattr);

        //-----------------------------------------------------------------------------
        // set tolerance
        if (Dcc::RCamera::IsRotate(iattr))
        {
            curve.m_Tolerance = rOpt.m_TolR;
        }
        else if (Dcc::RCamera::IsTranslate(iattr))
        {
            // mTolT は Magnify を掛ける前の値に対する許容値なので
            // Magnify を掛ける
            curve.m_Tolerance = static_cast<float>(rOpt.m_TolT * mMagnify);
        }
        else // Aspect
        {
            curve.m_Tolerance = Dcc::R_MAKE_KEY_TOL_CAMERA_OTHER;
        }
        //curve.UpdateConstantFlag();
    }

    //=============================================================================
    //	GetCameraKeyAnim
    //=============================================================================
    for (int iattr = 0; iattr < Dcc::RCamera::PARAM_COUNT; ++iattr)
    {
        //-----------------------------------------------------------------------------
        // get channel input
        Dcc::RAnimCurve& curve = m_Anims[iattr];

        //if( curve.mUseFlag )
        if ( curve.m_Keys.empty() )
        {
            //-----------------------------------------------------------------------------
            // make key (camera)
            //#ifdef ADJUST_CAM_ROT_ZXY_SW
            // rotate は３軸同時に補正しているので単一軸の補正は不要
            bool continuousAngleFlag = ( iattr == Dcc::RCamera::TWIST
                                        || iattr == Dcc::RCamera::PERSP_FOVY);
            //#else
            //bool continuousAngleFlag = true;
            //#endif
            curve.MakeKeys(GetFloatFrameFromSubFrame4f, nullptr, continuousAngleFlag);

            curve.UpdateConstantFlag();
        }

        curve.UpdateUseFlag( curve.m_Tolerance );
    }
}

//-----------------------------------------------------------------------------
//! @brief <camera_anim> 要素を出力します。
//-----------------------------------------------------------------------------
void RCamera::OutAnim(
                      std::ostream& os,
                      Dcc::RDataStreamArray& dataStreams,
                      const int tabCount,
                      const int index,
                      const Dcc::RExpOpt& rOpt) const
{
    static const char* const rotateModeStrs[] = { "aim", "euler_zxy" };
    static const char* const projectionModeStrs[] = { "ortho", "persp" };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << Dcc::RTab(tc) << "<camera_anim"
        << " index=\"" << index
        << "\" camera_name=\"" << Dcc::RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << Dcc::RTab(tc + 1) << "frame_count=\"" << rOpt.m_FrameCount
        << "\" loop=\"" << Dcc::RBoolStr(rOpt.m_LoopAnim) << "\"" << R_ENDL;
    os << Dcc::RTab(tc + 1) << "rotate_mode=\"" << rotateModeStrs[GetRotateMode()] << "\"" << R_ENDL;
    os << Dcc::RTab(tc + 1) << "projection_mode=\"" << projectionModeStrs[GetProjectionMode()] << "\"" << R_ENDL;
    os << Dcc::RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // camera anim targets
    for (int iParam = 0; iParam < Dcc::RCamera::PARAM_COUNT; ++iParam)
    {
        if (GetRotateMode() == Dcc::RCamera::ROTATE_AIM)
        {
            if (Dcc::RCamera::ROTATE_X <= iParam && iParam <= Dcc::RCamera::ROTATE_Z) continue;
        }
        else // ROTATE_EULER_ZXY
        {
            if (Dcc::RCamera::AIM_X <= iParam && iParam <= Dcc::RCamera::AIM_Z) continue;
            if (iParam == Dcc::RCamera::TWIST) continue;
        }

        if (GetProjectionMode() == Dcc::RCamera::ORTHO)
        {
            if (iParam == Dcc::RCamera::PERSP_FOVY) continue;
        }
        else // PERSP
        {
            if (iParam == Dcc::RCamera::ORTHO_HEIGHT) continue;
        }

        const Dcc::RAnimCurve& curve = m_Anims[iParam];
        //if( curve.m_UseFlag )
        {
            curve.Out(os, dataStreams, tc + 1, "camera_anim_target",
                Dcc::RCamera::GetParamName(iParam), false);
        }
    }

    //-----------------------------------------------------------------------------
    // end
    os << Dcc::RTab(tc) << "</camera_anim>" << R_ENDL;
}

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
