﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NintendoExportMax.h"
#include "MaterialInfo.h"

#define ANISOSHADER_CLASS_ID     0x2857f460

namespace Dcc = nn::gfx::tool::dcc;
using namespace nn::gfx::tool::dcc::utility;

int MtlInfo::m_AnimStart_ = 0;
int MtlInfo::m_AnimEnd_ = 0;
int MtlInfo::m_SubStart_ = 0;
int MtlInfo::m_SubEnd_ = -1;
int MtlInfo::m_SubStep_ = 1;
bool MtlInfo::exportTexture = false;


bool MtlInfo::Init(Mtl * mtl)
{
    MtlInfo();
    if(!mtl)
    {
        return true;
    }

    m_pMaxMtl = mtl;

    // 各種カラーを取得
    m_Diffuse = mtl->GetDiffuse();
    m_Ambient = mtl->GetAmbient();
    m_Specular = mtl->GetSpecular();
    m_Shiness = mtl->GetShininess();
    m_ShineLevel = mtl->GetShinStr();

    //名前を取得
    string orgName = M_2_A(mtl->GetName());
    m_Name = Dcc::RAdjustElementNameString(orgName);
    if(m_Name != orgName)
    {
        RLogger::LogMessagebyID(RLogger::kLogWRN_MaterialNameIsChanged, orgName.data());
    }

    // 自己照明を取得。
    // 自己照明がカラーでない場合は値からグレイスケールカラーを算出する
    if(mtl->GetSelfIllumColorOn())
    {
        m_Emittion = mtl->GetSelfIllumColor();
    }
    else
    {
        float emitVal = mtl->GetSelfIllum();
        m_Emittion= Color(emitVal, emitVal, emitVal);
    }

    // 透明度を取得
    m_Transparency = 1.0f - mtl->GetXParency();

    // 異方性反射の場合、シェーダのパラメータを取得
    Class_ID mtlcid = mtl->ClassID();
    bool hasShader = mtl->SupportsShaders() == TRUE;
    StdMat2* smtl = NULL;
    Shader* sh = NULL;

    if (mtl->ClassID() == Class_ID(DMTL_CLASS_ID,0)
        && mtl->SupportsShaders() )
    {
        //StdMat2* smtl = static_cast<StdMat2*>(mtl);
        smtl = (StdMat2*)mtl;
        sh = smtl->GetShader();
        if(sh)
        {
            if(sh->ClassID() ==  Class_ID(ANISOSHADER_CLASS_ID,0))
            {
                // 異方性シェーダはヘッダがないのでパラメータブロックから取得
                IParamBlock2* pblock = sh->GetParamBlock(0);
                float aniso = pblock->GetFloat(8); //異方性反射
                float dir = pblock->GetFloat(9); // 方向
                m_IsAnisotropic = true;
            }
        }
    }
    else
    {
        RLogger::LogMessagebyID(RLogger::kLogWRN_NonSupportedShader, m_Name.data());
    }

    bool ret;

    // マテリアルの全てのテクスチャを列挙する
    for(int i = 0; i < mtl->NumSubTexmaps(); i++)
    {
        if(!mtl->SubTexmapOn(i)) continue;
        Texmap* tex = mtl->GetSubTexmap(i);
        // マップチャンネルの判定
        int ch = ID_DI;
        if(smtl)
        {
            if		(i == smtl->StdIDToChannel(ID_DI)) ch = ID_DI;
            else if	(i == smtl->StdIDToChannel(ID_SP)) ch = ID_SP;
            else if	(i == smtl->StdIDToChannel(ID_OP)) ch = ID_OP;
            else if (i == smtl->StdIDToChannel(ID_BU)) ch = ID_BU;
            else if	(i == smtl->StdIDToChannel(ID_RL)) ch = ID_RL;
        }

        if(!tex) continue;
        if(tex->ClassID() == Class_ID(BMTEX_CLASS_ID,0) ) 		// 通常のビットマップ
        {
            ret = AddBitmap(static_cast<BitmapTex*>(tex), ch);
            if(!ret) return false;
        }
        else if( smtl && (i == smtl->StdIDToChannel(ID_BU))
                    && (tex->ClassID() == GNORMAL_CLASS_ID)) // 法線マップ
        {
            Texmap* smap = tex->GetSubTexmap(0);
            if(smap && smap->ClassID() == Class_ID(BMTEX_CLASS_ID,0))
            {
                ret = AddBitmap(static_cast<BitmapTex*>(smap), ch);
                if(!ret) return false;
                m_HasBump = true;
            }
        }
        else if(tex->ClassID() == Class_ID(COMPOSITE_CLASS_ID,0) ) // 合成マップ
        {
            // max2009以降では、1つのレイヤーにカラーテクスチャとマスクテクスチャが含まれる。
            // それを考慮してサブテクスチャを出力する
            int layerMapNum = (MAX_VERSION_MAJOR >= 11)? 2 : 1;
            int numSubmap = tex->NumSubTexmaps();
            // Mayaに合わせるため、レイヤーを逆順で出力する
            //for(int j = 0; j < numSubmap; j += layerMapNum)
            for(int j = numSubmap - layerMapNum; j >= 0; j -= layerMapNum)
            {
                Texmap* smap = tex->GetSubTexmap(j);
                if(smap && smap->ClassID() == Class_ID(BMTEX_CLASS_ID,0))
                {
                    ret = AddBitmap(static_cast<BitmapTex*>(smap), ch);
                    if(!ret) return false;
                }
            }
        }
        else
        {
            //utility::RLogger::g
            RLogger::LogMessagebyID(RLogger::kLogWRN_NonSupportedTex, std::string(M_2_A(tex->GetName().data())));
            //mprintf("WARNING: Unsupported Texmap(%s) is used in material(%s)\n", tex->GetName().data(), mtl->GetName().data());
            continue;
        }
        // 透明度にマップが貼られていた場合フラグを立てる
        if(i == ID_OP)
        {
            m_HasBitmapAlpha = true;
        }
    }
    return true;
}

// デストラクタ
MtlInfo::~MtlInfo()
{
/*
std::vector<MaxBitmapInfo>::iterator it;
    it = m_BitmapArray.begin();
    while(it != m_BitmapArray.end())
    {
        if(it->rImage)
        {
            delete it->rImage;
            it->rImage = NULL;
        }
        it++;
    }

    for(int i = 0; i < 3; i++)
    {
        it = animMapArray[i].begin();
        while(it != animMapArray[i].end())
        {
            if(it->rImage)
            {
                delete it->rImage;
                it->rImage = NULL;
            }
            it++;
        }
    }
*/
}

// バンプマップで使用するテクスチャ、UV座標を決定する
void MtlInfo::DecideBumpTexInfo(vector<int> mapCh)
{
    m_BumpTexIdx = -1;
    m_BumpTexCh = -1;

    // 実際に登録されたテクスチャを列挙する
    int numBitmap = GetNumBitmap();
    for(int j = 0; j < numBitmap; j++)
    {
        if(m_BumpTexIdx < 0 && GetBitmap(j)->isBump)
        {
            m_BumpTexIdx = j;
            StdUVGen* uv = GetBitmap(j)->bmtex->GetUVGen();

            for(UINT ch = 0; ch < mapCh.size(); ch++)
            {
                if(mapCh[ch] < 1) continue;

                if(mapCh[ch] == uv->GetMapChannel())
                {
                    m_BumpTexCh = mapCh[ch]; //maxのマップチャンネル
                    break;
                }
            }
        }
    }
}

static bool validTexSize(unsigned int n)
{
    if (n < Dcc::RImage::WidthHeightMin || n > Dcc::RImage::WidthHeightMax)
    {
        return false;
    }
    return true;
}

#if !defined( NW4F_C3T_DELETE )
// BitmapInfoで指定されたテクスチャからRImageを作成
bool MtlInfo::CreateRImage(MaxBitmapInfo& bmi)
{
    bool ret = true;
    BMMRES status;
    Bitmap* bmap;
    BitmapInfo bi;
    bi.SetName(A_2_M(bmi.filename.data()));
    //bi.SetName(WStr::FromCStr(bmi.filename));
    bmap = TheManager->Load(&bi, &status);

    if (bmap && status == BMMRES_SUCCESS)
    {
        int w = bmap->Width();
        int h = bmap->Height();

        /*
        if(!validTexSize(w) || !validTexSize(h))
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_TextureSize, bmi.filename.data());
            return false;
        }
        */

        const int bitmapByteSize = w * h * 4; // W * H * RGBA
        uchar* rawmap = new uchar[bitmapByteSize];
        BMM_Color_64* scanline = new BMM_Color_64[w];
        uchar* pix;
        bool alphaFlag = false;
        for(int y = 0; y < h; y++)
        {
            bmap->GetPixels(0, y, w, scanline);
            pix = rawmap + y * w * 4;
            for(int x = 0; x < w; x++)
            {
                pix[0] = scanline[x].r >> 8;
                pix[1] = scanline[x].g >> 8;
                pix[2] = scanline[x].b >> 8;
                pix[3] = scanline[x].a >> 8;
                alphaFlag = alphaFlag || pix[3] != 0xFF;
                pix += 4;
            }
        }
        RImage* ri = new RImage();
        ri->SetFilePath(string(bmi.filename.data()));
        ri->SetFullBitmap(rawmap, w, h, alphaFlag);
        bmap->DeleteThis();
        delete[] scanline;
        delete[] rawmap;
        bmi.rImage = ri;
    }
    else
    {
        switch (status)
        {
        case BMMRES_FILENOTFOUND:
            RLogger::LogMessagebyID(RLogger::kLogWRN_TexFileIsNotFound, bmi.filename.data());
            break;
        case BMMRES_INVALIDFORMAT:
            RLogger::LogMessagebyID(RLogger::kLogERR_TGA_File, bmi.filename.data());
            ret = false;
            break;
        default:
            RLogger::LogMessagebyID(RLogger::kLogERR_InternalError, bmi.filename.data());
            ret = false;
            break;
        }
    }
    return ret;
}
#endif
static bool isNumber(TCHAR c)
{
    return (c >= '0' && c <= '9');
}

typedef std::map<int, int> frameFileMap;

// IFLファイルをパースして、テクスチャライブラリとサンプラーに登録する。
bool MtlInfo::parseIFL(CStr &fn, MaxBitmapInfo* bmiParent, RSceneMaterials &sceneMtls, Dcc::RSampler& sampler)
{
    // IFLを開く
    FILE* file = fopen(fn.data(), "rt");
    if(!file) return false;

    int curSwizzle = 0;
    char	line[4096];
    int		frame = 0;
    int		step;
    int		count = 0;
    string	texname = "";

    TexPatAnim texPatAnim;
    texPatAnim.m_ObjName = "texpat_dummy";

    std::map<std::string, int>	fileNameToLocalIndexMap;
    frameFileMap frameFiles;
    int	globalIndex = 0;

    // 一行ずつ読んでいく
    while(fgets(line, 4096, file))
    {
        if (line[0] == ';' || line[0] == '#') continue;

        //改行を削除
        char* ptr = strrchr(line, '\r');
        if (ptr) *ptr = 0;
        ptr = strrchr(line, '\n');
        if (ptr) *ptr = 0;

        //ファイルネームとフレーム数を取得
        int l = (int)strlen(line);
        l--;
        // 後ろのスペースを削除
        while(line[l] <= ' ' && l >= 0)
        {
            l--;
        }
        if(l <= 0) continue;

        // フレーム数を読み取る
        if (isNumber(line[l]))
        {
            while (isNumber(line[--l]) && l);
            sscanf(&line[l],"%d",&step);
            // ファイル名のみになるようNULLを入れる
            line[l] = 0;
        }
        else
        {
            step = 1;
        }

        char imageFn[1024];

        // パスを解決する
        TCHAR iflPath[MAX_PATH];
        TCHAR texPath[MAX_PATH];
        TCHAR texFile[MAX_PATH];
        TCHAR texExt[MAX_PATH];

        BMMSplitFilename(A_2_M(line),texPath,texFile,texExt);
        BMMSplitFilename(A_2_M(fn.data()),iflPath,NULL,NULL);
        BMMAppendSlash(iflPath);

        // テクスチャファイル名にパスがなければIFLファイルのパスを使う
        if (texPath[0])
        {
            sprintf_s(imageFn, "%s%s%s", M_2_A(texPath),M_2_A(texFile),M_2_A(texExt));
        }
        else
        {
            sprintf_s(imageFn, "%s%s%s", M_2_A(iflPath),M_2_A(texFile),M_2_A(texExt));
        }
        // ファイルが存在するかどうか
        if(DoesFileExist(A_2_M(imageFn)))
        {
            MaxBitmapInfo bmi;
            bmi.bmtex = bmiParent->bmtex;
            bmi.filename = imageFn;
            bmi.name = M_2_A(texFile);
            bmi.mapch = bmiParent->mapch;
            bmi.isBump = bmiParent->isBump;
            bmi.isAlpha = bmiParent->isAlpha;
            bmi.isIFL = false;
#if !defined( NW4F_C3T_DELETE )
            bmi.rImage = NULL;
#endif

            texname = bmi.filename.data();

            // テクスチャライブラリに追加
            std::map<std::string, int>::iterator	it = fileNameToLocalIndexMap.find( bmi.filename.data() );

            if( it == fileNameToLocalIndexMap.end() )
            {
                //	参照渡しの使用しない引数のためのダミー
                int	iTexImg;
                //	GetTexImage の utility 実装
                Dcc::RStringArray	filePaths;
                filePaths.push_back(bmi.filename.data());
                sceneMtls.AddTextureFileForExport( iTexImg, filePaths, sampler, curSwizzle, GetNintendoOptions()->m_Opt );
                texPatAnim.m_TexImgIdxs.push_back( iTexImg );

                curSwizzle++;
                globalIndex = iTexImg;
                fileNameToLocalIndexMap[bmi.filename.data()] = globalIndex;
            }
            else
            {
                globalIndex = it->second;
            }

            frameFiles[frame] = globalIndex;

            //if(step > 1) frameFiles[frame + step - 1] = texname;

            count++;
        }
        frame += step;
    }

    fclose(file);


    // 一枚以上登録され、最後のファイルに長さの指定がある場合は終わりにもキーを付ける
    if(count > 1 && step > 1)
    {
        frameFiles[frame - 1] = globalIndex;
    }


    //	m_FullValues へのデータ設定について
    //	Maya 版では m_FullValues へキーをすべて追加してからキーを作成している。
    //	3ds Max 版では m_FullValues は使用せずにアニメーションキーを直接追加しているため
    //	通常、m_FullValues はからになっている。
    //	だが RAnimCurve は m_FullValues には必ず値が入っていることを前提としているため
    //	何かしらのデータを設定する必要がある。
    //	初めてアニメーションキーを追加するタイミングで同じデータを m_FullValues へ設定する。

    if(count == 1)
    {
        texPatAnim.m_ImgAnim.m_Keys.push_back(Dcc::RAnimKey(0.0f, static_cast<float>(globalIndex), Dcc::RAnimKey::STEP));
        if( texPatAnim.m_ImgAnim.m_FullValues.size() == 0 )
        {
            texPatAnim.m_ImgAnim.m_FullValues.push_back( static_cast<float>(globalIndex) );
        }
    }
    else if(count > 1)
    {
        // ループ設定を考慮してアニメーションを付ける
        TimeValue playStart = bmiParent->bmtex->GetStartTime();
        float playRate = bmiParent->bmtex->GetPlaybackRate();
        int playLoop = bmiParent->bmtex->GetEndCondition();
        int lastFrame = frame - 1;
        string texname;
        int lastKey = -999;

        for(int f = m_AnimStart_; f <= m_AnimEnd_; f++)
        {
            float difFrame = (float)(f - m_AnimStart_);
            float localFramef = (f - (float)playStart / (float)GetTicksPerFrame()) * playRate;
            int l = (int)floorf(localFramef + 0.5f);

            switch(playLoop)
            {
                case END_LOOP:
                    if(l < 0)
                    {
                        l += ((-l / frame) + 1) * frame;
                        assert(l >= 0);
                    }
                    l = l % frame;
                    break;
                case END_PINGPONG:
                    if(l < 0)
                    {
                        l += ((-l / (2*frame)) + 1) * frame * 2;
                        assert(l >= 0);
                    }
                    l = l % (frame * 2);
                    if(l >= frame)
                    {
                        l = 2 * frame - 1 - l;
                    }
                    break;
                case END_HOLD:
                default:
                    if(l < 0) l = 0;
                    else if(l > lastFrame) l = lastFrame;
                    break;
            }

            frameFileMap::iterator it;
            frameFileMap::iterator lastit = frameFiles.end();
            bool foundKey = false;

            for(it = frameFiles.begin(); it != frameFiles.end(); it++)
            {
                if(it->first > l)
                {
                    if(lastit != frameFiles.end())
                    {
                        if(lastKey != lastit->first)
                        {
                            texPatAnim.m_ImgAnim.m_Keys.push_back(Dcc::RAnimKey(difFrame, static_cast<float>(lastit->second), Dcc::RAnimKey::STEP));
                            if( texPatAnim.m_ImgAnim.m_FullValues.size() == 0 )
                            {
                                texPatAnim.m_ImgAnim.m_FullValues.push_back( static_cast<float>(lastit->second) );
                            }
                        }
                        lastKey = lastit->first;
                        foundKey = true;
                        break;
                    }
                    else
                    {
                        if(lastKey != it->first)
                        {
                            texPatAnim.m_ImgAnim.m_Keys.push_back(Dcc::RAnimKey(difFrame, static_cast<float>(it->second), Dcc::RAnimKey::STEP));
                            if( texPatAnim.m_ImgAnim.m_FullValues.size() == 0 )
                            {
                                texPatAnim.m_ImgAnim.m_FullValues.push_back( static_cast<float>(it->second) );
                            }
                        }
                        lastKey = it->first;
                        foundKey = true;
                        break;
                    }
                }
                lastit = it;
            }

            if(!foundKey)
            {
                if(lastKey != frameFiles.rbegin()->first)
                {
                    //	Maya 版では m_FullValues へキーをすべて追加してからキーを作成している。
                    //	3ds Max 版では m_FullValues は使用せずにアニメーションキーを直接追加しているため
                    //	通常、m_FullValues はからになっている。
                    //	だが RAnimCurve は m_FullValues には必ず値が入っていることを前提としているため
                    //	何かしらのデータを設定する必要がある。
                    //	初めてアニメーションキーを追加するタイミングで同じデータを m_FullValues へ設定する。
                    texPatAnim.m_ImgAnim.m_Keys.push_back(Dcc::RAnimKey(difFrame, static_cast<float>(frameFiles.rbegin()->second), Dcc::RAnimKey::STEP));
                    if( texPatAnim.m_ImgAnim.m_FullValues.size() == 0 )
                    {
                        texPatAnim.m_ImgAnim.m_FullValues.push_back( static_cast<float>(frameFiles.rbegin()->second) );
                    }
                }
                lastKey = frameFiles.rbegin()->first;
            }
        }
    }

    //	IFL ファイルで一番最初に出現したキーの値を sampler へデフォルトのテクスチャとして設定する。
    if( fileNameToLocalIndexMap.size() > 0 )
    {
        std::string	szFileName = Dcc::RGetNoExtensionFilePath( Dcc::RGetFileNameFromFilePath( fileNameToLocalIndexMap.begin()->first ) );
        sampler.m_TexName = szFileName;
        sampler.m_TexPath = std::string(Dcc::RExpOpt::TexFolderName) + "/" + szFileName + ".ftx";
    }

    //	アニメーションカーブが Constant かどうかを判定し、カーブのパラメータを設定する
    const int keyCount = static_cast<int>(texPatAnim.m_ImgAnim.m_Keys.size());
    if( keyCount > 0 )
    {
        texPatAnim.m_ImgAnim.m_ConstantFlag = true;
        float	fConstCheckValue = texPatAnim.m_ImgAnim.m_Keys[0].m_Value;
        for (int ikey = 1; ikey < keyCount; ++ikey)
        {
            Dcc::RAnimKey& key = texPatAnim.m_ImgAnim.m_Keys[ikey];
            if( fConstCheckValue != key.m_Value )
            {
                texPatAnim.m_ImgAnim.m_ConstantFlag = false;
                break;
            }
        }
    }

    sampler.m_TexPatAnimIndex = static_cast<int>(sceneMtls.getNumTexPatAnimData());
    sceneMtls.addTexPatAnimData(texPatAnim);

    return true;
}

bool MtlInfo::AddBitmap(BitmapTex* bmtex, int id)
{
    if(!bmtex) return false;
    // ビットマップの枚数制限を超えた場合の処理
#if	0
    if(bitmapArray.size() >= R_TEXTURE_MAX)
#else
    //	TODO:	定数に置き換える
    if(m_BitmapArray.size() >= 256 )
#endif
    {
        RLogger::LogMessagebyID(RLogger::kLogWRN_NumberOfTexture, std::string(M_2_A(bmtex->GetName().data())));
        //mprintf("WARNING: Too many maps(%s) are used in material(%s)\n", bmtex->GetName().data(), name.c_str());

        // ビットマップがバンプマップの場合は登録されたビットマップを
        // 一つ削除して、優先的に登録する
        if(id == ID_BU)
        {
            m_BitmapArray.pop_back();
        }
        else
        {
            // 登録はされないが、警告のみなのでtrueを返す
            return true;
        }
    }

    IPathConfigMgr* iPath = IPathConfigMgr::GetPathConfigMgr();
    TSTR fn, ffn;
#if (MAX_VERSION_MAJOR < 12) // max2009まで
    fn = bmtex->GetMapName();
    ffn = iPath->GetFullFilePath(fn, false);
#else
    IFileResolutionManager* iFile = IFileResolutionManager::GetInstance();
    ffn = iFile->GetFullFilePath(bmtex->GetMapName(), MaxSDK::AssetManagement::kBitmapAsset);
#endif

    MaxSDK::Util::Path path(ffn), outPath;
    if(iPath->DoesFileExist(path))
    {
        const TCHAR* p = path.GetCStr();
        TCHAR path[2048], name[2048], ext[2048];
        BMMSplitFilename(ffn, path, name, ext);
        // 拡張子のチェック
        // exportTexがfalseの場合、拡張子が違っても登録する。
        //if(!exportTex || _strcmpi(ext, ".tga") == 0)
        {
            StdUVGen* uv = bmtex->GetUVGen();
            int ch = uv->GetMapChannel();

            // 同じファイルが透明・不透明のビットマップで使用された場合は一つのファイルのみ出力する。
            for(UINT i = 0; i < m_BitmapArray.size(); i++)
            {
                if( (m_BitmapArray[i].name == CStr(M_2_A(name))) &&
                    (m_BitmapArray[i].mapch == ch) )
                {
                    if(m_BitmapArray[i].isAlpha != (id == ID_OP))
                    {
                        // 登録はされないが、すでに同じファイルが登録されているのでtrueを返す
                        return true;
                    }
                }
            }

            {
                Bitmap* bmap = bmtex->GetBitmap(0);
                if(bmap)
                {
                    if(!validTexSize(bmap->Width()) || !validTexSize(bmap->Height()))
                    {
                        RLogger::LogMessagebyID(RLogger::kLogERR_TextureSize, std::string(M_2_A(ffn.data())));
                        return false;
                    }
                }
            }


            MaxBitmapInfo bmi;
            bmi.bmtex = bmtex;
            bmi.filename = M_2_A(ffn);
            bmi.name = M_2_A(name);
            bmi.mapch = ch;
            bmi.isBump = (id == ID_BU);
            bmi.isAlpha = (id == ID_OP);
            bmi.isIFL = false;
#if !defined( NW4F_C3T_DELETE )
            bmi.rImage = NULL;
#endif

            switch(id)
            {
                case	ID_DI:
                    bmi.hint = Dcc::RSampler::ALBEDO;
                break;
                case	ID_SP:
                    bmi.hint = Dcc::RSampler::SPECULAR;
                break;
                case	ID_OP:
                    bmi.hint = Dcc::RSampler::OPACITY;
                break;
                case	ID_BU:
                    bmi.hint = Dcc::RSampler::NORMAL;
                break;
                case	ID_RL:
                    bmi.hint = Dcc::RSampler::REFLECTION;
                break;
            }
            //tga以外のフォーマットはRImageを作成して登録する。
            // IFLファイルの場合
            if(_strcmpi(M_2_A(ext), ".ifl") == 0)
            {
                bmi.isIFL = true;
            }
            else if(_strcmpi(M_2_A(ext), ".tga") != 0)
            {
/*
                if(exportTexture)
                {
                    bool ret = CreateRImage(bmi);
                    if(!ret) return false;
                }
*/
            }

            m_BitmapArray.push_back(bmi);
        }
        //else
        {
            //RLogger::LogMessagebyID(RLogger::kLogWRN_NonSupportedTexFormat, fn.data());
            //mprintf("WARNING: map file(%s) is not TGA format\n", fn.data());
        }
    }
    else
    {
        string mes;
        mes = M_2_A(fn.data());
        mes += "(";
        mes += M_2_A(m_pMaxMtl->GetName().data());
        mes += ")";

        RLogger::LogMessagebyID(RLogger::kLogWRN_TexFileIsNotFound, mes);
        //mprintf("WARNING: map file(%s) is not exist\n", fn.data());
        // 登録はされないが、警告のみなのでtrueを返す
        return true;
    }
    return true;
}

static Dcc::RVec4 ColorToRVec4(Color c, float a = 1.0f)
{
    return Dcc::RVec4(c.r, c.g, c.b, a);
}

#define ANISOSHADER_CLASS_ID     0x2857f460

/*
 * PBlockからコントロールを取得
 * 2012からメソッド名が変わったので、この関数で取得する。
 */
static Control* getControlFromPBlockByIndex(IParamBlock2* pblock, int index)
{
    if(!pblock)
    {
        return NULL;
    }

#if (MAX_VERSION_MAJOR >= 14) // max2012以降
    return pblock->GetControllerByIndex(index);
#else
    return pblock->GetController(index);
#endif

}

//情報をRMaterialにコピーする
void MtlInfo::CopyToFMaterial(RSceneMaterials &sceneMtls, FMaterial* fm, Dcc::RIntArray& mapCh, Dcc::RIntArray& hintCh)
{
    // maxでマテリアルが割り当てられてない場合はダミーの名前を入力
    if(m_pMaxMtl == NULL)
    {
        fm->m_Name = "_nw4f_3dsmax_nomaterial_";
        return;
    }

    // カラーなどをコピー
    fm->m_Name = GetName();

    fm->SetDiffuse(GetDiffuse());
    fm->SetEmission(GetEmittion());
    fm->SetSpecular(GetSpecular());
    fm->SetAmbient(GetAmbient());
    fm->SetOpacity(m_Transparency);
/*
    fm->EnableLighting(true, false);
    fm->SetTextureCoordinateConfig("Config0120");
*/
    bool isColorTransparent = (fabsf(GetDiffuse().w - 1.0f) > 0.001f);

#if	1
    //	TODO:
    // カラーアニメーション (標準マテリアルのみサポート)
    if(sceneMtls.GetDoesExportAnimation()
        && (m_pMaxMtl->ClassID() == Class_ID(DMTL_CLASS_ID,0)) )
    {
        StdMat2* mtl = (StdMat2*)m_pMaxMtl;
        bool useEmitColor = mtl->GetSelfIllumColorOn() == TRUE;
        bool hasDiffuseAnim = false;
        bool hasAmbientAnim = false;
        bool hasAlphaAnim = false;
        bool hasSpecularAnim = false;
        bool hasEmissionAnim = false;

        // シェーダのパラメータブロックから、各カラーのキー情報を取得
        Shader* shader = mtl->GetShader();
        if(shader)
        {
            // シェーダに応じてディフューズ、スペキュラ、エミット量、
            // エミットカラーのパラメータIDをセット
#if (MAX_PRODUCT_YEAR_NUMBER > 2016) // max2016まで確認済み
            #pragma message(TODO("シェーダのパラメータを決め打ちで取得している。バージョン依存？"))
#endif
            int pid_dif = -1, pid_amb, pid_spec, pid_emit, pid_emitcol;
            if(shader->ClassID() == Class_ID(BLINNClassID,0) ||
                shader->ClassID() == Class_ID(PHONGClassID,0)
            )
            {
                pid_amb = 0;
                pid_dif = 1;
                pid_spec = 2;
                pid_emit = 7;
                pid_emitcol = 8;
            }
            else if(shader->ClassID() == Class_ID(ANISOSHADER_CLASS_ID,0))
            {
                pid_amb = 0;
                pid_dif = 1;
                pid_spec = 2;
                pid_emit = 6;
                pid_emitcol = 3;
            }

            // シェーダーのパラメータを列挙
            /*
            {
                IParamBlock2* pblock = shader->GetParamBlock(0);
                if(pblock)
                {
                    MCHAR* sname = pblock->GetLocalName();
                    for(int i = 0; i < pblock->NumParams(); ++i)
                    {
                        ParamID id = pblock->IndextoID(i);
                        Control* c;
                        c = getControlFromPBlockByIndex(pblock, i);
                        if(c)
                        {
                        }
                        MCHAR* pname = pblock->SubAnimName(i);
                        //pblock->SubAnimName()
                        //MCHAR* pname = pblock->GetLocalName(id);
                        if(pname)
                        {
                            DebugPrint("index:%d, id=%d, name = %s\n", i, id,  pname);
                        }
                    }
                }

            }
            */


            // シェーダのパラメータブロックからコントローラを取り出し、
            // キーが有れば強制的にアニメーション出力するフラグを立てる
            if(pid_dif >= 0)
            {
                IParamBlock2* pblock = shader->GetParamBlock(0);
                if(pblock)
                {
                    Control* c;
                    c = getControlFromPBlockByIndex(pblock, pid_dif);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        hasDiffuseAnim = true;
                    }

                    c = getControlFromPBlockByIndex(pblock, pid_amb);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        hasAmbientAnim = true;
                    }

                    c = getControlFromPBlockByIndex(pblock, pid_spec);
                    if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                    {
                        hasSpecularAnim = true;
                    }

                    if(useEmitColor)
                    {
                        c = getControlFromPBlockByIndex(pblock, pid_emitcol);
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            hasEmissionAnim = true;
                        }
                    }
                    else
                    {
                        c = getControlFromPBlockByIndex(pblock, pid_emit);
                        if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                        {
                            hasEmissionAnim = true;
                        }
                    }
                }
            }
            // マテリアルのパラメータブロックから透明度のコントローラを取り出し、
            // キーが有れば強制的にアニメーション出力するフラグを立てる
            IParamBlock2* pbext = mtl->GetParamBlock(1);
            if(pbext)
            {
                Control* c;
                c = getControlFromPBlockByIndex(pbext, 1);
                if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                {
                    hasAlphaAnim = true;
                }
            }
        }

        // 各フレームの値を評価しアニメーションをセット
        float emitVal;
        Color difCol, ambCol;
        if(hasDiffuseAnim || hasAmbientAnim || hasAlphaAnim || hasSpecularAnim || hasEmissionAnim)
        {
            for(int t = m_SubStart_; t <= m_SubEnd_; t++)
            {
                int f = t * m_SubStep_;

                if(hasDiffuseAnim)
                {
                    difCol = mtl->GetDiffuse(f);
                    fm->SetDiffuseAnim(Dcc::RVec3(difCol.r, difCol.g, difCol.b) );
                }

                if(hasAmbientAnim)
                {
                    ambCol = mtl->GetAmbient(f);
                    fm->SetAmbientAnim(Dcc::RVec3(ambCol.r, ambCol.g, ambCol.b) );
                }

                if(hasAlphaAnim)
                {
                    fm->SetOpacityAnim( mtl->GetOpacity(f) );
                }

                if(hasSpecularAnim)
                {
                    fm->SetSpecularAnim( ColorToRVec4(mtl->GetSpecular(f)) );
                }

                if(hasEmissionAnim)
                {
                    if(useEmitColor)
                    {
                        fm->SetEmissionAnim( ColorToRVec4(mtl->GetSelfIllumColor(f)) );
                    }
                    else
                    {
                        emitVal = mtl->GetSelfIllum(f);
                        fm->SetEmissionAnim( Dcc::RVec4(emitVal, emitVal, emitVal, 1.0f) );
                    }
                }

                if(!isColorTransparent)
                {
                    isColorTransparent = ((1.0f - mtl->GetOpacity(f)) > 0.001f);
                }
            }
        }
    }
#endif

    // ブレンドモードを設定（カスタムアトリビュートが有れば再設定される）
    /* ブレンドモードは常にOPA
     *if(m_HasBitmapAlpha || m_HasVtxAlpha || isColorTransparent)
    {
        fm->SetRenderStateMode(Dcc::RRenderState::XLU);
    }
    else
    */
    {
        fm->SetRenderStateMode(Dcc::RRenderState::OPA);
    }

    // 異方性反射、スペキュラー
    if(m_IsAnisotropic)
    {
        fm->EnableAnisotropic(true);
    }
    else if(m_ShineLevel > 0.0f)
    {
        //	Shininess に値を設定しているが FDC からは値を出力していない。
#if	0
        //	TODO:
        fm->SetSpecularLevel(shineLevel);
#endif
        fm->SetShiness(m_Shiness * 100.0f);
        fm->EnableSpecular(true);
    }

#if	1
    //////////////////////////////////////////////////////////////////////////////////
    // マテリアル設定プラグインで付加された情報を設定する

    Mtl* ccMtl = m_pMaxMtl;
    CustAttrib* attrib = NULL;
    while(ccMtl && attrib == NULL)
    {
        ICustAttribContainer* cc = ccMtl->GetCustAttribContainer();
        if(cc)
        {
            // Nw4fMtlCustAttribが存在すればそれを返す
            for(int i = 0; i < cc->GetNumCustAttribs(); i++)
            {
                if(cc->GetCustAttrib(i)->ClassID() == NINTENDOEXPORT_MTL_ATTRIB_CLASS_ID)
                {
                    attrib = cc->GetCustAttrib(i);
                    break;
                }
            }
        }
        if(attrib == NULL)
        {
            if(ccMtl == m_pMaxMtl)
            {
                ccMtl = m_pParentMaxMtl;
            }
            else
            {
                ccMtl = NULL;
            }
        }

    }

    // Nw4fMtlCustAttribが有る場合はフラグをチェック
    if(attrib)
    {
        IParamBlock2* pblock = attrib->GetParamBlock(0);
        if(pblock)
        {
            bool compress = (pblock->GetInt(nw4f_mtl_compress) == 0);
            fm->m_CompressEnable = compress;
            const MCHAR* opt = pblock->GetStr(nw4f_mtl_userSettingString);
            if(opt)
            {
                string userOptions = M_2_A(opt);
                SetUserDataFromString(userOptions, fm, NULL);
            }
            /*
            bool fLight = (pblock->GetInt(nw4f_mtl_fragmentLighting) == TRUE);
            bool vLight = (pblock->GetInt(nw4f_mtl_vertexLighting) == TRUE);
            int  cullMode = pblock->GetInt(nw4f_mtl_faceCulling);
            bool hasPriority = (pblock->GetInt(nw4f_mtl_renderPriorityType) == 1);
            int  priority = pblock->GetInt(nw4f_mtl_renderPriority);
            int  blendMode = pblock->GetInt(nw4f_mtl_blendMode);



            // Render proority
            if(hasPriority) fm->SetRenderPriority(priority);

            // lighting
            fm->EnableLighting(vLight, fLight);

            // face culling
            RRasterization raster;
            switch(cullMode)
            {
                case 1:
                    raster.mCullingMode = RRasterization::FrontFace;
                    break;
                case 2:
                    raster.mCullingMode = RRasterization::Always;
                    break;
                case 3:
                    raster.mCullingMode = RRasterization::Never;
                    break;
                case 0:
                default:
                    raster.mCullingMode = RRasterization::BackFace;
                    break;
            }
            fm->SetRasterization(raster);

            // Blend Mode
            RMaterial::BlendMode mode = RMaterial::BLEND_MODE_OPAQUE;
            switch(blendMode)
            {
                case 1:
                    mode = RMaterial::BLEND_MODE_OPAQUE; break;
                case 2:
                    mode = RMaterial::BLEND_MODE_ALPHA_TEST; break;
                case 3:
                    mode = RMaterial::BLEND_MODE_TRANSLUCENT; break;
                case 4:
                    mode = RMaterial::BLEND_MODE_ADD; break;
                case 5:
                    mode = RMaterial::BLEND_MODE_SUB; break;
                case 6:
                    mode = RMaterial::BLEND_MODE_MUL; break;
                case 0:
                default:
                    if(hasBitmapAlpha || hasVtxAlpha || isColorTransparent)
                    {
                        mode = RMaterial::BLEND_MODE_TRANSLUCENT;
                    }
                    else
                    {
                        mode = RMaterial::BLEND_MODE_OPAQUE;
                    }
                    break;
            }
            fm->SetBlendMode(mode);
            */
        }
    }
#endif

    // テクスチャを列挙する
    int numBitmap = GetNumBitmap();
    //	使用用途ごとのテクスチャ使用数
    int	nNumTexPerHint[Dcc::RSampler::HINT_COUNT];

    memset( nNumTexPerHint, 0, sizeof(nNumTexPerHint) );

    for(int j = 0; j < numBitmap; j++)
    {
        float sc_u = 1.0f, sc_v = 1.0f, tl_u = .0f, tl_v = .0f, rot = .0f;
        int usech = 1;
        // sceneMaterialにビットマップを追加
        MaxBitmapInfo* bmInfo = GetBitmap(j);
        if(!bmInfo) continue;
        StdUVGen* uv = bmInfo->bmtex->GetUVGen();

        Dcc::RSampler sampler(bmInfo->hint, nNumTexPerHint[bmInfo->hint] );

        if(bmInfo->isIFL)
        {
            //	IFL の場合、parseIFL で使用しているファイル名が決定してから以下のパラメータを設定する
            //sampler.m_TexName = bmInfo->name;
            //sampler.m_TexPath = std::string(Dcc::RExpOpt::TexFolderName) + "/" + bmInfo->name.data() + ".ftx";
        }
        else
        {
            //	テクスチャを出力するためにイメージを登録する。
            Dcc::RStringArray	filePaths;
            filePaths.push_back(bmInfo->filename.data());
            //	GetTexImage の utility 実装
            sceneMtls.AddTextureFileForExport( sampler.m_ImageIndex, filePaths, sampler, nNumTexPerHint[bmInfo->hint], GetNintendoOptions()->m_Opt);

            const Dcc::RImage& texImg = sceneMtls.getTexture(sampler.m_ImageIndex);

            sampler.m_TexName = texImg.GetName();
            //	TODO: GetTexImagePath を実装する。
            //sampler.m_TexPath = GetTexImagePath(texImg.GetName(), yopt);
            sampler.m_TexPath = std::string(Dcc::RExpOpt::TexFolderName) + "/" + texImg.GetName() + ".ftx";
        }

        // サンプラーアトリビュートのユーザー設定
        //////////////////////////////////////////////////////////////////////////////////
        // マテリアル設定プラグインで付加された情報を設定する

        {
            CustAttrib* texattrib = NULL;
            ICustAttribContainer* cc = bmInfo->bmtex->GetCustAttribContainer();
            if(cc)
            {
                // Nw4fSamplerCustAttribが存在すればそれを返す
                for(int i = 0; i < cc->GetNumCustAttribs(); i++)
                {
                    if(cc->GetCustAttrib(i)->ClassID() == NINTENDOEXPORT_SAMPLER_ATTRIB_CLASS_ID)
                    {
                        texattrib = cc->GetCustAttrib(i);
                        break;
                    }
                }
                if(texattrib)
                {
                    IParamBlock2* pblock = texattrib->GetParamBlock(0);
                    if(pblock)
                    {
                        const MCHAR* samplerName = pblock->GetStr(nw4f_sampler_name);
                        if(samplerName)	sampler.m_NameOverride = M_2_A(samplerName);
                        const MCHAR* samplerHint = pblock->GetStr(nw4f_sampler_hint);
                        if(samplerName)	sampler.m_HintOverride = M_2_A(samplerHint);
                    }
                }
            }
        }

        //-----------------------------------------------------------------------------
        // テクスチャのラップ設定を取得します。
        sampler.m_WrapU = Dcc::RSampler::CLAMP;
        sampler.m_WrapV = Dcc::RSampler::CLAMP;
        //if (!texNode.IsEnvMap() && !texNode.m_TexObj.isNull())
        {
            // RMaterialにTexureMapperを追加
            int tiling = uv->GetTextureTiling();
            if(tiling & U_WRAP)
            {
                sampler.m_WrapU = Dcc::RSampler::REPEAT;
            }
            else if(tiling & U_MIRROR)
            {
                //	TODO:	MierroredRepeat -> REPEAT
                sampler.m_WrapU = Dcc::RSampler::REPEAT;
            }

            if(tiling & V_WRAP)
            {
                sampler.m_WrapV= Dcc::RSampler::REPEAT;
            }
            else if(tiling & V_MIRROR)
            {
                //	TODO:	MierroredRepeat -> REPEAT
                sampler.m_WrapV = Dcc::RSampler::REPEAT;
            }
        }

        //	TODO:
        // IFL ファイルの処理
        if(bmInfo->isIFL)
        {
            parseIFL(bmInfo->filename, bmInfo, sceneMtls, sampler);
        }

        // RMaterialにTexureCoordinatorを追加
        // テクスチャのマップチャンネルを探す
        int assignCh = -1;
        for(UINT ch = 0; ch < mapCh.size(); ch++)
        {
            if(mapCh[ch] == bmInfo->mapch)
            {
                assignCh = ch;
                break;
            }
        }

        //テクスチャのマップチャンネルが見つからない場合は適当なチャンネルを割り当てる。
        if(assignCh < 0)
        {
            char buf[2048];
            sprintf_s(buf, "Material: %s, MapCh: %d.", GetName().c_str(), uv->GetMapChannel());
            RLogger::LogMessagebyID(RLogger::kLogWRN_MapChannelDoesNotExist, buf);

            for(UINT ch = 0; ch < mapCh.size(); ch++)
            {
                if(mapCh[ch] > 0)
                {
                    assignCh = ch;
                    bmInfo->mapch = mapCh[ch];
                    break;
                }
            }
        }

        //for(UINT ch = 0; ch < mapCh.size(); ch++)
        if(assignCh >= 0)
        {
            fm->m_UseCafeUV[assignCh] = true;

            int stf = m_SubStart_ * m_SubStep_;
            usech = assignCh;
            sc_u = uv->GetUScl(stf);
            sc_v = uv->GetVScl(stf);
            rot  = uv->GetWAng(stf) * RAD_TO_DEG;
            tl_u = uv->GetUOffs(stf);
            tl_v = uv->GetVOffs(stf);

            //-----------------------------------------------------------------------------
            // テクスチャ SRT を設定します。
            Dcc::ROriginalTexsrt& texSrt = sampler.m_OriginalTexsrt;
            //	TODO:	設定を拾う
            //texSrt.m_Mode        = static_cast<ROriginalTexsrt::Mode>(yopt.m_TexSrtMode);
            texSrt.m_Mode		 = Dcc::ROriginalTexsrt::MAX;
            texSrt.m_Scale.x     = sc_u;
            texSrt.m_Scale.y     = sc_v;
            texSrt.m_Rotate      = rot;
            texSrt.m_Translate.x = tl_u;
            texSrt.m_Translate.y = tl_v;
            //texSrt.m_UvHintIdx = hintCh[assignCh];
            fm->m_SamplerReferUVs.push_back(assignCh);

#if	1
            /*
            int numCoord = fm->AddTexureCoordinator(
                usech,
                RTexCoordinator::UvCoordinateMap,
                RTexCoordinator::Dcc3dsMax,
                sc_u, sc_v, rot, tl_u, tl_v);
            */
            // UVアニメーション
            if(sceneMtls.GetDoesExportAnimation()) // && numCoord > 0)
            {
                //RTexCoordinator* coord = fm->GetTexCoordinator(numCoord - 1);
                bool hasTexScaleUAnim = false;
                bool hasTexScaleVAnim = false;
                bool hasTexRotateAnim = false;
                bool hasTexTransUAnim = false;
                bool hasTexTransVAnim = false;

#if (MAX_PRODUCT_YEAR_NUMBER > 2016) // max2016まで確認済み
                #pragma message(TODO("直接UV座標のコントローラが取得できないので、サブアニメーション経由で取得している。バージョン依存？"))
#endif
                // 直接UV座標のコントローラが取得できないので、サブアニメーション経由で取得
                int numsub = uv->NumSubs();
                if(numsub > 0)
                {
                    MSTR snmae = uv->SubAnimName(0);
                    Animatable* anim = uv->SubAnim(0);

                    int numasub = anim->NumSubs();
                    enum
                    {
                        texOffU = 0,
                        texOffV = 1,
                        texSclU = 2,
                        texSclV = 3,
                        texRot  = 6,
                    };

                    // 間違って変なアニメーションにアクセスしないことを期待…
                    if(numasub >= 13)
                    {
                        Animatable* sa;

                        sa = anim->SubAnim(texSclU);
                        if(sa->NumKeys() > 0 || sa->IsAnimated())
                            {	hasTexScaleUAnim = true;	}

                        sa = anim->SubAnim(texSclV);
                        if(sa->NumKeys() > 0 || sa->IsAnimated())
                            {	hasTexScaleVAnim = true;	}

                        sa = anim->SubAnim(texRot);
                        if(sa->NumKeys() > 0 || sa->IsAnimated())
                            {	hasTexRotateAnim = true;	}

                        sa = anim->SubAnim(texOffU);
                        if(sa->NumKeys() > 0 || sa->IsAnimated())
                            {	hasTexTransUAnim = true;	}

                        sa = anim->SubAnim(texOffV);
                        if(sa->NumKeys() > 0 || sa->IsAnimated())
                            {	hasTexTransVAnim = true;	}
                    }

                    /*
                    for(int ai = 0; ai < numasub; ai++)
                    {
                        MSTR ssname = anim->SubAnimName(ai);
                        Animatable* sanim = anim->SubAnim(ai);
                        int numkey = sanim->NumKeys();
                        Class_ID acid = sanim->ClassID();
                        DebugPrint("%d: %s(%d)(%X,%X)\n", ai, ssname.data(), numkey, acid.PartA(), acid.PartB());
                    }
                    */
                }

                if(hasTexScaleUAnim || hasTexScaleVAnim
                    || hasTexRotateAnim
                    || hasTexTransUAnim || hasTexTransVAnim)
                {
                    TexSrtAnim texSrtAnim;
                    texSrtAnim.m_ObjName = "texsrt_dummy";

                    for(int t = m_SubStart_; t <= m_SubEnd_; t++)
                    {
                        int f = t * m_SubStep_;
                        if(hasTexScaleUAnim)
                        {
                            texSrtAnim.m_Anims[Dcc::ROriginalTexsrt::SCALE_X].m_FullValues.push_back(Dcc::RSnapToZero(uv->GetUScl(f)));
                        }
                        if(hasTexScaleVAnim)
                        {
                            texSrtAnim.m_Anims[Dcc::ROriginalTexsrt::SCALE_Y].m_FullValues.push_back(Dcc::RSnapToZero(uv->GetVScl(f)));
                        }
                        if(hasTexRotateAnim)
                        {
                            texSrtAnim.m_Anims[Dcc::ROriginalTexsrt::ROTATE].m_FullValues.push_back(Dcc::RSnapToZero(uv->GetWAng(f) * RAD_TO_DEG));
                        }
                        if(hasTexTransUAnim)
                        {
                            texSrtAnim.m_Anims[Dcc::ROriginalTexsrt::TRANSLATE_X].m_FullValues.push_back(Dcc::RSnapToZero(uv->GetUOffs(f)));
                        }
                        if(hasTexTransVAnim)
                        {
                            texSrtAnim.m_Anims[Dcc::ROriginalTexsrt::TRANSLATE_Y].m_FullValues.push_back(Dcc::RSnapToZero(uv->GetVOffs(f)));
                        }
                    }
                    sampler.m_TexSrtAnimIndex = static_cast<int>(sceneMtls.getNumTexSrtAnimData());
                    sceneMtls.addTexSrtAnimData(texSrtAnim);
                }
            }
#endif
        }
        fm->m_Samplers.push_back( sampler );
        nNumTexPerHint[bmInfo->hint]++;
    }

    // カラーテクスチャマップが使われている場合は、ディフューズを白にする。
    if(GetNumColorBitmap() > 0)
    {
        Dcc::RVec4 col(Dcc::RVec4::kWhite);
        col.w = GetDiffuse().w;
        fm->SetDiffuse(col);
    }

#if	0
    // 法線マップ使用時
    if(bumpTexIdx >= 0 && bumpTexCh >= 0)
    {
        fm->SetNormalMap(bumpTexIdx);
    }
#endif
}


bool MultiMtlInfo::Init(Mtl* mtl, bool exportTex)
{
    bool ret;
    // マルチマテリアルの場合
    if(mtl && mtl->IsMultiMtl())
    {
        int numSubMtl = mtl->NumSubMtls();
        for(int i =0; i < numSubMtl; i++)
        {
            Mtl* subMaxMtl = mtl->GetSubMtl(i);
            MtlInfo* info = AddSubMtl(i);
            if(info)
            {
                ret = info->Init(subMaxMtl);
                if(!ret) return false;
                info->m_pParentMaxMtl = mtl;
            }
        }
    }
    else
    {
        MtlInfo* info = AddSubMtl(0);
        if(info)
        {
            ret = info->Init(mtl);
            if(!ret) return false;
        }
    }
    return true;
}

MtlInfo* MultiMtlInfo::AddSubMtl(int id)
{
    MtlInfo* mtl = NULL;

    subMtlMap::iterator it = m_SubMtl.find(id);
    if(it == m_SubMtl.end())
    {
        mtl = new MtlInfo();
        m_SubMtl.insert(subMtlMap::value_type(id, mtl));
    }
    else
    {
        mtl = it->second;
    }
    return mtl;
}

MtlInfo* MultiMtlInfo::GetSubMtl(int id)
{
    MtlInfo* mtl = NULL;

    if(m_SubMtl.size() > 0)
    {
        id = id % (int)m_SubMtl.size();

        subMtlMap::iterator it = m_SubMtl.find(id);
        if(it != m_SubMtl.end())
        {
            mtl = it->second;
        }
    }
    return mtl;
}

