﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <util/UtilString.h>

#include <filesystem>

namespace filesystem = std::tr2::sys;

namespace nw { namespace g3d { namespace tool {
namespace util {

std::string ToUTF8(const wchar_t* str)
{
    int len = WideCharToMultiByte(CP_UTF8, 0, str, -1, nullptr, 0, nullptr, nullptr);	// \0 を含む長さ
    std::string utf8;
    utf8.resize(len-1);
    int result = WideCharToMultiByte(CP_UTF8, 0, str, -1, const_cast<char*>(utf8.data()), len, nullptr, nullptr);

    if (result == 0)
    {
        return "";
    }

    return utf8;
}

std::string ToAscii(const wchar_t* str)
{
    int len = static_cast<int>(wcslen(str));
    std::string cStr;
    cStr.resize(len);

    char* pDst = const_cast<char*>(cStr.data());
    for (const wchar_t* pSrc = str; *pSrc; ++pSrc, ++pDst)
    {
        *pDst = static_cast<char>(*pSrc);
        if (*pDst != *pSrc)
        {
            return "";
        }
    }

    return cStr;
}

std::string ToLocale(const std::string& str)
{
    std::string cStr;

    // ワイド文字に変換
    int wcsLen = MultiByteToWideChar(CP_UTF8, 0, str.c_str(), static_cast<int>(str.length()), nullptr, 0);
    std::unique_ptr<wchar_t> wcs(new wchar_t[wcsLen]);
    wcsLen = MultiByteToWideChar(CP_UTF8, 0,
        str.c_str(), static_cast<int>(str.length()), wcs.get(), wcsLen);

    // ロケールの文字コードに変換
    int mbsLen = WideCharToMultiByte(0, 0,
        wcs.get(), wcsLen, nullptr, 0, nullptr, nullptr);
    cStr.resize(mbsLen);
    mbsLen = WideCharToMultiByte(0, 0,
        wcs.get(), wcsLen, &cStr[0], mbsLen, nullptr, nullptr);

    return cStr;
}

std::wstring ToWstr(const std::string& str)
{
    // ワイド文字に変換
    setlocale(LC_CTYPE, "");
    std::wstring wcStr;
    int wcsLen = MultiByteToWideChar(CP_UTF8, 0, str.c_str(), static_cast<int>(str.length()), nullptr, 0);
    std::unique_ptr<wchar_t> wcs(new wchar_t[wcsLen]);
    wcsLen = MultiByteToWideChar(CP_UTF8, 0,
        str.c_str(), static_cast<int>(str.length()), wcs.get(), wcsLen);
    wcStr.resize(wcsLen);
    wcStr.assign(wcs.get());
    return wcStr;
}

void Split(std::vector<std::string>& v, const std::string& str, const std::string& sep, bool delSep)

{
    size_t i = str.find_first_not_of(sep);
    size_t size = str.size();

    v.clear();
    while (0 <= i && i < size)
    {
        size_t last = str.find_first_of(sep, i);
        if (last < 0 || last > size)
        {
            last = size;
        }
        v.push_back(str.substr(i, last - i));

        i = last + 1;
        if (delSep)
        {
            i = str.find_first_not_of(sep, i);
        }
    }
}

std::string Trim(const std::string& line, const std::string blank)
{
    std::string str = line.substr(0, line.find_last_not_of(blank) + 1);
    std::string::size_type begin = str.find_first_not_of(blank);
    if (begin != str.npos)
    {
        return str.substr(begin);
    }

    return "";
}

std::string TrimLeft(const std::string& line, const std::string blank)
{
    std::string::size_type begin = line.find_first_not_of(blank);
    if (begin != line.npos)
    {
        return line.substr(begin);
    }

    return "";
}

std::string TrimRight(const std::string& line, const std::string blank)
{
    return line.substr(0, line.find_last_not_of(blank) + 1);
}

std::string TrimString(const std::string& str, const std::string& trim)
{
    std::string::size_type pathPos = 0;
    if (strncmp(trim.data(), str.data(), trim.length() - 1) == 0)
    {
        pathPos = trim.length() - 1;
    }

    return str.substr(pathPos);
}

std::string EraseToken(const std::string& str, const std::vector<std::string>& tokens)
{
    std::string newstring(str);
    for (auto token = tokens.cbegin(); token != tokens.cend(); ++token)
    {
        for( ; ; )
        {
            auto found = newstring.find_first_of(*token);
            if (found == std::string::npos)
            {
                break;
            }

            newstring.erase(found, (*token).length());
        }
    }

    return newstring;
}

std::string EraseNull(const std::string& str)
{
    std::string newString(str);

    if (newString[newString.size() - 1] == '\0')
    {
        newString.resize(newString.size() - 1);
    }

    return newString;
}

bool ModifyFloatValue(std::string& str)
{
    std::vector<std::string> tokens;
    tokens.push_back("f");
    tokens.push_back("F");
    str = EraseToken(str, tokens);
    char** endptr = nullptr;
    strtod(str.c_str(), endptr);

    if (endptr != nullptr)
    {
        // float ではない文字が含まれる。
        return false;
    }

    return true;
}

bool ModifyIntValue(std::string& str)
{
    char** endptr = nullptr;
    strtol(str.c_str(), endptr, 10);
    if (endptr != nullptr)
    {
        return false;
    }

    return true;
}

std::string ModifyBlank(const std::vector<std::string>& v)
{
    std::stringstream ss;
    for (auto value = v.cbegin(); value != v.cend(); ++value)
    {
        if (value != v.cbegin())
        {
            ss << " ";
        }
        ss << Trim(*value);
    }

    return ss.str();
}

std::string EscapeString(const std::string& str)
{
    const std::string pre_amp = "&";
    const std::string post_amp = "&amp;";
    const std::string pre_lt = "<";
    const std::string post_lt = "&lt;";
    const std::string pre_gt = ">";
    const std::string post_gt = "&gt;";
    const std::string pre_quot = "\"";
    const std::string post_quot = "&quot;";
    const std::string pre_apos = "'";
    const std::string post_apos = "&apos;";

    std::string result(str);
    result = ReplaceAll(result, pre_amp, post_amp);
    result = ReplaceAll(result, pre_lt, post_lt);
    result = ReplaceAll(result, pre_gt, post_gt);
    result = ReplaceAll(result, pre_quot, post_quot);
    result = ReplaceAll(result, pre_apos, post_apos);

    return result;
}

bool TryParse(int& value, const std::string& str)
{
    const char* nptr = str.c_str();
    char* endptr = nullptr;
    int tempValue = 0;
    tempValue = std::strtol(nptr, &endptr, 10);

    if (*endptr != '\0')
    {
        return false;
    }

    value = tempValue;
    return true;
}

bool TryParse(int& value, const std::wstring& str)
{
    const wchar_t* nptr = str.c_str();
    wchar_t* endptr = nullptr;
    int tempValue = 0;
    tempValue = std::wcstol(nptr, &endptr, 10);

    if (*endptr != '\0')
    {
        return false;
    }

    value = tempValue;
    return true;
}

std::string GetModuleName(HINSTANCE hInstance)
{
    char modulePath[MAX_PATH];
    if (!GetModuleFileNameA(hInstance, modulePath, MAX_PATH))
    {
        return std::string();
    }
    filesystem::path path(modulePath);
    return std::move(path.filename().string());
}

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
