﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <util/UtilPrint.h>
#include <util/UtilException.h>
#include <util/UtilBackTrace.h>

#include <vector>
#include <functional>
#include <io.h>
#include <Fcntl.h>

namespace nw { namespace g3d { namespace tool {

#define PROC_STRING_RESOURCE_ADDRESS(handle, name, type)             \
auto name = reinterpret_cast<type>(GetProcAddress(handle, #name));   \

// エラー、警告、ログの出力設定を行うクラス
class LogConfig
{
public:
    enum Level
    {
        Verbose,
        ShowWarning,
        ShowSystemLog,
        Silent = ShowSystemLog,
        ShowError,
        SilentAll
    };

    static void SetOutStream(FILE* out)
    {
        s_Out = out;
    }

    static void SetWarningStream(FILE* out)
    {
        s_Warning = out;
    }

    static void SetErrorStream(FILE* err)
    {
        s_Err = err;
    }

    static bool InitializeStringResource(const char* path);

    static void FinalizeStringResource()
    {
        PROC_STRING_RESOURCE_ADDRESS(s_StringResourceDll, FinalizeStringResource, bool(*)());
        FinalizeStringResource();
        FreeLibrary(s_StringResourceDll);
    }

    static Level s_Level;
    static FILE* s_Out;
    static FILE* s_Warning;
    static FILE* s_Err;
    static bool s_Trace;
    static bool s_InternalTrace;
    static HINSTANCE s_StringResourceDll;
};

inline
    void Trace(const wchar_t* path, int line)
{
    wchar_t fname[_MAX_FNAME];
    wchar_t ext[_MAX_EXT];
    _wsplitpath_s(path, nullptr, 0, nullptr, 0, fname, _MAX_FNAME, ext, _MAX_EXT);
    ::nw::g3d::tool::util::PrintLn(LogConfig::s_Err, "    %ls%ls(%d)", fname, ext, line);
}

// エラーと同様の扱いになるので、サイレントで消えない
#define PRINT_SYSTEM_LOG(format, ...)                                                             \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::ShowSystemLog)              \
    {                                                                                              \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Out, format, __VA_ARGS__);                     \
    }                                                                                              \

#define PRINT_ERROR_LOG(format, ...)                                                               \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::ShowError)                  \
    {                                                                                              \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, format, __VA_ARGS__);                     \
    }                                                                                              \

// サイレントオプションで消しても良いログ
#define PRINT_LOG(format, ...)                                                                     \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::Verbose)                    \
    {                                                                                              \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Out, format, __VA_ARGS__);      \
    }                                                                                              \

#define LOG_LEVEL_0(messege) "\n* " messege
#define LOG_LEVEL_1(messege) "\n*** " messege
#define LOG_LEVEL_2(messege) "\n****** " messege
#define LOG_LEVEL_SHADER(messege) "\n------ " messege

#define PRINT_TRACE() ::nw::g3d::tool::Trace(__FILEW__, __LINE__)

#define PRINT_INFO(stream, type, no, format, ...)                                                  \
    {                                                                                              \
        ::nw::g3d::tool::util::Print(stream, "%s(%#x): ", type, no);                               \
        ::nw::g3d::tool::util::PrintLn(stream, format, __VA_ARGS__);                               \
    }                                                                                              \

#define PRINT_WARNING(no, format, ...)                                                             \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::ShowWarning)                \
    {                                                                                              \
        PRINT_INFO(nw::g3d::tool::LogConfig::s_Warning, L"Warning", no, format, __VA_ARGS__);                         \
    }                                                                                              \

#if (_DEBUG)
#define DEBUG_BREAK()                                                                              \
    if (IsDebuggerPresent())                                                                       \
    {                                                                                              \
        ::DebugBreak();                                                                            \
    }                                                                                              \

#else
#define DEBUG_BREAK() (void)0
#endif


#define THROW_ERROR_DETAIL(trace, no, format, ...)                                                 \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::ShowError)                  \
    {                                                                                              \
        PRINT_INFO(nw::g3d::tool::LogConfig::s_Err, L"Error", no, format, __VA_ARGS__);            \
        if (trace)                                                                                 \
        {                                                                                          \
            PRINT_TRACE();                                                                         \
        }                                                                                          \
        DEBUG_BREAK();                                                                             \
        throw ::nw::g3d::tool::util::Exception(no, __FILEW__, __LINE__);                           \
    }                                                                                              \


#define THROW_ERROR_INTERNAL(no, format, ...)                                                      \
    THROW_ERROR_DETAIL(nw::g3d::tool::LogConfig::s_InternalTrace, no, format, __VA_ARGS__)         \

#if (_DEBUG)
#define THROW_ERROR(no, format, ...)                                                               \
    THROW_ERROR_DETAIL(nw::g3d::tool::LogConfig::s_InternalTrace, no, format, __VA_ARGS__)         \

#else
#define THROW_ERROR(no, format, ...)                                                               \
    THROW_ERROR_DETAIL(nw::g3d::tool::LogConfig::s_Trace, no, format, __VA_ARGS__)                 \

#endif

// BinaryBlock 又はその派生クラス内で例外を投げる場合のみ使用可能
#define THROW_BINARY_BLOCK_ERROR(no, format, ...)                                                  \
    PRINT_SYSTEM_LOG("---------------")                                                            \
    PRINT_SYSTEM_LOG("Error occured at...\n")                                                      \
    DumpLogRecursively();                                                                          \
    PRINT_SYSTEM_LOG("---------------")                                                            \
    THROW_ERROR(no, format, __VA_ARGS__)                                                           \

#define THROW_SHADER_ERROR(log, type, no)                                                          \
    if (nw::g3d::tool::LogConfig::s_Level <= nw::g3d::tool::LogConfig::ShowError)                  \
    {                                                                                              \
        DEBUG_BREAK();                                                                             \
        throw ::nw::g3d::tool::util::ShaderException<type>(no, __FILEW__, __LINE__, log);          \
    }                                                                                              \

#define CATCH_ERROR_ALL(result)                                                                    \
    catch(::nw::g3d::tool::util::Exception&)                                                       \
    {                                                                                              \
        return result;                                                                             \
    }                                                                                              \
    catch (std::exception& except)                                                                 \
    {                                                                                              \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "Internal Error (STD)");   \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "%hs", except.what());     \
        return result;                                                                             \
    }                                                                                              \
    catch (_EXCEPTION_POINTERS* ep)                                                                \
    {                                                                                              \
        PEXCEPTION_RECORD rec = ep->ExceptionRecord;                                               \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "Internal Error (SEH)");                  \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "code:%x flag:%x addr:%p params:%d\n",    \
            rec->ExceptionCode,                                                                    \
            rec->ExceptionFlags,                                                                   \
            rec->ExceptionAddress,                                                                 \
            rec->NumberParameters                                                                  \
        );                                                                                         \
                                                                                                   \
        char temp[512];                                                                            \
        nw::g3d::tool::BackTrace::Get()->AddressToSymbolString(rec->ExceptionAddress, temp, 512);  \
        temp[511] = '\0';                                                                          \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "%hs", temp);              \
        return result;                                                                             \
    }                                                                                              \
    catch (...)                                                                                    \
    {                                                                                              \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, "Internal Error (Unknown)");              \
        return result;                                                                             \
    }                                                                                              \


#define CATCH_THROW_WITH_MSG_DETAIL(isTrace, format, ...)                                          \
    catch (...)                                                                                    \
    {                                                                                              \
        if(isTrace)                                                                                \
        {																						   \
            PRINT_TRACE();                                                                         \
        }																						   \
        ::nw::g3d::tool::util::PrintLn(nw::g3d::tool::LogConfig::s_Err, format, __VA_ARGS__);      \
        throw;                                                                                     \
    }                                                                                              \

#define PRINT_TRANSLATED_ERROR_LOG(identifier, ...)                                                                      \
    std::unique_ptr<char[]> log(new char[256]);                                                                          \
    PROC_STRING_RESOURCE_ADDRESS(nw::g3d::tool::LogConfig::s_StringResourceDll, GetString, const char*(*)(const char*)); \
    snprintf(log.get(), 256, GetString(identifier), __VA_ARGS__);                                                        \
    PRINT_ERROR_LOG("%hs", log.get());                                                                                   \

#define THROW_TRANSLATED_ERROR_INTERNAL(no, identifier, ...)                                                             \
    std::unique_ptr<char[]> log(new char[256]);                                                                          \
    PROC_STRING_RESOURCE_ADDRESS(nw::g3d::tool::LogConfig::s_StringResourceDll, GetString, const char*(*)(const char*)); \
    snprintf(log.get(), 256, GetString(identifier), __VA_ARGS__);                                                         \
    THROW_ERROR_DETAIL(nw::g3d::tool::LogConfig::s_InternalTrace, no, "%hs", log.get())                                  \

#define THROW_TRANSLATED_ERROR(no, identifier, ...)                                                                      \
    std::unique_ptr<char[]> log(new char[256]);                                                                          \
    PROC_STRING_RESOURCE_ADDRESS(nw::g3d::tool::LogConfig::s_StringResourceDll, GetString, const char*(*)(const char*)); \
    snprintf(log.get(), 256, GetString(identifier), __VA_ARGS__);                                                        \
    THROW_ERROR(no, "%hs", log.get());                                                                                   \

#define THROW_TRANSLATED_BINARY_BLOCK_ERROR(no, identifier, ...)                                                         \
    PRINT_SYSTEM_LOG("---------------")                                                                                  \
    PRINT_SYSTEM_LOG("Error occured at...\n")                                                                            \
    DumpLogRecursively();                                                                                                \
    PRINT_SYSTEM_LOG("---------------")                                                                                  \
    THROW_TRANSLATED_ERROR(no, identifier, __VA_ARGS__)                                                                  \

#if defined(_DEBUG)
#define CATCH_THROW_WITH_MSG(format, ...) CATCH_THROW_WITH_MSG_DETAIL(nw::g3d::tool::LogConfig::s_InternalTrace, format, __VA_ARGS__)
#else
#define CATCH_THROW_WITH_MSG(format, ...) CATCH_THROW_WITH_MSG_DETAIL(nw::g3d::tool::LogConfig::s_Trace, format, __VA_ARGS__)
#endif

#define THROW_INTERNAL_ERROR() \
    THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.")

void PrintErrorSource(const char* source);
void PrintErrorSource(const std::vector<const char*>* sources);

template <typename T, typename S, typename Comp>
bool CompInSrcType(S src, T compVal, Comp comp)
{
    return comp(src, static_cast<S>(compVal));
}

template <typename T, typename S, typename Comp, bool Check = true>
struct CastCheckerImpl
{
    static void Check(S src, T compVal, Comp comp)
    {
        if (!CompInSrcType(src, compVal, comp))
        {
            THROW_ERROR(ERRCODE_CAST_OVERFLOW, "cast error %hs -> %hs.",
                typeid(S).name(), typeid(T).name());
        }
    }
};
template <typename T, typename S, typename Comp>
struct CastCheckerImpl<T, S, Comp, false>{ static void Check(S, T, Comp){} };

template <typename T, typename S, bool Over, bool Under>
struct CastCheckerOverUnder
{
    static void Check(S src)
    {
        CastCheckerImpl<T, S, std::less_equal<S>, Over>::Check(
            src, (std::numeric_limits<T>::max)(), std::less_equal<S>());
        CastCheckerImpl<T, S, std::greater_equal<S>, Under>::Check(
            src, (std::numeric_limits<T>::lowest)(), std::greater_equal<S>());
    }
};

template <typename T, typename S>
struct CastChecker
    : public CastCheckerOverUnder<T, S , std::is_floating_point<S>::value
        || sizeof(S) >= sizeof(T) , std::is_signed<S>::value>
{
};

template <typename T, typename S>
inline T NumericCast(S src)
{
    CastChecker<T, S>::Check(src);
    return static_cast<T>(src);
}

} // namespace tool
} // namespace g3d
} // namespace nw
