﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Skeleton.h>
#include <algorithm>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    bone, billboard,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(world_viewvector),
    S11N_DEFINE_ENUM_ID(world_viewpoint),
    S11N_DEFINE_ENUM_ID(screen_viewvector),
    S11N_DEFINE_ENUM_ID(screen_viewpoint),
    S11N_DEFINE_ENUM_ID(yaxis_viewvector),
    S11N_DEFINE_ENUM_ID(yaxis_viewpoint)
);

G3DIF_DEFINE_ENUM_TABLE(
    bone, side,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(left),
    S11N_DEFINE_ENUM_ID(right),
    S11N_DEFINE_ENUM_ID(center),
);

G3DIF_DEFINE_ENUM_TABLE(
    skeleton_info, scale_mode,
    S11N_DEFINE_ENUM_ID(standard),
    S11N_DEFINE_ENUM_ID(maya),
    S11N_DEFINE_ENUM_ID(softimage)
);

G3DIF_DEFINE_ENUM_TABLE(
    skeleton_info, rotate_mode,
    S11N_DEFINE_ENUM_ID(euler_xyz),
    S11N_DEFINE_ENUM_ID(quaternion)
);

const float ScaleAdjuster::s_MinScale = 1.0e-9f;

void elem_bone::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        parent_name << pElem;
        matrix_index << pElem;
        scale << pElem;
        rotate << pElem;
        translate << pElem;
        scale_compensate << pElem;
        billboard << pElem;
        visibility << pElem;
        side << pElem;

        if (matrix_index->size != 2)
        {
            THROW_ERROR(ERRCODE_XML_INVALID_STREAM, "Invalid matrix_index stream.");
        }

        const util::XMLElement* pInv = pElem->Child(tbl_identifier[id_inv_model_matrix]);
        if (pInv)
        {
            inv_model_matrix.Validate();
            Read(pInv->Text(), inv_model_matrix.Get());
        }

        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_skeleton_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        scale_enable << pElem;
        scale_mode << pElem;
        rotate_mode << pElem;
        motion_mirroring_enable << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_skeleton::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        skeleton_info << pElem->Child(elem_skeleton_info::Id());
        bone_array << pElem->Child(elem_bone::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

// 親の BoneIndex の解決
// MatrixPalette (matrixIndex と boneIndex のペア) のリストを作成
void elem_skeleton::ResolveSkeleton()
{
    int boneIndex = 0;
    for (auto iterBone = bone_array.begin(); iterBone != bone_array.end(); ++iterBone, ++boneIndex)
    {
        elem_bone& bone = *iterBone;
        // 深さ優先で並んでいるので、遡りながら親を探す。
        for (int parentIndex = boneIndex - 1; parentIndex >= 0; --parentIndex)
        {
            if (bone.parent_name.value == bone_array[parentIndex].name.value)
            {
                bone.parent_index = parentIndex;
                break;
            }
        }

        if (iterBone->matrix_index->data.get()[0] >= 0)
        {
            MatrixPalette palette;
            palette.matrixIndex = iterBone->matrix_index->data.get()[0];
            palette.boneIndex = boneIndex;
            matrixPalette.push_back(palette);
        }

        if (iterBone->matrix_index->data.get()[1] >= 0)
        {
            MatrixPalette palette;
            palette.matrixIndex = iterBone->matrix_index->data.get()[1];
            palette.boneIndex = boneIndex;
            matrixPalette.push_back(palette);
        }
    }

    // MatrixPalette での順番でソート
    std::sort(matrixPalette.begin(), matrixPalette.end(),
        [](MatrixPalette lhs, MatrixPalette rhs) { return lhs.matrixIndex < rhs.matrixIndex; });
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
