﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <ostream>

#include <g3dif/Serialization.h>
#include <g3dif/ShaderConfig.h>
#include <g3dif/Material.h>
#include <g3dif/Vertex.h>

namespace nw { namespace g3d { namespace tool {
namespace shdrcvtr {

class ShaderProgram;

}
namespace g3dif {

class elem_uniform_var;

class elem_shader_symbol
{
public:
    elem_shader_symbol()
        : name()
    {}

    G3DIF_DEFINE_ELEM(shader_symbol);

    // ステージ別の id 取得
    static const nw::g3d::tool::util::LiteralStr& Id(int stage)
    {
        return nw::g3d::tool::g3dif::tbl_identifier[GetIdentifier(stage)];
    }

    static const Identifier GetIdentifier(int stage)
    {
        switch (stage)
        {
        case ShaderStage_Vertex:
            return nw::g3d::tool::g3dif::id_vertex_symbol;
        case ShaderStage_Geometry:
            return nw::g3d::tool::g3dif::id_geometry_symbol;
        case ShaderStage_Fragment:
            return nw::g3d::tool::g3dif::id_fragment_symbol;
        case ShaderStage_Compute:
            return nw::g3d::tool::g3dif::id_compute_symbol;
        default:
            break;
        }
        THROW_ERROR(ERRCODE_INTERNAL, "internal error");
        return nw::g3d::tool::g3dif::id_nw4f_3dif;
    }

    G3DIF_DEFINE_ATTRIB(std::string, name);
};

class elem_ui_label
{
public:
    elem_ui_label()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_label);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_comment
{
public:
    elem_ui_comment()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_comment);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_group
{
public:
    elem_ui_group()
        : group_name()
    {
    }

    G3DIF_DEFINE_ELEM(ui_group);

    G3DIF_DEFINE_ATTRIB(std::string, group_name);
};

class elem_ui_order
{
public:
    elem_ui_order()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_order);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_alias
{
public:
    elem_ui_alias()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_alias);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_item
{
public:
    elem_ui_item()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_item);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_min
{
public:
    elem_ui_min()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_min);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_max
{
public:
    elem_ui_max()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_max);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_visible
{
public:
    elem_ui_visible()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_visible);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_default_min
{
public:
    elem_ui_default_min()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_default_min);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_default_max
{
public:
    elem_ui_default_max()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_default_max);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};


class elem_ui_editable
{
public:
    elem_ui_editable()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_editable);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_ui_anim_editable
{
public:
    elem_ui_anim_editable()
        : value()
    {
    }

    G3DIF_DEFINE_ELEM(ui_anim_editable);

    G3DIF_DEFINE_ATTRIB(std::string, value);
};

class elem_option_var
{
public:
    // Array には未対応
    G3DIF_DEFINE_ENUM(type,
        type_static,
        type_dynamic);

    elem_option_var()
        : id()
        , type(type_static)
        , choice()
        , default_value()
        , branch(false)
        , shader_symbol()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_order()
        , ui_alias()
        , ui_visible()
        , choiceArray()
        , choiceUintArray()
        , bitWidth(0)
        , keyOffset(0)
        , defaultValueIndex(-1)
    {
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            altUniformSymbol[stageIndex] = std::string();
        }
    }

    G3DIF_DEFINE_ELEM_ARRAY(option_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(enum_type, type);
    G3DIF_DEFINE_ATTRIB(std::string, choice);
    // default が予約語なので別名で受け取る。
    util::Attribute<std::string, id_default> default_value;
    G3DIF_DEFINE_ATTRIB(bool, branch);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];
    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_alias> ui_alias;
    util::Optional<elem_ui_visible> ui_visible;

    // choice を展開した文字列集合。
    // choice と alias のペア
    std::vector<std::pair<std::string, std::string>> choiceArray;

    // choice の uint 表現（branch で利用）
    std::vector<uint32_t> choiceUintArray;
    // 必要なビット長
    size_t bitWidth;
    // 先頭から choice までのオフセット
    size_t keyOffset;
    // デフォルト値のインデクス。
    int defaultValueIndex;

    // ステージごとの uniform 定義による代替文字列
    std::string altUniformSymbol[ShaderStage_StageCount];

    // fsc で定義されるフィルターを用いて有効な choice のみを残す。
    void Filter(util::StringRef choice_filter, util::StringRef default_filter);

    // fsd の choice や default を展開する。
    void Expand(const elem_uniform_var* el_option_uniform);

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;

    // 文字列から型を取得します。
    static enum_type GetType(std::string str);
};

class elem_attrib_var
{
public:
    elem_attrib_var()
        : id()
        , hint()
        , type()
        , vertex_symbol()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_editable()
        , ui_order()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(attrib_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, hint);
    G3DIF_DEFINE_ATTRIB(elem_vtx_attrib::enum_type, type);

    elem_shader_symbol vertex_symbol;

    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_editable> ui_editable;
    util::Optional<elem_ui_order> ui_order;

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;
};

class elem_sampler_var
{
public:
    G3DIF_DEFINE_ENUM(type,
        isampler_1d,
        isampler_2d,
        isampler_2drect,
        isampler_3d,
        isampler_cube,
        isampler_buffer,
        isampler_2dms,
        isampler_1d_array,
        isampler_2d_array,
        isampler_cube_array,
        isampler_2dms_array,
        usampler_1d,
        usampler_2d,
        usampler_2drect,
        usampler_3d,
        usampler_cube,
        usampler_buffer,
        usampler_2dms,
        usampler_1d_array,
        usampler_2d_array,
        usampler_cube_array,
        usampler_2dms_array,
        sampler_1d,
        sampler_2d,
        sampler_2drect,
        sampler_3d,
        sampler_cube,
        sampler_buffer,
        sampler_2dms,
        sampler_1d_array,
        sampler_2d_array,
        sampler_cube_array,
        sampler_2dms_array,
        sampler_1d_shadow,
        sampler_2d_shadow,
        sampler_2drect_shadow,
        sampler_cube_shadow,
        sampler_1d_array_shadow,
        sampler_2d_array_shadow,
        sampler_cube_array_shadow
        );

    elem_sampler_var()
        : id()
        , hint()
        , type()
        , alt()
        , nostrip(false)
        , shader_symbol()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_order()
        , ui_visible()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(sampler_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, hint);
    G3DIF_DEFINE_ATTRIB(enum_type, type);
    G3DIF_DEFINE_ATTRIB(std::string, alt);
    G3DIF_DEFINE_ATTRIB(bool, nostrip);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];
    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_visible> ui_visible;

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;

    // 文字列から型を取得します。
    static enum_type GetType(std::string str);
};

class elem_uniform_var
{
public:
    elem_uniform_var()
        : id()
        , hint()
        , type()
        , default_value()
        , nostrip(false)
        , shader_symbol()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_order()
        , ui_item()
        , ui_min()
        , ui_max()
        , ui_visible()
        , rawdata()
        , dataSize(0)
        , converter()
        , depend()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(uniform_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, hint);
    G3DIF_DEFINE_ATTRIB(elem_shader_param::enum_type, type);
    // default が予約語なので別名で受け取る。
    util::Attribute<std::string, id_default> default_value;
    G3DIF_DEFINE_ATTRIB(bool, nostrip);
    G3DIF_DEFINE_ATTRIB(std::string, converter);
    G3DIF_DEFINE_ATTRIB(std::string, depend);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];

    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_item> ui_item;
    util::Optional<elem_ui_min> ui_min;
    util::Optional<elem_ui_max> ui_max;
    util::Optional<elem_ui_visible> ui_visible;
    util::Optional<elem_ui_default_min> ui_default_min;
    util::Optional<elem_ui_default_max> ui_default_max;

    std::shared_ptr<void> rawdata;
    size_t dataSize;

    // 型に応じて既定の値を取得します。
    static std::string GetDefault(elem_shader_param::enum_type type);
    static int GetDefaultCount(elem_shader_param::enum_type type);

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;

    void Expand();
};

class elem_ssbo_uniform_var
{
public:
    elem_ssbo_uniform_var()
        : id()
        , hint()
        , type()
        , default_value()
        , nostrip(false)
        , shader_symbol()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_order()
        , ui_item()
        , ui_min()
        , ui_max()
        , ui_visible()
        , rawdata()
        , dataSize(0)
        , converter()
        , depend()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(buffer_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, hint);
    G3DIF_DEFINE_ATTRIB(elem_shader_param::enum_type, type);
    // default が予約語なので別名で受け取る。
    util::Attribute<std::string, id_default> default_value;
    G3DIF_DEFINE_ATTRIB(bool, nostrip);
    G3DIF_DEFINE_ATTRIB(std::string, converter);
    G3DIF_DEFINE_ATTRIB(std::string, depend);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];
    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_item> ui_item;
    util::Optional<elem_ui_min> ui_min;
    util::Optional<elem_ui_max> ui_max;
    util::Optional<elem_ui_visible> ui_visible;
    util::Optional<elem_ui_default_min> ui_default_min;
    util::Optional<elem_ui_default_max> ui_default_max;

    std::shared_ptr<void> rawdata;
    size_t dataSize;

    // 型に応じて既定の値を取得します。
    static std::string GetDefault(elem_shader_param::enum_type type);
    static int GetDefaultCount(elem_shader_param::enum_type type);

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;

    void Expand();
};


class elem_block_var
{
public:
    G3DIF_DEFINE_ENUM(type,
        none,
        material,
        shape,
        skeleton,
        option
        );

    elem_block_var()
        : id()
        , type()
        , shader_symbol()
        , uniform_var_array()
        , pConvertedDefaultValue()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(block_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(enum_type, type);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];

    // 文字列から型を取得します。
    static enum_type GetType(std::string str);

    std::vector<elem_uniform_var> uniform_var_array;

    void Expand();

    // Res から復元したデフォルト値です。merge 時にのみ使用します。
    std::shared_ptr<void> pConvertedDefaultValue;
};

class elem_ssbo_block_var
{
public:
    // xsdではnone,materialしか許可されていないがelem_block_varと同じtypeを定義する。
    // 後段でelem_ssbo_block_var、elem_uniform_block_varの処理を特殊化しないで済むようにするため。
    G3DIF_DEFINE_ENUM(type,
        none,
        material,
        shape,
        skeleton,
        option
        );

    elem_ssbo_block_var()
        : id()
        , type()
        , shader_symbol()
        , uniform_var_array()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(shader_storage_block_var);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(enum_type, type);

    util::Optional<elem_shader_symbol> shader_symbol[ShaderStage_StageCount];

    // 文字列から型を取得します。
    static enum_type GetType(std::string str);
    static bool IsValidType( elem_ssbo_block_var::enum_type type );
    std::vector<elem_ssbo_uniform_var> uniform_var_array;

    void Expand();
};

class elem_render_info_slot
{
public:
    G3DIF_DEFINE_ENUM(type,
        type_int,
        type_float,
        type_string
        );

    elem_render_info_slot()
        : name()
        , type()
        , count()
        , optional(true)
        , choice()
        , default_value()
        , ui_label()
        , ui_comment()
        , ui_group()
        , ui_order()
        , ui_item()
        , ui_visible()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(render_info_slot);

    G3DIF_DEFINE_ATTRIB(std::string, name);
    G3DIF_DEFINE_ATTRIB(std::string, count);
    G3DIF_DEFINE_ATTRIB(bool, optional);
    G3DIF_DEFINE_ATTRIB(enum_type, type);
    G3DIF_DEFINE_ATTRIB(std::string, choice);
    // default が予約語なので別名で受け取る。
    util::Attribute<std::string, id_default> default_value;

    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_item> ui_item;
    util::Optional<elem_ui_visible> ui_visible;

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;

    static enum_type GetType(std::string str);
};

class elem_textblock
{
public:
    elem_textblock()
        : id()
        , text()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(textblock);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, text);

    util::Optional<elem_ui_group> ui_group;
    util::Optional<elem_ui_order> ui_order;

    void WriteStream(std::ostream& outStream, int indent) const;
};

class elem_interleave
{
public:
    elem_interleave()
        : id_set()
        , binarize()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(interleave);

    G3DIF_DEFINE_ATTRIB(std::string, id_set);
    G3DIF_DEFINE_ATTRIB(std::string, binarize);
};

class elem_group
{
public:
    elem_group()
        : name()
        , condition()
        , ui_label()
        , ui_comment()
        , ui_order()
        , ui_group()
        , page_name()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(group);

    G3DIF_DEFINE_ATTRIB(std::string, name);

    // condition はエスケープされた文字列で格納されているので、
    // 式の判定をするためにはエスケープを解除する必要がある。
    G3DIF_DEFINE_ATTRIB(std::string, condition);

    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_order> ui_order;
    util::Optional<elem_ui_group> ui_group;

    G3DIF_DEFINE_ATTRIB(std::string, page_name);

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream, int indent) const;
};

class elem_page
{
public:
    elem_page()
        : name()
        , ui_label()
        , ui_comment()
        , ui_order()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(page);

    G3DIF_DEFINE_ATTRIB(std::string, name);

    util::Optional<elem_ui_label> ui_label;
    util::Optional<elem_ui_comment> ui_comment;
    util::Optional<elem_ui_order> ui_order;

    void WriteStream(std::ostream& outStream, int index) const;
};

class elem_streamout
{
public:
    elem_streamout()
        : varying()
    {
    }

    G3DIF_DEFINE_ELEM(streamout);

    G3DIF_DEFINE_ATTRIB(std::string, varying);
};

class elem_shader_stage
{
public:
    elem_shader_stage()
        : src_index()
        , path()
    {}

    G3DIF_DEFINE_ELEM(shader_stage);

    // ステージ別の id 取得
    static const nw::g3d::tool::util::LiteralStr& Id(int stage)
    {
        return nw::g3d::tool::g3dif::tbl_identifier[GetIdentifier(stage)];
    }

    static const Identifier GetIdentifier(int stage)
    {
        switch (stage)
        {
        case ShaderStage_Vertex:
            return nw::g3d::tool::g3dif::id_vertex_stage;
        case ShaderStage_Geometry:
            return nw::g3d::tool::g3dif::id_geometry_stage;
        case ShaderStage_Fragment:
            return nw::g3d::tool::g3dif::id_fragment_stage;
        case ShaderStage_Compute:
            return nw::g3d::tool::g3dif::id_compute_stage;
        default:
            break;
        }
        THROW_ERROR(ERRCODE_INTERNAL, "internal error");
        return nw::g3d::tool::g3dif::id_nw4f_3dif;
    }

    G3DIF_DEFINE_ATTRIB(int, src_index);

    std::string path;
};

class elem_shading_model
{
public:
    elem_shading_model()
        : name()
        , material_shader(true)
        , revision()
        , shader_stage()
        , macro_array()
        , option_var_array()
        , attrib_var_array()
        , sampler_var_array()
        , block_var_array()
        , render_info_slot_array()
        , textblock_array()
        , interleave_array()
        , group_array()
        , streamout()
        , keyLength(0)
        , defaultKey()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(shading_model);

    G3DIF_DEFINE_ATTRIB(std::string, name);
    G3DIF_DEFINE_ATTRIB(bool, material_shader);
    G3DIF_DEFINE_ATTRIB(int, revision);

    util::Optional<elem_shader_stage> shader_stage[ShaderStage_StageCount];
    std::vector<elem_macro> macro_array;
    std::vector<elem_option_var> option_var_array;
    std::vector<elem_attrib_var> attrib_var_array;
    std::vector<elem_sampler_var> sampler_var_array;
    std::vector<elem_block_var> block_var_array;
    std::vector<elem_ssbo_block_var> ssbo_block_var_array;
    std::vector<elem_render_info_slot> render_info_slot_array;
    std::vector<elem_textblock> textblock_array;
    std::vector<elem_interleave> interleave_array;
    std::vector<elem_group> group_array;
    std::vector<elem_page> page_array;

    util::Optional<elem_streamout> streamout;

    size_t keyLength;
    // デフォルト値のキー
    std::shared_ptr<void> defaultKey;

    // 指定されたオプションの UniformBlock を返します。
    const elem_uniform_var* GetOptionUniform(const elem_option_var& el_option) const;

    // 処理順に記述する

    // システム定義のオプションを追加します。
    void AddSystemOption();

    // choice を配列に展開し、keyLength を求める。
    void Expand();

    // 指定されたストリームに書き出します。
    // fsd を書き出す場合のみ処理する。
    void WriteStream(std::ostream& outStream) const;

    // key の生成を行います。
    void CreateKey(u32* key, const std::vector<int>& choiceArray) const;
};

class elem_shader_src
{
public:
    elem_shader_src()
        : include_path()
        , path()
        , stream_index(-1)
        , stream()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(shader_src);

    G3DIF_DEFINE_ATTRIB(std::string, include_path);
    G3DIF_DEFINE_ATTRIB(std::string, path);
    G3DIF_DEFINE_ATTRIB(int, stream_index);

    // 終端文字を含むマルチバイト文字のストリーム。
    DataStream stream; // 参照解決後に利用可。
};

class elem_force_include
{
public:
    elem_force_include()
        : src_index()
        , path()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(force_include);

    G3DIF_DEFINE_ATTRIB(int, src_index);

    std::string path;
};

class elem_shader_definition_info
{
public:
    elem_shader_definition_info()
        : code_page(UINT_MAX)
        , config_path()
    {
    }

    G3DIF_DEFINE_ELEM(shader_definition_info);

    G3DIF_DEFINE_ATTRIB(int, code_page);
    G3DIF_DEFINE_ATTRIB(std::string, config_path);
};

//--------------------------------------------------------------------------------------------------

class elem_shader_definition
{
public:
    elem_shader_definition()
        : name()
        , path()
        , shader_definition_info()
        , force_include_array()
        , shading_model_array()
        , shader_src_array()
        , stream_array()
    {
    }

    G3DIF_DEFINE_ELEM(shader_definition);

    G3DIF_DEFINE_ATTRIB(Version, version);

    void PostProcess();
    void PostBinaryProcess(void* data);
    void SetName(const char* fileName) { this->name.assign(fileName); };
    void SetPath(const char* filePath) { this->path.assign(filePath); }
    void SetExt(const char* fileExt) { this->ext.assign(fileExt); }

    // 読み込んだデータの整合性を確認します。
    void CheckData(int flag);

    // 指定されたストリームに書き出します。
    void WriteStream(std::ostream& outStream) const;

    // 指定されたストリームにバイナリを書き出します。
    // ostream にはバイナリストリームを指定してください。
    void WriteBinaryStream(std::ostream& outStream) const;

    std::string name;
    std::string path;
    std::string ext;
    elem_shader_definition_info shader_definition_info;
    std::vector<elem_force_include> force_include_array;
    std::vector<elem_shading_model> shading_model_array;
    std::vector<elem_shader_src> shader_src_array;
    std::vector<elem_stream> stream_array;
};

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
