﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <clocale>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <eh.h>
#include <io.h>
#include <Fcntl.h>

#include <shdrdefs.h>
#include <ShdrConverter.h>

#include <util/UtilGLContext.h>

#define DLLEXPORT extern "C" __declspec(dllexport)

#include <nn/os.h>
#include <nn/init.h>

using namespace nw::g3d::tool;

bool g_IsDllInitialized = false;

static nn::g3dTool::Converter g_Converter;
FILE* g_pDefaultLogFile = NULL;
FILE* g_pWarningLogFile = NULL;
FILE* g_pErrorLogFile = NULL;
std::wstring s_DllPath;
const char* g_StringResourceName = "3dConverterStringResource.csv";

void se_translator_function(unsigned int /*code*/, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; //標準C++の例外を発生させる。
}

// nng3dToolShaderConverterInitialize 後に呼ぶ必要があります。
DLLEXPORT
bool nng3dToolShaderConverterSetLogStream(const nngfxToolSetLogStreamArg* pArg)
{
    if (!g_IsDllInitialized)
    {
        PRINT_ERROR_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
        return false;
    }
    if (pArg->hLogStream)
    {
        int nHandle = _open_osfhandle(reinterpret_cast<intptr_t>(pArg->hLogStream), _O_TEXT);
        switch (static_cast<nn::gfxTool::Logger::LogType>(pArg->logType))
        {
        case nn::gfxTool::Logger::LogType::Default:
            {
                g_pDefaultLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetOutStream(g_pDefaultLogFile);
            }
            break;
        case nn::gfxTool::Logger::LogType::Warning:
            {
                g_pWarningLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetWarningStream(g_pWarningLogFile);
            }
            break;
        case nn::gfxTool::Logger::LogType::Error:
            {
                g_pErrorLogFile = _fdopen(nHandle, "wt");
                nw::g3d::tool::LogConfig::SetErrorStream(g_pErrorLogFile);
            }
            break;
        default:
            PRINT_ERROR_LOG("Invalid LogType.");
            return false;
        }
    }

    // (gfx)ShaderConverter のログの出力設定
    return g_Converter.SetLogStream(pArg);
}


// 移行が済んだら削除する
DLLEXPORT
void nng3dToolShaderConverterOpenLogFile(const wchar_t* out[2], const wchar_t* err[2])
{
    if (out[0] && out[1] && _wfopen_s(&g_pDefaultLogFile, out[0], out[1]) == 0)
    {
        nw::g3d::tool::LogConfig::SetOutStream(g_pDefaultLogFile);
    }

    if (err[0] && err[1] && _wfopen_s(&g_pErrorLogFile, err[0], err[1]) == 0)
    {
        nw::g3d::tool::LogConfig::SetErrorStream(g_pErrorLogFile);
    }
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolShaderConverterCloseLogFile()
{
    if (g_pDefaultLogFile != NULL)
    {
        fclose(g_pDefaultLogFile);
        g_pDefaultLogFile = NULL;
    }

    if (g_pWarningLogFile != NULL)
    {
        fclose(g_pWarningLogFile);
        g_pWarningLogFile = NULL;
    }

    if (g_pErrorLogFile != NULL)
    {
        fclose(g_pErrorLogFile);
        g_pErrorLogFile = NULL;
    }

    nw::g3d::tool::LogConfig::SetOutStream(stdout);
    nw::g3d::tool::LogConfig::SetWarningStream(stdout);
    nw::g3d::tool::LogConfig::SetErrorStream(stderr);
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolShaderConverterInitializeGlContext()
{
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolShaderConverterShutdownGlContext()
{
}

DLLEXPORT
int nng3dToolShaderConverterGetConverterVersion()
{
    return g_Converter.GetCvtrVersion();
}

DLLEXPORT
bool nng3dToolShaderConverterInitialize()
{
    if (g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is already initialized.");
    }
    nn::gfxTool::Logger::SetDefaultStream();

    // エラー用の英語もしくは日本語のメッセージリストを初期化
    std::string stringResourcePath;
#if defined(_M_IX86)
    stringResourcePath = nw::g3d::tool::util::ToUTF8(s_DllPath.c_str()) + "../" + g_StringResourceName;
#else
    stringResourcePath = nw::g3d::tool::util::ToUTF8(s_DllPath.c_str()) + g_StringResourceName;
#endif
    if (!nw::g3d::tool::LogConfig::InitializeStringResource(stringResourcePath.c_str()))
    {
        // メッセージリスト構築に失敗してもコンバート自体は出来るので、エラーにはしない。
        PRINT_SYSTEM_LOG("Faild to setup StringResource");
    }

    bool result = g_Converter.Initialize(s_DllPath);
    g_IsDllInitialized = true;
    return result;
}

DLLEXPORT
bool nng3dToolShaderConverterShutdown()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    nw::g3d::tool::LogConfig::FinalizeStringResource();
    bool result = g_Converter.Shutdown();
    g_IsDllInitialized = false;
    return result;
}

DLLEXPORT
bool nng3dToolShaderConverterClear()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.Clear();
}

DLLEXPORT
bool nng3dToolShaderConverterSetOptions(const wchar_t* options[])
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.SetOptions(options);
}

DLLEXPORT
bool nng3dToolShaderConverterParseSetOptions(int argc, const wchar_t* argv[])
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.SetOptions(argc, argv);
}

DLLEXPORT
bool nng3dToolShaderConverterAddFile(void* pData, size_t dataSize, const wchar_t* path[3], const wchar_t* options[])
{
    NN_UNUSED(options);
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.AddFile(pData, dataSize, path);
}

DLLEXPORT
size_t nng3dToolShaderConverterCalculateArchiveSize()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    size_t size = g_Converter.CalculateArchiveSize();
    if( size == 0 )
    {
        return 0;
    }

    if( !g_Converter.Convert(nullptr, 0) )
    {
        return 0;
    }
    return g_Converter.GetBaseSize() + g_Converter.GetRelocationTableSize();
}

DLLEXPORT
bool nng3dToolShaderConverterConvert(void* pBuffer, size_t bufferSize)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    size_t serializedDataSize = g_Converter.GetBaseSize();
    size_t relocationTblSize = g_Converter.GetRelocationTableSize();
    size_t fileSize = serializedDataSize + relocationTblSize;	// relocation table を含めたサイズ
    if( fileSize > bufferSize )
    {
        return false;
    }

    // データ部分をコピー
    std::memcpy( pBuffer, g_Converter.GetBasePtr(), serializedDataSize );

    // リロケーションテーブルをコピー
    char* pDst = static_cast< char* >( pBuffer ) + serializedDataSize;
    std::memcpy( pDst, g_Converter.GetRelocationTablePtr(), relocationTblSize );
    return true;
}

DLLEXPORT
bool nng3dToolShaderConverterSwapEndian(void* pBuffer, size_t size)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.SwapEndian(pBuffer, size);
}

DLLEXPORT
bool nng3dToolShaderConverterMake()
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.Make();
}

DLLEXPORT
size_t nng3dToolShaderConverterCalculateDefinitionSize(const wchar_t* fullpath)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.CalculateDefinitionSize(fullpath);
}

DLLEXPORT
bool nng3dToolShaderConverterWrite(void* pBuffer, size_t size)
{
    if (!g_IsDllInitialized)
    {
        PRINT_SYSTEM_LOG("3dShaderConverter.dll is not initialized. Please call nng3dToolShaderConverterInitialize().");
    }
    return g_Converter.Write(pBuffer, size);
}

//--------------------------------------------------------------------------------------------------
// テスト用関数

// 移行が済んだら削除する
DLLEXPORT
bool nng3dToolShaderConverterCalculateExpandSize(size_t* shaderSize)
{
    *shaderSize = 0;
    return true;
}

// 移行が済んだら削除する
DLLEXPORT
bool nng3dToolShaderConverterExpandDeffinition(void* pBuffer[nn::g3dTool::ShaderStage_StageCount])
{
    NN_UNUSED(pBuffer);
    return true;
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterForceNwArchive(void)
{
}

// 移行が済んだら削除する
DLLEXPORT
void nng3dToolBinaryConverterForceNnArchive(void)
{
}

// 移行が済んだら削除する
DLLEXPORT
bool nng3dToolBinaryConverterIsNwArchive(void)
{
    return false;
}

DLLEXPORT
uint32_t nng3dToolShaderConverterGetBinaryVersion( void )
{
    return g_Converter.GetCvtrBinaryVersion();
}

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID /*lpvReserved*/)
{
    wchar_t dllfile[_MAX_PATH];
    wchar_t drive[_MAX_DRIVE];
    wchar_t dllpath[_MAX_DIR];
    std::wstringstream wss;
    int size = 0;
    switch (fdwReason)
    {
    case DLL_PROCESS_ATTACH:
        setlocale(LC_CTYPE, "");
        _set_se_translator(se_translator_function); // このスレッドでのみ有効。

        size = GetModuleFileName(hinstDLL, dllfile, _MAX_PATH);
        if (size != 0)
        {
            _wsplitpath_s(dllfile, drive, _MAX_DRIVE, dllpath, _MAX_DIR, NULL, 0, NULL, 0);

            wss << drive << dllpath;
            s_DllPath = wss.str();
            std::replace(s_DllPath.begin(), s_DllPath.end(), L'\\', L'/');
        }

        break;
    case DLL_PROCESS_DETACH:
        nng3dToolShaderConverterCloseLogFile();
        break;
    case DLL_THREAD_ATTACH:
        break;
    case DLL_THREAD_DETACH:
        break;
    }

    return TRUE;
}
