﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <shdrdefs.h>
#include <nn/g3d/g3d_ResShader.h>

#include <ShaderCompilerManager.h>

namespace nn { namespace g3dTool {


class ShaderMacro;
class SymbolName;

typedef std::unordered_map<std::string, size_t> UniformBlockMap;
typedef UniformBlockMap SsboBlockMap;	//!< uniform, bufferで名前がダブる事が無いので分ける必要はないが一応分ける。（buffer,uniform名で同一の物があるとコンパイルエラーになる。）
typedef std::unordered_map<std::string, ShaderCompilerManager::UniformInfo> UniformMap;
typedef UniformMap SsboMap;

class ShaderProgram
{
public:
    struct ShaderSet
    {
        ShaderSet()
            : arrayCount(0)
        {
            for (int i = 0; i < ShaderStage_StageCount; ++i)
            {
                index[i] = -1;
            }
        }

        int8_t index[ShaderStage_StageCount];
        uint32_t arrayCount;
    };

    explicit ShaderProgram(std::shared_ptr<void> key, size_t keyLength, const std::vector<SymbolName>* choiceArray )
        : m_Key(key)
        , m_KeyLength(keyLength)
        , m_ChoiceArray(choiceArray)
        , m_pResShaderFile( nullptr )
        , m_ResShaderVariationIdx( -1 )
    {
    }

    // uniform_block_array
    // uniform ブロックの名前とブロックのサイズ。
    UniformBlockMap m_UniformBlockMap;
    UniformBlockMap m_SsboBlockMap;

    // uniform 名とコンパイルバイナリからの抽出結果
    // インスタンス化していない場合はuniform/bufferでメンバ名のダブりが発生し得るので分けて管理が必要
    UniformMap	m_UniformMap;
    SsboMap		m_SsboMap;

    std::vector<ShaderSet> m_AttribTable;
    std::vector<ShaderSet> m_SamplerTable;
    std::vector<ShaderSet> m_UniformBlockTable;
    std::vector<ShaderSet> m_ShaderStorageBlockTable;

    // Expand 後のシェーダソース
    // メモリ節約のためコンパイル後は破棄されている
    std::string m_ShaderSource[ShaderStage_StageCount];

    std::shared_ptr<void> m_Key;
    size_t m_KeyLength;
    const std::vector<SymbolName>* m_ChoiceArray;

    // gfx 用
    std::shared_ptr<nn::gfx::ResShaderFile>	m_pResShaderFile;
    //!< TODO: ShaderProgram クラスの配列は key でソートされないといけない。
    int		m_ResShaderVariationIdx;
};

class ProgramFinder
{
public:
    ProgramFinder(const ShaderProgram* program)
        : m_Program(program)
    {}

    bool operator()(const ShaderProgram& program)
    {
        return memcmp(program.m_Key.get(), m_Program->m_Key.get(), program.m_KeyLength * sizeof(uint32_t)) == 0;
    }

    const ShaderProgram* m_Program;
};

} // namespace g3dTool
} // namespace nn
