﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief バッファに関する API の宣言
*/

#pragma once

#include <nn/utilTool/utilTool_BinarizerContext.h>
#include <ShaderProgram.h>
#include <memory>

namespace nn {
namespace g3dTool {

class BinaryBlock;
class BinShader;
struct Chunk;

/**
* @brief シリアライズに用いるヒープを管理するコンテキストです。
*/
class Context
{
public:
    struct ShaderContext
    {
        std::vector<ShaderProgram> m_Programs;
        std::vector<const ShaderProgram*> m_pPrograms;
        std::shared_ptr<ShaderProgram> m_SharedProgram;
        UniformBlockMap m_UniformBlockMap;
        SsboBlockMap m_SsboBlockMap;
        UniformMap m_UniformMap;
        SsboMap		m_SsboMap;
        size_t		m_ResShaderFileSize;
        size_t		m_ResShaderFileAlignment;
        BinShader*	m_pBinShader;
    };

    /**
    * @brief メモリブロックの種類を表す列挙子です。
    */
    enum MemBlockType
    {
        MemBlockType_Main,			//!< データ構造を格納するメモリブロックです。
        MemBlockType_ShaderFile,	//!< ResShaderFile のメモリブロックです。
        MemBlockType_Count,			//!< メモリブロックの個数です。
    };

    /**
    * @brief コンストラクタです。
    *
    * @post
    * - 各パラメータがゼロ初期化されている
    */
    Context() : blocks(), m_pStrPoolMemBlock( nullptr )
    {
        m_BinarlizerContext.Initialize( MemBlockType_Count );
    }

    /**
    * @brief コンテキストを構築します。
    *
    * @param[in] name 出力ファイル名
    *
    * @pre
    * - 各メモリブロックのサイズ、アライメントが設定されている
    * - 使用する全ての文字列が文字列プールに登録されている
    *
    * @post
    * - 各メモリブロックのヒープが確保され利用可能になる
    */
    void Build( const nn::util::string_view& name )
    {
        m_BinarlizerContext.SetName( name );

        for( auto memBlockId = 0; memBlockId < MemBlockType_Count; ++memBlockId )
        {
            m_BinarlizerContext.AddMemoryBlock( memBlockId, &m_MemBlockArray[ memBlockId ] );
        }

        m_pStrPoolMemBlock = m_BinarlizerContext.BuildStringPool();
        m_BinarlizerContext.AddMemoryBlock( 0, m_pStrPoolMemBlock );
        m_BinarlizerContext.AllocateBase();
        m_BinarlizerContext.ConvertStringPool();
    }

    /**
    * @brief 各メモリブロックをファイルに出力できる状態にコンバートします。
    *
    * @pre
    * - メモリブロック内を参照する全てのポインタのリンクが完了している
    * - 全てのバイナリブロックヘッダが登録されている
    *
    * @post
    * - ポインタがオフセットに変換される
    * - リロケーションテーブルが生成される
    *
    */
    void Convert( void )
    {
        m_BinarlizerContext.AllocateRelocationTable();
        m_BinarlizerContext.ConvertRelocationTable();
        m_BinarlizerContext.ConvertHeaders();
    }

    //! @brief バイナリブロックヘッダを登録します。
    void AddBinBlockHeader( nn::util::BinaryBlockHeader* pHeader )
    {
        m_BinarlizerContext.AddHeader( pHeader );
    }

    //! @brief 文字列プールに文字列を登録します。
    void SetStr( const nn::util::string_view& str )
    {
        m_BinarlizerContext.InsertString( str );
    }

    /**
    * @brief 文字列プールに登録されている文字列を参照するポインタを登録します。
    *
    * @post
    * - dest に参照先文字列へのオフセットが書き込まれる
    */
    void LinkStr( nn::util::BinPtrToString* dest, const nn::util::string_view& str )
    {
        m_BinarlizerContext.LinkString( dest, str );
    }

    /**
    * @brief メモリブロック内を参照するポインタを登録します。
    *
    * @post
    * - dest に参照先へのオフセットが書き込まれる
    */
    template< typename TType >
    void LinkPtr( nn::util::BinTPtr<TType>* dest, void* src )
    {
        m_BinarlizerContext.LinkPtr( dest, src );
    }

    //! @brief type で指定するメモリブロックのサイズをセットします。
    void SetMemBlockSize( MemBlockType type, size_t size )
    {
        m_MemBlockArray[ type ].SetSize( size );
    }

    //! @brief type で指定するメモリブロックのアライメントをセットします。
    void SetAlignment( MemBlockType type, size_t size )
    {
        m_MemBlockArray[ type ].SetAlignment( size );
    }

    //! @brief ヒープの先頭アドレスを返します。
    void* GetBasePtr( void )
    {
        return m_BinarlizerContext.GetBasePtr();
    }

    //! @brief ヒープサイズを返します。リロケーションテーブルは含みません。
    size_t GetBaseSize( void )
    {
        return m_BinarlizerContext.GetBaseSize();
    }

    //! @brief type で指定するメモリブロックの先頭アドレスを返します。
    void* GetMemBlockPtr( MemBlockType type )
    {
        return m_MemBlockArray[ type ].Get( GetBasePtr() );
    }

    //! @brief type で指定するメモリブロックのサイズを返します。
    size_t GetMemBlockSize( MemBlockType type ) const
    {
        return m_MemBlockArray[ type ].GetSize();
    }

    //! @brief リロケーションテーブルのサイズを返します。
    size_t GetRelocationTblSize( void ) const
    {
        return m_BinarlizerContext.GetRelocationTableSize();
    }

    //! @brief リロケーションテーブルの先頭アドレスを返します。
    void* GetRelocationTblPtr( void )
    {
        return m_BinarlizerContext.GetRelocationTablePtr();
    }

    //! @brief 文字列プールのメモリブロックを返します。
    const nn::util::MemorySplitter::MemoryBlock* GetStrPoolMemBlock( void ) const
    {
        return m_pStrPoolMemBlock;
    }

public:
    std::vector<BinaryBlock*>	blocks;
    std::list<ShaderContext> m_ShaderContexts;

private:
    nn::util::MemorySplitter::MemoryBlock	m_MemBlockArray[ MemBlockType_Count ];	//!< メモリブロックです。それぞれ独立したセクションに配置されます。
    nn::util::MemorySplitter::MemoryBlock*	m_pStrPoolMemBlock;						//!< バイナライザコンテキストが持つ文字列プールブロックへのポインタです。
    nn::utilTool::BinarizerContext			m_BinarlizerContext;
};

}
}
