﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.G3dTool.Entities;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

namespace nw.g3d.iftexutil
{

    public enum LoadLibraryFlags : uint
    {
        DONT_RESOLVE_DLL_REFERENCES = 0x00000001,
        LOAD_IGNORE_CODE_AUTHZ_LEVEL = 0x00000010,
        LOAD_LIBRARY_AS_DATAFILE = 0x00000002,
        LOAD_LIBRARY_AS_DATAFILE_EXCLUSIVE = 0x00000040,
        LOAD_LIBRARY_AS_IMAGE_RESOURCE = 0x00000020,
        LOAD_WITH_ALTERED_SEARCH_PATH = 0x00000008
    }

    public class GenericConverter : ITextureConverter
    {
        [DllImport("kernel32.dll", SetLastError = true)]
        public static extern IntPtr LoadLibraryEx(string lpFileName, IntPtr hReservedNull, LoadLibraryFlags dwFlags);
        [DllImport("kernel32.dll", SetLastError = true)]
        internal static extern bool FreeLibrary(IntPtr hModule);
        [DllImport("kernel32.dll", SetLastError = true, ExactSpelling = false, BestFitMapping = false, ThrowOnUnmappableChar = true)]
        internal static extern IntPtr GetProcAddress(IntPtr hModule, [MarshalAs(UnmanagedType.LPStr)] string lpProcName);

        /// <summary>
        /// TextureConverterEncoder.dll の ConvertFormat() 関数ポインタ。
        /// - この DLL の呼び出し規約は cdecl。
        /// - 関数の戻り値は C++ の bool 型 (1Byte) だが、UnmanagedType.Bool は BOOL 型 (4Byte) なので、サイズを合わせるために UnmagedType.I1 でマーシャリングする。
        /// </summary>
        /// <param name="dst">Out 属性を指定しなくても DLL 側での変更を取得できるが、これは C# の最適化による結果であり仕様ではないため、Out 指定が必要。https://msdn.microsoft.com/en-us/library/75dwhxf7.aspx </param>
        /// <param name="src"></param>
        /// <param name="dstFormat"></param>
        /// <param name="srcFormat"></param>
        /// <param name="quality"></param>
        /// <param name="encodeFlag"></param>
        /// <param name="dimension"></param>
        /// <param name="imageW"></param>
        /// <param name="imageH"></param>
        /// <param name="imageD"></param>
        /// <param name="mipCount"></param>
        /// <returns></returns>
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        [return: MarshalAs(UnmanagedType.I1)]
        private delegate bool ConvertFormatExtFunc(
            [In, Out, MarshalAs(UnmanagedType.LPArray)] byte[] dst,
            [MarshalAs(UnmanagedType.LPArray)] byte[] src,
            [MarshalAs(UnmanagedType.LPWStr)] string dstFormat,
            [MarshalAs(UnmanagedType.LPWStr)] string srcFormat,
            [MarshalAs(UnmanagedType.LPWStr)] string quality,
            int encodeFlag,
            int dimension,
            int imageW,
            int imageH,
            int imageD,
            int mipCount);
        private ConvertFormatExtFunc ConvertFormatExt;

        /// <summary>
        /// TextureConverterEncoder.dll の GetDataSize() 関数ポインタ。
        /// - この DLL の呼び出し規約は cdecl。
        /// - 関数の戻り値は C++ の size_t 型であり 32/64bit 環境でサイズが変わるため、UnmagedType.SysUInt でマーシャリングする。
        /// </summary>
        /// <param name="format"></param>
        /// <param name="dimension"></param>
        /// <param name="imageW"></param>
        /// <param name="imageH"></param>
        /// <param name="imageD"></param>
        /// <param name="mipCount"></param>
        /// <returns></returns>
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        [return: MarshalAs(UnmanagedType.SysUInt)]
        private delegate UIntPtr GetDataSizeExtFunc(
            [MarshalAs(UnmanagedType.LPWStr)] string format,
            int dimension,
            int imageW,
            int imageH,
            int imageD,
            int mipCount);
        private GetDataSizeExtFunc GetDataSizeExt;

        private IntPtr hDll;

        public bool Initialize(string[] basePath)
        {
            StringBuilder errorMessage = new StringBuilder();
            foreach (var path in basePath)
            {
                var dllpath = Path.GetFullPath(Path.Combine(path, @"TextureConverterEncoder.dll"));
                if (File.Exists(dllpath))
                {
                    hDll = LoadLibraryEx(dllpath, IntPtr.Zero, LoadLibraryFlags.LOAD_WITH_ALTERED_SEARCH_PATH);
                    if (hDll != IntPtr.Zero)
                    {
                        var convertFormat = GetProcAddress(hDll, @"ConvertFormat");
                        var getDataSize = GetProcAddress(hDll, @"GetDataSize");
                        ConvertFormatExt = Marshal.GetDelegateForFunctionPointer(convertFormat, typeof(ConvertFormatExtFunc)) as ConvertFormatExtFunc;
                        GetDataSizeExt = Marshal.GetDelegateForFunctionPointer(getDataSize, typeof(GetDataSizeExtFunc)) as GetDataSizeExtFunc;
                        return true;
                    }
                    else
                    {
                        errorMessage.Append($"Loading \"{dllpath}\" was failed with error code {Marshal.GetLastWin32Error()}\n");
                    }
                }
            }

            if (errorMessage.Length > 0)
            {
                // dll が存在するのにひとつも読めなかった場合はエラー扱いにしておく
                throw new Exception(errorMessage.ToString());
            }

            return false;
        }

        public string PlatformName
        {
            get
            {
                return "EncodedImage";
            }
        }

        public bool IsConverterLoaded
        {
            get
            {
                return (hDll != IntPtr.Zero);
            }
        }

        public void Destroy()
        {
            if (hDll != IntPtr.Zero)
            {
                FreeLibrary(hDll);
                hDll = IntPtr.Zero;
            }
        }

        // TextureConverter/Encoder/Encoder.h より
        /**
        * @brief イメージの次元です。
        */
        private enum ImageDimension
        {
            ImageDimension_1d, //!< 一次元です。
            ImageDimension_2d, //!< 二次元です。
            ImageDimension_3d, //!< 三次元です。
            ImageDimension_CubeMap, //!< キューブマップイメージです。
            ImageDimension_1dArray, //!< 一次元配列です。
            ImageDimension_2dArray, //!< 二次元配列です。
            ImageDimension_2dMultisample, //<! マルチサンプルイメージです。
            ImageDimension_2dMultisampleArray, //!< マルチサンプルイメージ配列です。
            ImageDimension_CubeMapArray, //!< キューブマップイメージ配列です。

            ImageDimension_End
        }

        /**
        * @brief エンコードフラグです。
        */
        private enum EncodeFlag
        {
            EncodeFlag_Default = 0, //!< デフォルトのエンコード方法です。
            EncodeFlag_ReverseRgba = 0x1 << 0, //!< 16bit RGB(A) / float_11_11_10 フォーマットの場合に下位 bit から RGB(A) の順に格納します。
            EncodeFlag_NormalMapLa = 0x1 << 1  //!< 法線マップの X 成分が RGB 成分、Y 成分が A 成分となるようにエンコードします。ASTC フォーマットでのみ有効です。
        }

        private static bool IsSnormOrSint(texture_info_quantize_typeType quantizeType)
        {
            return quantizeType.ToString().Contains("snorm_") || quantizeType.ToString().Contains("sint_");
        }

        private static int ConvertToEncoderDimension(texture_info_dimensionType dimension)
        {
            switch (dimension)
            {
                case texture_info_dimensionType.Item1d:
                    return (int)ImageDimension.ImageDimension_1d;
                case texture_info_dimensionType.Item1d_array:
                    return (int)ImageDimension.ImageDimension_1dArray;
                case texture_info_dimensionType.Item2d:
                    return (int)ImageDimension.ImageDimension_2d;
                case texture_info_dimensionType.Item2d_array:
                    return (int)ImageDimension.ImageDimension_2dArray;
                case texture_info_dimensionType.Item3d:
                    return (int)ImageDimension.ImageDimension_3d;
                case texture_info_dimensionType.cube:
                    return (int)ImageDimension.ImageDimension_CubeMap;
                case texture_info_dimensionType.cube_array:
                    return (int)ImageDimension.ImageDimension_CubeMapArray;
                default:
                    throw new Exception("Unexpected default");
            }
        }

        private byte[] ConvertFormat(texture_infoType info, byte[] srcBytes)
        {
            Ensure.Operation.True(IsConverterLoaded);
            var dimension = ConvertToEncoderDimension(info.dimension);
            var dstFormat = IsSnormOrSint(info.quantize_type)
                ? texture_info_quantize_typeType.snorm_8_8_8_8.ToString()
                : texture_info_quantize_typeType.unorm_8_8_8_8.ToString();
            var dstSize = GetDataSizeExt(dstFormat, dimension, info.width, info.height, info.depth, info.mip_level);
            var dstBytes = new byte[(UIntPtr.Size == 4) ? dstSize.ToUInt32() : dstSize.ToUInt64()];

            bool ret = ConvertFormatExt(
                    dstBytes,
                    srcBytes,
                    dstFormat,
                    info.quantize_type.ToString(),
                    string.Empty,
                    (int)EncodeFlag.EncodeFlag_ReverseRgba,
                    dimension,
                    info.width,
                    info.height,
                    info.depth,
                    info.mip_level);

            if (!ret)
            {
                return null;
            }

            return dstBytes;
        }

        private byte[] ConvertFormat(TextureInfo info, byte[] srcBytes)
        {
            Ensure.Operation.True(IsConverterLoaded);
            var dimension = ConvertToEncoderDimension(info.Dimension);
            var dstFormat = IsSnormOrSint(info.QuantizeType)
                ? texture_info_quantize_typeType.snorm_8_8_8_8.ToString()
                : texture_info_quantize_typeType.unorm_8_8_8_8.ToString();
            var dstSize = GetDataSizeExt(dstFormat, dimension, info.Width, info.Height, info.Depth, info.MipLevel);
            var dstBytes = new byte[(UIntPtr.Size == 4) ? dstSize.ToUInt32() : dstSize.ToUInt64()];

            bool ret = ConvertFormatExt(
                    dstBytes,
                    srcBytes,
                    dstFormat,
                    info.QuantizeType.ToString(),
                    string.Empty,
                    (int)EncodeFlag.EncodeFlag_ReverseRgba,
                    dimension,
                    info.Width,
                    info.Height,
                    info.Depth,
                    info.MipLevel);

            if (!ret)
            {
                return null;
            }

            return dstBytes;
        }

        private static unsafe Bitmap BytesToBitmap(byte[] dstBytes, int width, int height, int step, int srcIdxBase, bool isSnorm)
        {
            var dst = new Bitmap(width, height, PixelFormat.Format32bppArgb);
            var dstBitmapData = dst.LockBits(
                new Rectangle(0, 0, dst.Width, dst.Height),
                ImageLockMode.ReadWrite,
                dst.PixelFormat);
            var dstBmpPtr = (byte*)(void*)dstBitmapData.Scan0;
            var dstStride = dstBitmapData.Stride;
            //          for (var y = 0; y < height; y++)
            G3dParallel.For(0, height, y =>
            {
                var dstIdx = y * dstStride;
                var srcIdx = y * width * step + srcIdxBase;
                if (isSnorm)
                {
                    fixed (void* pBytes = &dstBytes[0])
                    {
                        var sbytes = (sbyte*)pBytes;
                        for (var x = 0; x < width; x++)
                        {
                            dstBmpPtr[dstIdx + 2] = (byte)((sbytes[srcIdx + 0] + 128) & 0xff); // R
                            dstBmpPtr[dstIdx + 1] = (byte)((sbytes[srcIdx + 1] + 128) & 0xff); // G
                            dstBmpPtr[dstIdx + 0] = (byte)((sbytes[srcIdx + 2] + 128) & 0xff); // B
                            dstBmpPtr[dstIdx + 3] = (byte)((sbytes[srcIdx + 3] + 128) & 0xff); // A
                            dstIdx += 4;
                            srcIdx += step;
                        }
                    }
                }
                else
                {
                    for (var x = 0; x < width; x++)
                    {
                        dstBmpPtr[dstIdx + 2] = dstBytes[srcIdx + 0]; // R
                        dstBmpPtr[dstIdx + 1] = dstBytes[srcIdx + 1]; // G
                        dstBmpPtr[dstIdx + 0] = dstBytes[srcIdx + 2]; // B
                        dstBmpPtr[dstIdx + 3] = dstBytes[srcIdx + 3]; // A
                        dstIdx += 4;
                        srcIdx += step;
                    }
                }
            });

            dst.UnlockBits(dstBitmapData);
            return dst;
        }

        private Bitmap[] ConvertToBitmaps(textureType texture, List<G3dStream> streams)
        {
            Ensure.Operation.True(IsConverterLoaded);
            Ensure.Argument.True(texture != null);
            Ensure.Argument.True(streams != null);

            if (!IsConverterLoaded || texture == null || streams == null)
            {
                return null;
            }

            var info = texture.texture_info;
            var srcBytes = streams[info.stream_index].ByteData.ToArray();
            var bitmaps = new Bitmap[info.mip_level];
            var isSnorm = IsSnormOrSint(info.quantize_type);
            var dstBytes = ConvertFormat(info, srcBytes);
            // デコードできなかった場合はOriginalImageを使う
            if (dstBytes == null)
            {
                return OriginalImageUtility.ConvertOriginalImageToBitmaps(texture, streams);
            }

            var srcIdxBase = 0;
            const int Step = 4;
            for (var i = 0; i < info.mip_level; i++)
            {
                var width = Math.Max(info.width >> i, 1);
                var height = Math.Max(info.height >> i, 1);

                bitmaps[i] = BytesToBitmap(dstBytes, width, height, Step, srcIdxBase, isSnorm);
                srcIdxBase += width * height * Step;
            }

            return bitmaps;
        }

        private Bitmap[] ConvertToBitmaps(Texture texture)
        {
            Ensure.Operation.True(IsConverterLoaded);
            Ensure.Argument.True(texture != null);

            if (!IsConverterLoaded || texture == null)
            {
                return null;
            }

            var info = texture.TextureInfo;
            var srcBytes = (info.Stream as StreamByte).Values.ToArray();
            var bitmaps = new Bitmap[info.MipLevel];
            var isSnorm = IsSnormOrSint(info.QuantizeType);
            var dstBytes = ConvertFormat(info, srcBytes);
            // デコードできなかった場合はOriginalImageを使う
            if (dstBytes == null)
            {
                return OriginalImageUtility.ConvertOriginalImageToBitmaps(texture);
            }

            var srcIdxBase = 0;
            const int Step = 4;
            for (var i = 0; i < info.MipLevel; i++)
            {
                var width = Math.Max(info.Width >> i, 1);
                var height = Math.Max(info.Height >> i, 1);

                bitmaps[i] = BytesToBitmap(dstBytes, width, height, Step, srcIdxBase, isSnorm);
                srcIdxBase += width * height * Step;
            }

            return bitmaps;
        }

        private Bitmap[][] ConvertToBitmapsArray(textureType texture, List<G3dStream> streams)
        {
            Ensure.Operation.True(IsConverterLoaded);
            Ensure.Argument.True(texture != null);
            Ensure.Argument.True(streams != null);

            if (!IsConverterLoaded || texture == null || streams == null)
            {
                return null;
            }

            var info = texture.texture_info;
            var srcBytes = streams[info.stream_index].ByteData.ToArray();
            var isSnorm = IsSnormOrSint(info.quantize_type);
            var dstBytes = ConvertFormat(info, srcBytes);
            // デコードできなかった場合はOriginalImageを使う
            if (dstBytes == null)
            {
                return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
            }

            var bitmapsArray = new Bitmap[info.depth][];
            for (var d = 0; d < info.depth; d++)
            {
                bitmapsArray[d] = new Bitmap[info.mip_level];
            }

            var srcIdxBase = 0;
            const int Step = 4;
            for (var i = 0; i < info.mip_level; i++)
            {
                var width = Math.Max(info.width >> i, 1);
                var height = Math.Max(info.height >> i, 1);
                var depth = (info.dimension == texture_info_dimensionType.Item3d) ? Math.Max(info.depth >> i, 1) : info.depth;
                for (var d = 0; d < depth; d++)
                {
                    int d2 = d;
                    switch (texture.texture_info.dimension)
                    {
                        case texture_info_dimensionType.cube:
                        case texture_info_dimensionType.cube_array:
                            // キューブマップのテクスチャーデータは右手座標系で +X、-X、+Y、-Y、-Z、+Z の順に格納されます。
                            // オリジナルイメージの順番にあわせて +X、-X、+Y、-Y、+Z、-Z の順にして読み込みます。
                            if (depth % 6 == 0)
                            {
                                switch (d % 6)
                                {
                                    case 4:
                                        d2 = d + 1;
                                        break;
                                    case 5:
                                        d2 = d - 1;
                                        break;
                                }
                            }
                            break;
                    }
                    bitmapsArray[d2][i] = BytesToBitmap(dstBytes, width, height, Step, srcIdxBase, isSnorm);
                    srcIdxBase += width * height * Step;
                }
            }
            return bitmapsArray;
        }

        private Bitmap[][] ConvertToBitmapsArray(Texture texture)
        {
            Ensure.Operation.True(IsConverterLoaded);
            Ensure.Argument.True(texture != null);

            if (!IsConverterLoaded || texture == null)
            {
                return null;
            }

            var info = texture.TextureInfo;
            var srcBytes = (info.Stream as StreamByte).Values.ToArray();
            var isSnorm = IsSnormOrSint(info.QuantizeType);
            var dstBytes = ConvertFormat(info, srcBytes);
            // デコードできなかった場合はOriginalImageを使う
            if (dstBytes == null)
            {
                return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
            }

            var bitmapsArray = new Bitmap[info.Depth][];
            for (var d = 0; d < info.Depth; d++)
            {
                bitmapsArray[d] = new Bitmap[info.MipLevel];
            }

            var srcIdxBase = 0;
            const int Step = 4;
            for (var i = 0; i < info.MipLevel; i++)
            {
                var width = Math.Max(info.Width >> i, 1);
                var height = Math.Max(info.Height >> i, 1);
                var depth = (info.Dimension == texture_info_dimensionType.Item3d) ? Math.Max(info.Depth >> i, 1) : info.Depth;
                for (var d = 0; d < depth; d++)
                {
                    int d2 = d;
                    switch (texture.TextureInfo.Dimension)
                    {
                        case texture_info_dimensionType.cube:
                        case texture_info_dimensionType.cube_array:
                            // キューブマップのテクスチャーデータは右手座標系で +X、-X、+Y、-Y、-Z、+Z の順に格納されます。
                            // オリジナルイメージの順番にあわせて +X、-X、+Y、-Y、+Z、-Z の順にして読み込みます。
                            if (depth % 6 == 0)
                            {
                                switch (d % 6)
                                {
                                    case 4:
                                        d2 = d + 1;
                                        break;
                                    case 5:
                                        d2 = d - 1;
                                        break;
                                }
                            }
                            break;
                    }
                    bitmapsArray[d2][i] = BytesToBitmap(dstBytes, width, height, Step, srcIdxBase, isSnorm);
                    srcIdxBase += width * height * Step;
                }
            }
            return bitmapsArray;
        }

        public Bitmap[] ConvertTo1d2dBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmaps(texture, streams);
        }

        public Bitmap[][] ConvertTo2dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsArray(texture, streams);
        }

        public Bitmap[][] ConvertTo3dBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsArray(texture, streams);
        }

        public Bitmap[][] ConvertToCubeArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsArray(texture, streams);
        }

        public Bitmap[][] ConvertToCubeBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsArray(texture, streams);
        }

        public Bitmap[][] ConvertTo1dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsArray(texture, streams);
        }

        private TextureData[][] ConvertToBitmapsStream(textureType texture, List<G3dStream> streams)
        {
            var bitmaps = ConvertToBitmapsArray(texture, streams);
            if (bitmaps == null)
            {
                return null;
            }
            var results = new TextureData[bitmaps.Length][];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                results[i] = new TextureData[bitmaps[i].Length];
                for (var j = 0; j < bitmaps[i].Length; ++j)
                {
                    var info = texture.texture_info;
                    results[i][j] = new TextureData
                    {
                        QuantizeType = info.quantize_type,
                        DimensionType = info.dimension,
                        Width = info.width,
                        Height = info.height,
                        Depth = info.depth,
                        IsFloat = false,
                        OriginalImage = bitmaps[i][j],
                        ByteImage = null,
                        FloatImage = null
                    };
                }
            }

            return results;
        }

        public TextureData[] ConvertTo1d2dStream(textureType texture, List<G3dStream> streams)
        {
            var bitmaps = ConvertTo1d2dBitmap(texture, streams);
            if (bitmaps == null)
            {
                return null;
            }
            var results = new TextureData[bitmaps.Length];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                var info = texture.texture_info;
                results[i] = new TextureData
                {
                    QuantizeType = info.quantize_type,
                    DimensionType = info.dimension,
                    Width = info.width,
                    Height = info.height,
                    Depth = info.depth,
                    IsFloat = false,
                    OriginalImage = bitmaps[i],
                    ByteImage = null,
                    FloatImage = null
                };
            }

            return results;
        }

        public TextureData[][] ConvertTo3dStream(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsStream(texture, streams);
        }

        public TextureData[][] ConvertToCubeStream(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsStream(texture, streams);
        }

        public TextureData[][] ConvertTo1dArrayStream(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsStream(texture, streams);
        }

        public TextureData[][] ConvertTo2dArrayStream(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsStream(texture, streams);
        }

        public TextureData[][] ConvertToCubeArrayStream(textureType texture, List<G3dStream> streams)
        {
            return ConvertToBitmapsStream(texture, streams);
        }


        public Bitmap[] ConvertTo1d2dBitmap(Texture texture)
        {
            return ConvertToBitmaps(texture);
        }

        public Bitmap[][] ConvertTo2dArrayBitmap(Texture texture)
        {
            return ConvertToBitmapsArray(texture);
        }

        public Bitmap[][] ConvertTo3dBitmap(Texture texture)
        {
            return ConvertToBitmapsArray(texture);
        }

        public Bitmap[][] ConvertToCubeArrayBitmap(Texture texture)
        {
            return ConvertToBitmapsArray(texture);
        }

        public Bitmap[][] ConvertToCubeBitmap(Texture texture)
        {
            return ConvertToBitmapsArray(texture);
        }

        public Bitmap[][] ConvertTo1dArrayBitmap(Texture texture)
        {
            return ConvertToBitmapsArray(texture);
        }

        private TextureData[][] ConvertToBitmapsStream(Texture texture)
        {
            var bitmaps = ConvertToBitmapsArray(texture);
            if (bitmaps == null)
            {
                return null;
            }
            var results = new TextureData[bitmaps.Length][];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                results[i] = new TextureData[bitmaps[i].Length];
                for (var j = 0; j < bitmaps[i].Length; ++j)
                {
                    var info = texture.TextureInfo;
                    results[i][j] = new TextureData
                    {
                        QuantizeType = info.QuantizeType,
                        DimensionType = info.Dimension,
                        Width = info.Width,
                        Height = info.Height,
                        Depth = info.Depth,
                        IsFloat = false,
                        OriginalImage = bitmaps[i][j],
                        ByteImage = null,
                        FloatImage = null
                    };
                }
            }

            return results;
        }

        public TextureData[] ConvertTo1d2dStream(Texture texture)
        {
            var bitmaps = ConvertTo1d2dBitmap(texture);
            if (bitmaps == null)
            {
                return null;
            }
            var results = new TextureData[bitmaps.Length];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                var info = texture.TextureInfo;
                results[i] = new TextureData
                {
                    QuantizeType = info.QuantizeType,
                    DimensionType = info.Dimension,
                    Width = info.Width,
                    Height = info.Height,
                    Depth = info.Depth,
                    IsFloat = false,
                    OriginalImage = bitmaps[i],
                    ByteImage = null,
                    FloatImage = null
                };
            }

            return results;
        }

        public TextureData[][] ConvertTo3dStream(Texture texture)
        {
            return ConvertToBitmapsStream(texture);
        }

        public TextureData[][] ConvertToCubeStream(Texture texture)
        {
            return ConvertToBitmapsStream(texture);
        }

        public TextureData[][] ConvertTo1dArrayStream(Texture texture)
        {
            return ConvertToBitmapsStream(texture);
        }

        public TextureData[][] ConvertTo2dArrayStream(Texture texture)
        {
            return ConvertToBitmapsStream(texture);
        }

        public TextureData[][] ConvertToCubeArrayStream(Texture texture)
        {
            return ConvertToBitmapsStream(texture);
        }
    }
}
