﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    internal class UniformGridNode<T>
    {
        internal ResizableList<T> Objects { get; set; } = new ResizableList<T>();
        internal BoundingBox BBox { get; set; } = new BoundingBox();
    }

    internal class UniformGrid<T>
    {
        protected int[] GridSize { get; set; } = new int[3];
        protected Vector3 UnitLen { get; set; } = new Vector3();
        protected Vector3 SlackLen { get; set; } = new Vector3();
        protected BoundingBox BoundingBox { get; set; } = new BoundingBox();

        protected Dictionary<Tuple<int, int, int>, UniformGridNode<T>> Node { get; set; } = new Dictionary<Tuple<int, int, int>, UniformGridNode<T>>();

        public BoundingBox BBox
        {
            get { return BoundingBox; }
            set { BoundingBox = value; }
        }

        public int X
        {
            get { return GridSize[0]; }
            set { GridSize[0] = value; }
        }
        public int Y
        {
            get { return GridSize[1]; }
            set { GridSize[1] = value; }
        }
        public int Z
        {
            get { return GridSize[2]; }
            set { GridSize[2] = value; }
        }

        /// <summary>
        /// 空間を初期化
        /// </summary>
        /// <param name="bb"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        public void Initialize(BoundingBox bb, int x, int y, int z, bool buildNode = true)
        {
            BoundingBox = bb;
            GridSize[0] = x;
            GridSize[1] = y;
            GridSize[2] = z;

            Vector3 size = BoundingBox.Size();

            UnitLen[0] = size[0] / (double)x;
            UnitLen[1] = size[1] / (double)y;
            UnitLen[2] = size[2] / (double)z;
            SlackLen = UnitLen * 0.1;

            if (buildNode)
            {
                for (int i = 0; i < x; i++)
                {
                    for (int j = 0; j < x; j++)
                    {
                        for (int k = 0; k < x; k++)
                        {
                            Tuple<int, int, int> key = Tuple.Create(i, j, k);

                            BoundingBox nodeBB = ComputeSpaceBB(i, j, k);
                            UniformGridNode<T> newNode = new UniformGridNode<T>();
                            newNode.BBox = nodeBB;
                            Node.Add(key, newNode);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 指定の位置がどの空間に所属しているかを判定します
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <param name="pos"></param>
        public void GetSpacePos(ref int x, ref int y, ref int z, Vector3 pos)
        {
            Vector3 localPos = pos - BoundingBox.Min;

            x = (int)(localPos.x / UnitLen.x);
            y = (int)(localPos.y / UnitLen.y);
            z = (int)(localPos.z / UnitLen.z);

            x = Math.Max(0, x);
            y = Math.Max(0, y);
            z = Math.Max(0, z);

            x = Math.Min(GridSize[0], x);
            y = Math.Min(GridSize[1], y);
            z = Math.Min(GridSize[2], z);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <returns></returns>
        public BoundingBox ComputeSpaceBB(int x, int y, int z)
        {
            if (x < 0 || x >= GridSize[0])
            {
                throw ExcepHandle.CreateException("Size x is illegal");
            }
            if (y < 0 || y >= GridSize[1])
            {
                throw ExcepHandle.CreateException("Size y is illegal");
            }
            if (z < 0 || z >= GridSize[2])
            {
                throw ExcepHandle.CreateException("Size z is illegal");
            }

            BoundingBox result = new BoundingBox();

            result.Min[0] = BoundingBox.Min[0];
            result.Min[1] = BoundingBox.Min[1];
            result.Min[2] = BoundingBox.Min[2];

            result.Min[0] += UnitLen[0] * x;
            result.Min[1] += UnitLen[1] * y;
            result.Min[2] += UnitLen[2] * z;

            result.Max = result.Min + UnitLen;

            result.Min -= SlackLen;
            result.Max += SlackLen;

            return result;
        }

        /// <summary>
        /// 指定のノードのBBoxを取得
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <returns></returns>
        public BoundingBox GetSpaceBB(int x, int y, int z)
        {
            if (x < 0 || x >= GridSize[0])
            {
                throw ExcepHandle.CreateException("Size x is illegal");
            }
            if (y < 0 || y >= GridSize[1])
            {
                throw ExcepHandle.CreateException("Size y is illegal");
            }
            if (z < 0 || z >= GridSize[2])
            {
                throw ExcepHandle.CreateException("Size z is illegal");
            }

            UniformGridNode<T> node = Node[Tuple.Create(x, y, z)];
            if (node == null)
            {
                throw ExcepHandle.CreateException("key does not exist");
            }
            return node.BBox;
        }

        /// <summary>
        /// Slack領域まで含まて含まれる
        /// </summary>
        /// <param name="spaceList"></param>
        /// <param name="pos"></param>
        public void GetSpacePos(List<int[]> spaceList, Vector3 pos)
        {
            if (spaceList == null)
            {
                throw ExcepHandle.CreateException("spaceList is null");
            }

            int[] center = new int[3];
            GetSpacePos(ref center[0], ref center[1], ref center[2], pos);
            BoundingBox bb = GetSpaceBB(center[0], center[1], center[2]);

            spaceList.Add(center);

            Vector3 v0 = pos - bb.Min;
            Vector3 v1 = bb.Max - pos;

            //8 近傍に対して含まれているかどうか？をチェックする
            if (v0.x < SlackLen.x)
            {
                spaceList.Add(new int[] { center[0] - 1, center[1], center[2] });
            }
            if (v1.x < SlackLen.x)
            {
                spaceList.Add(new int[] { center[0] + 1, center[1], center[2] });
            }

            if (v0.y < SlackLen.y)
            {
                spaceList.Add(new int[] { center[0], center[1] - 1, center[2] });
            }
            if (v1.y < SlackLen.y)
            {
                spaceList.Add(new int[] { center[0], center[1] + 1, center[2] });
            }

            if (v0.z < SlackLen.z)
            {
                spaceList.Add(new int[] { center[0], center[1], center[2] - 1 });
            }
            if (v1.z < SlackLen.z)
            {
                spaceList.Add(new int[] { center[0], center[1], center[2] + 1 });
            }
        }

        /// <summary>
        /// オブジェクトを追加します。
        /// </summary>
        /// <param name="obj"></param>
        /// <param name="pos"></param>
        public void AddObject(T obj, Vector3 pos)
        {
            List<int[]> spaceList = new List<int[]>();
            GetSpacePos(spaceList, pos);

            foreach (var intse in spaceList)
            {
                if (intse.Length < 3)
                {
                    throw ExcepHandle.CreateException("index count is shorter than 3.");
                }
                Tuple<int, int, int> key = Tuple.Create(intse[0], intse[1], intse[2]);

                if (Node.ContainsKey(key))
                {
                    if (!Node[key].Objects.Exists(v => v.Equals(obj)))
                    {
                        Node[key].Objects.Add(obj);
                    }
                }
            }
        }

        /// <summary>
        /// 指定のBounding Boxで、空間リストをクエリします
        /// </summary>
        /// <param name="arvSpaces"></param>
        /// <param name="min"></param>
        /// <param name="max"></param>
        /// <returns></returns>
        public bool WindowQueryForBB(ResizableList<UniformGridNode<T>> arvSpaces, Vector3 min, Vector3 max)
        {
            BoundingBox bb = new BoundingBox();
            bb.Update(min);
            bb.Update(max);
            foreach (var node in Node)
            {
                if (node.Value.BBox.Intersect(bb))
                {
                    arvSpaces.Add(node.Value);
                }
            }
            return true;
        }
    }
}
