﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;

// ReSharper disable PossibleIntendedRethrow
// ReSharper disable TooWideLocalVariableScope
// ReSharper disable RedundantAssignment
// ReSharper disable SuggestUseVarKeywordEvident
// ReSharper disable CompareOfFloatsByEqualityOperator
// ReSharper disable RedundantCast

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        /// <summary>
        /// 指定のウェッジでの、異なるインデックスのプロパティ間の
        /// 最大の距離を取得する。
        /// </summary>
        /// <returns> 距離</returns>
        private double ComputePropDist(WedgeVertex wv, int index)
        {
            double dist = 0;

            SourceType type = wv.Keys[index].type;
            string name = wv.Keys[index].name;
            MeshSourceBase source = mMesh.GetSource(type, name);

            int count = wv.Props[index].Count;

            // UV のみで
            if (type == SourceType.Texture)
            {
                for (int i = 0; i < count - 1; i++)
                {
                    int prid_i = wv.Props[index][i];
                    Vector2 vct_i = new Vector2();
                    MeshSourceDblCtrler.GetAsVec2(source, prid_i, ref vct_i);

                    for (int j = i; j < count; j++)
                    {
                        int prid_j = wv.Props[index][j];
                        Vector2 vct_j = new Vector2();
                        MeshSourceDblCtrler.GetAsVec2(source, prid_j, ref vct_j);

                        dist = Math.Max((vct_i - vct_j).Norm(), dist);
                    }
                }
            }
            return dist;
        }

        /// <summary>
        /// 各エッジに接する頂点フラグを設定
        /// </summary>
        /// <param name="rBoder"></param>
        private void ComputeVertexFlagsFromEdge(List<int[]> rBoder)
        {
            var count = rBoder.Count;
            //開放端を固定化
            foreach (WingedEdge we in WingEdedgeMesh.RepresentativeEdges)
            {
                if (we.Boder)
                {
                    //新パラメータが有効の場合
                    if (LockOpenEdgeAngle > 0.0)
                    {
                        var borders0 = GetNeighborOpenEdges(we, 0);
                        var angle = 0.0;

                        for (int j = 0; j < borders0.Count; j++)
                        {
                            angle = Math.Max(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders0[j], we));
                        }

                        if (angle <= LockOpenEdgeAngle && LockOpenEdgeAngle <= 180.0)
                        {
                            for (int i = 0; i < count; i++)
                            {
                                rBoder[i][we.Vertex0] |= (int)QemVertexType.Corner;
                            }
                        }

                        var borders1 = GetNeighborOpenEdges(we, 1);
                        angle = 0.0;
                        //新パラメータが有効になっている場合
                        if (LockOpenEdgeAngle > 0)
                        {
                            for (int i = 0; i < borders1.Count; i++)
                            {
                                angle = Math.Max(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders1[i], we));
                            }

                            if (angle <= LockOpenEdgeAngle && LockOpenEdgeAngle <= 180.0)
                            {
                                for (int i = 0; i < count; i++)
                                {
                                    rBoder[i][we.Vertex1] |= (int)QemVertexType.Corner;
                                }
                            }
                        }
                    }
                    // 旧パラメータ有効
                    else
                    {
                        var borders0 = GetNeighborOpenEdges(we, 0);
                        var angle = 0.0;
                        for (int j = 0; j < borders0.Count; j++)
                        {
                            angle = Math.Max(angle, WingedEdgeMesh.Old_ComputeAngle(WingEdedgeMesh, borders0[j], we));
                        }

                        if (angle >= OldLockOpenEdgeAngle && OldLockOpenEdgeAngle < 90.0)
                        {
                            for (int i = 0; i < count; i++)
                            {
                                rBoder[i][we.Vertex0] |= (int)QemVertexType.Corner;
                            }
                        }

                        var borders1 = GetNeighborOpenEdges(we, 0);
                        angle = 0.0;
                        // 旧パラメータ有効
                        for (int j = 0; j < borders1.Count; j++)
                        {
                            angle = Math.Max(angle, WingedEdgeMesh.Old_ComputeAngle(WingEdedgeMesh, borders1[j], we));
                        }

                        if (angle >= OldLockOpenEdgeAngle && OldLockOpenEdgeAngle < 90.0)
                        {
                            for (int i = 0; i < count; i++)
                            {
                                rBoder[i][we.Vertex1] |= (int)QemVertexType.Corner;
                            }
                        }
                    }

                    for (int i = 0; i < count; i++)
                    {
                        rBoder[i][we.Vertex0] |= (int)QemVertexType.OpenEdge;
                        rBoder[i][we.Vertex1] |= (int)QemVertexType.OpenEdge;
                    }
                }
                else
                {
                    if (IsSubmeshBorderEdge(we))
                    {
                        for (int i = 0; i < count; i++)
                        {
                            rBoder[i][we.Vertex0] |= (int)QemVertexType.SubMesh;
                            rBoder[i][we.Vertex1] |= (int)QemVertexType.SubMesh;
                        }
                    }
                    if (IsShapeBorderEdge(we))
                    {
                        for (int i = 0; i < count; i++)
                        {
                            rBoder[i][we.Vertex0] |= (int)QemVertexType.ShapeBdr;
                            rBoder[i][we.Vertex1] |= (int)QemVertexType.ShapeBdr;
                        }
                    }
                }
                var ci = 0;
                foreach (var cep in we.ChildEdges)
                {
                    foreach (var ce in cep.Value)
                    {
                        if (ce.IsOpen)
                        {
                            if (cep.Key.type == SourceType.Texture)
                            {
                                var uv0 = ce.Vertex0;
                                var uv1 = ce.Vertex1;
                                var pv0 = we.Vertex0;
                                var pv1 = we.Vertex1;

                                WingEdedgeMesh.GetOrderedChildIndices(ce, ref uv0, ref uv1);
                                //スワップしてる
                                if (uv0 == ce.Vertex1)
                                {
                                    pv0 = we.Vertex1;
                                    pv1 = we.Vertex0;
                                }
                                var borders0 = GetNeighborOpenEdges(ce, 0);
                                var angle = 0.0;

                                if (LockUvHardEdgeAngle > 0.0)
                                {
                                    for (int i = 0; i < borders0.Count; i++)
                                    {
                                        angle = Math.Max(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders0[i], ce));
                                    }

                                    if (angle <= LockUvHardEdgeAngle && LockUvHardEdgeAngle <= 180.0)
                                    {
                                        rBoder[ci][pv0] |= (int)QemVertexType.UvCorner;
                                    }

                                    var borders1 = GetNeighborOpenEdges(ce, 1);
                                    angle = 0.0;
                                    for (int i = 0; i < borders1.Count; i++)
                                    {
                                        angle = Math.Max(angle, WingedEdgeMesh.ComputeAngle(WingEdedgeMesh, borders1[i], ce));
                                    }
                                    if (angle <= LockUvHardEdgeAngle && LockUvHardEdgeAngle <= 180.0)
                                    {
                                        rBoder[ci][pv1] |= (int)QemVertexType.UvCorner;
                                    }
                                }
                                else
                                {
                                    for (int i = 0; i < borders0.Count; i++)
                                    {
                                        angle = Math.Max(angle, WingedEdgeMesh.Old_ComputeAngle(WingEdedgeMesh, borders0[i], ce));
                                    }

                                    if (angle >= OldLockUvHardEdgeAngle && OldLockUvHardEdgeAngle < 90.0)
                                    {
                                        rBoder[ci][pv0] |= (int)QemVertexType.UvCorner;
                                    }

                                    var borders1 = GetNeighborOpenEdges(ce, 1);
                                    angle = 0.0;
                                    for (int i = 0; i < borders1.Count; i++)
                                    {
                                        angle = Math.Max(angle, WingedEdgeMesh.Old_ComputeAngle(WingEdedgeMesh, borders1[i], ce));
                                    }
                                    if (angle >= OldLockUvHardEdgeAngle && OldLockUvHardEdgeAngle < 90.0)
                                    {
                                        rBoder[ci][pv1] |= (int)QemVertexType.UvCorner;
                                    }
                                }
                                rBoder[ci][pv0] |= (int)QemVertexType.UVEdge;
                                rBoder[ci][pv1] |= (int)QemVertexType.UVEdge;
                            }
                        }
                    }
                    ci++;
                }
            }
        }

        /// <summary>
        /// UVハードエッジの数を数える
        /// </summary>
        /// <returns></returns>
        private int ComputeUvHardEdgeCount()
        {
            int count = 0;
            foreach (WingedEdge we in WingEdedgeMesh.RepresentativeEdges)
            {
                foreach (var cep in we.ChildEdges)
                {
                    if ( cep.Key.type == SourceType.Texture &&
                         cep.Value.Count() >= 2 &&
                         cep.Value.All(o => o.IsOpen))
                    {
                        count++;
                    }
                }
            }
            return count;
        }

        /// <summary>
        /// 非多様体エッジに接する頂点を検出
        /// </summary>
        /// <param name="rBoder"></param>
        private void ComputeVertexFlagsFromNonmanifoldEdge(List<int[]> rBoder)
        {
            int sn = mMesh.SourceN;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);

            foreach (var sourceKey in WingEdedgeMesh.SourceKeys)
            {
                var uvNmfdEdges = WingEdedgeMesh.GetNonmanifoldEdges(sourceKey.type, sourceKey.name);
                foreach (WingedEdge we in uvNmfdEdges)
                {
                    var edge = we;
                    if (we.SpaceKey.type != SourceType.Position)
                    {
                        edge = we.ParentEdges.First();
                    }
                    foreach (var border in rBoder)
                    {
                        border[edge.Vertex0] |= (int)QemVertexType.Nonmanifold;
                        border[edge.Vertex1] |= (int)QemVertexType.Nonmanifold;
                    }
                }

                var wingedEdges = WingEdedgeMesh.GetWingedEdges(sourceKey.type, sourceKey.name);
                Dictionary<int, int> hitCount = new Dictionary<int, int>();
                Dictionary<int, List<int>> vtxmap = new Dictionary<int, List<int>>();
                foreach (WingedEdge we in wingedEdges)
                {
                    WingedEdge parent = null;
                    if (we.SpaceKey.type != SourceType.Position)
                    {
                        parent = we.ParentEdges.First();
                    }

                    var edge = we;
                    if (edge.IsOpen)
                    {
                        if (!hitCount.ContainsKey(edge.Vertex0))
                        {
                            hitCount.Add(edge.Vertex0, 0);
                            vtxmap.Add(edge.Vertex0, new List<int>());
                        }
                        if (!hitCount.ContainsKey(edge.Vertex1))
                        {
                            hitCount.Add(edge.Vertex1, 0);
                            vtxmap.Add(edge.Vertex1, new List<int>());
                        }

                        hitCount[edge.Vertex0]++;
                        hitCount[edge.Vertex1]++;
                        if (parent != null)
                        {
                            int uv0 = 0;
                            int uv1 = 0;
                            WingEdedgeMesh.GetOrderedChildIndices(we, ref uv0, ref uv1);
                            vtxmap[uv0].Add(parent.Vertex0);
                            vtxmap[uv1].Add(parent.Vertex1);
                        }
                    }
                }

                var nmfd = hitCount.Where(o => o.Value > 2);

                foreach (var kvp in nmfd)
                {
                    foreach (var border in rBoder)
                    {
                        var posIds = vtxmap[kvp.Key].Distinct();
                        foreach (var vid in posIds)
                        {
                            border[vid] |= (int)QemVertexType.Nonmanifold;
                        }
                    }
                }
            }

            //非多様体頂点を検出
            for (int vi = 0; vi < mMesh.VertexN; vi++)
            {
                int nonmaniCnt = 0;
                int opneCnt = 0;
                foreach (var fi in VtxTriMesh.VTL[vi].TriList)
                {
                    int vn = mMesh.VNums[fi];
                    int vo = mMesh.VOffset[fi];
                    int error = 0;
                    for (int vj = 0; vj < vn; vj++)
                    {
                        int mvid = mMesh.mVbuffs[(vo + vj) * sn + po];
                        if (mvid != vi)
                        {
                            List<int> list = VtxTriMesh.GetNeighborTrigs(fi, vi, mvid);
                            if (list.Count == 0)
                            {
                                error++;
                            }
                        }
                    }
                    if (error == 2)
                    {
                        nonmaniCnt++;
                    }
                    if (error == 1)
                    {
                        opneCnt++;
                    }
                }

                // 平面の角
                if (VtxTriMesh.VTL[vi].TriList.Count == 1 && nonmaniCnt == 1)
                {
                    foreach (var border in rBoder)
                    {
                        border[vi] |= (int)QemVertexType.Corner;
                    }
                }

                // 扇を考える
                if (opneCnt >= 3 || nonmaniCnt >= 2)
                {
                    foreach (var border in rBoder)
                    {
                        border[vi] |= (int)QemVertexType.Nonmanifold;
                    }
                }
            }
        }

        /// <summary>
        /// 除外メッシュリストから、
        /// </summary>
        /// <param name="rBoder"></param>
        private void ComputeVertexFlagFromExcludeMeshes(List<int[]> rBoder)
        {
            var faceN = mMesh.FaceN;
            var faceGroupId = mMesh.FaceMaterialShapeId;

            var sourceN = mMesh.SourceN;
            var posIndexOffset = mMesh.GetSourceOfset(SourceType.Position, null);
            ExcludeFaceNum = 0;
            // 各ポリゴンの頂点を除外する
            // 除外フラグを設定する
            for (int i = 0; i < faceN; i++)
            {
                var fgid = faceGroupId[i];
                var shapeInfo = mMesh.MaterialShapeInformationList.Find(o => o.Index == fgid);

                // 除外対象に含まれる場合は、簡略化からは除外
                if (shapeInfo != null && shapeInfo.IsExcludeForSimplification)
                {
                    ExcludeFaceNum++;
                    var vtxOffset = mMesh.VOffset[i];
                    var vtxNum = mMesh.VNums[i];

                    // vi 頂点カウンター
                    for (var vi = 0; vi < vtxNum; vi++)
                    {
                        var vid = mMesh.VBuffer[(vi + vtxOffset) * sourceN + posIndexOffset];
                        foreach (var border in rBoder)
                        {
                            border[vid] |= (int)QemVertexType.Exclude;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 頂点ごとの属性フラグを計算する
        /// 不動な頂点などが決まる
        /// </summary>
        /// <param name="rBoder">境界線上の頂点かどうかのフラグ</param>
        /// <param name="wingm">ウイングドエッジ</param>
        /// <param name="meshWedge">ウェッジ</param>
        private void ComputeVertexFlag(ref List<int[]> rBoder)
        {
            // 各頂点が境界線上かどうか？を検出します
            int vtxN = mMesh.VertexN;
            if (rBoder == null)
            {
                var uvN = 0;
                foreach (SourceType type in Enum.GetValues(typeof(SourceType)))
                {
                    uvN = Math.Max(mMesh.GetSources(type).Count, uvN);
                }
                rBoder = new List<int[]>();
                for (int i = 0; i < uvN; i++)
                {
                    rBoder.Add(new int[vtxN]);
                    for (int j = 0; j < vtxN; j++)
                    {
                        rBoder[i][j] = (int)QemVertexType.Inner;
                    }
                }
            }
            if (OptPosPolicy == QemOptPos.ZeroOne)
            {
                PropBorderCheckByIndex = true;
            }
            ComputeVertexFlagsFromEdge(rBoder);
            ComputeVertexFlagsFromNonmanifoldEdge(rBoder);

            if (FeatureFaceAngle > 0)
            {
                ComputeVertexFlagsForFeature(rBoder);
            }
            else
            {
                Old_ComputeVertexFlagsForFeature(rBoder);
            }

            // 除外リストを更新する
            ComputeVertexFlagFromExcludeMeshes(rBoder);
        }

        /// <summary>
        /// 頂点周辺の面の法線から、特徴的な角を検出する
        /// </summary>
        /// <param name="rBoder"></param>
        private void Old_ComputeVertexFlagsForFeature(List<int[]> rBoder)
        {
            for (int vi = 0; vi < mMesh.VertexN; vi++)
            {
                var vtl = VtxTriMesh.VTL[vi];
                var vnormal = new Vector3();
                foreach (var faceId in vtl.TriList)
                {
                    vnormal += mMesh.mFaceNormal[faceId] * mMesh.mFaceArea[faceId];
                }
                vnormal.Normalize();
                var maxAngle = 0.0;
                foreach (var faceId in vtl.TriList)
                {
                    var angle = Math.Acos(vnormal.Dot(mMesh.mFaceNormal[faceId]));
                    angle = (angle / Math.PI) * 180.0;

                    maxAngle = Math.Max(angle, maxAngle);
                }
                // 特徴的な形状のコーナー
                if (maxAngle * 2 > OldFeatureFaceAngle)
                {
                    foreach (var border in rBoder)
                    {
                        border[vi] |= (int)QemVertexType.FeatEdge;
                    }
                }
            }
        }

        /// <summary>
        /// 頂点周辺の面の法線から、特徴的な角を検出する
        /// </summary>
        /// <param name="rBoder"></param>
        private void ComputeVertexFlagsForFeature(List<int[]> rBoder)
        {
            ResizableList<WingedEdge> edges = WingEdedgeMesh.RepresentativeEdges;
            // 各エッジのコストを計算
            // 各頂点での最小コストのエッジを計算
            foreach (var edge in edges)
            {
                if (edge.IsOpen)
                {
                    continue;
                }

                var angle = Math.Acos(mMesh.mFaceNormal[edge.Face1].Dot(mMesh.mFaceNormal[edge.Face0]));
                // 特徴的な形状のコーナー
                if ((180 - angle) <= FeatureFaceAngle)
                {
                    foreach (var border in rBoder)
                    {
                        if (edge.Vertex0 != int.MaxValue && edge.Vertex0 >= 0)
                        {
                            border[edge.Vertex0] |= (int)QemVertexType.FeatEdge;
                        }
                        if (edge.Vertex1 != int.MaxValue && edge.Vertex1 >= 0)
                        {
                            border[edge.Vertex1] |= (int)QemVertexType.FeatEdge;
                        }
                    }
                }
            }

            // 角の様に尖った個所を検出する為に、周辺平均をとって検査
            for (int vi = 0; vi < mMesh.VertexN; vi++)
            {
                var vtl = VtxTriMesh.VTL[vi];
                var vnormal = new Vector3();
                foreach (var faceId in vtl.TriList)
                {
                    vnormal += mMesh.mFaceNormal[faceId] * mMesh.mFaceArea[faceId];
                }
                vnormal.Normalize();
                var maxAngle = 0.0;
                foreach (var faceId in vtl.TriList)
                {
                    var angle = Math.Acos(vnormal.Dot(mMesh.mFaceNormal[faceId]));
                    angle = (angle / Math.PI) * 180.0;

                    maxAngle = Math.Max(angle, maxAngle);
                }
                maxAngle = Math.Max(maxAngle, 0);
                maxAngle = Math.Min(maxAngle * 2, 180);

                // 特徴的な形状のコーナー
                if (FeatureFaceAngle > 0)
                {
                    if ((180 - maxAngle * 2) <= FeatureFaceAngle)
                    {
                        foreach (var border in rBoder)
                        {
                            border[vi] |= (int)QemVertexType.FeatEdge;
                        }
                    }
                }
                else
                {
                    if (maxAngle * 2 > OldFeatureFaceAngle)
                    {
                        foreach (var border in rBoder)
                        {
                            border[vi] |= (int)QemVertexType.FeatEdge;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 指定頂点周辺から、コストを計算する
        /// </summary>
        /// <param name="vtmesh"></param>
        /// <param name="keep"></param>
        /// <returns></returns>
        private double ComputeQuadWeight(VtxTriMesh vtmesh, int keep)
        {
            if (!IsRequiredComputingWeight)
            {
                return 1.0;
            }

            double result = 0.0f;
            if (IsBlendWeightEnabled)
            {
                // BlendWeight を用いた計算から最大重みを取得する
                result = ComputeMotionWeight(vtmesh, keep);
            }
            else
            {
#if OLDQUAD
                int  cnt = vtmesh.VTL[keep].triList.Count;
                for (int i = 0; i < cnt; i++)
                {
                    int fid = vtmesh.VTL[keep].triList[i];
                    result += ComputeQuadWeight(fid);
                }
#else
                result = 1;
#endif
            }
            return result;
        }

        /// <summary>
        /// 各頂点ごとのQuadricを計算します
        /// </summary>
        /// <param name="vtri">頂点インデックス</param>
        /// <param name="vid">頂点ID</param>
        /// <param name="plist">座標（参照）</param>
        private void ComputeVertexQuadrics(VtxTriMesh vtri, int vid, ref VectorN[] plist)
        {
            int num = 0;
            int fid = 0;

            num = vtri.VTL[vid].TriList.Count;
            if (num == 0) { return; }
            if (mQuads[vid] == null)
            {
                mQuads[vid] = new Quadric(3, DistOffset);
            }
            else
            {
                mQuads[vid].Clear();
            }

            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            int sn = mMesh.SourceN;
            Vector3 tmp = new Vector3();
            for (int f = 0; f < num; f++)
            {
                _tmpQ.Clear();
                fid = vtri.VTL[vid].TriList[f];

                plist[0].SetZero();
                plist[1].SetZero();
                plist[2].SetZero();
                int vo = mMesh.mVoffs[fid];
                for (int vi = 0; vi < 3; vi++)
                {
                    int vj = mMesh.mVbuffs[(vo + vi) * sn + po];
                    mMesh.GetVertexPos(vj, ref tmp);
                    plist[vi][0] = tmp[0];
                    plist[vi][1] = tmp[1];
                    plist[vi][2] = tmp[2];
                }
                _tmpQ.DistOffset = DistOffset;
                _tmpQ.Compute(plist[0], plist[1], plist[2]);

                mQuads[vid] += _tmpQ;
            }
        }

        /// <summary>
        /// ＵＶ境界のコストを求める
        /// </summary>
        /// <param name="we"></param>
        /// <returns></returns>
        private BoundaryMetric ComputeUVBoundaryMetric(WingedEdge we)
        {
            List<WingedEdge> neighbors = new List<WingedEdge>();
            if (we.IsOpen)
            {
                neighbors.Add(we);
            }

            ResizableList<VtxWnEdge> winged_edges = WingEdedgeMesh.GetVtxWEList(we.SpaceKey.type, we.SpaceKey.name);
            foreach (var edge in winged_edges[we.Vertex0].EdgeList)
            {
                if (edge.IsOpen && !neighbors.Contains(edge))
                {
                    neighbors.Add(edge);
                }
            }

            foreach (var edge in winged_edges[we.Vertex1].EdgeList)
            {
                if (edge.IsOpen && !neighbors.Contains(edge))
                {
                    neighbors.Add(edge);
                }
            }
            if (neighbors.Count == 0)
            { return null; }

            BoundaryMetric result = new BoundaryMetric(3);
            result.Clear();
            MeshSourceBase poslist = mMesh.GetSource(we.SpaceKey.type, we.SpaceKey.name);
            foreach (var wingedEdge in neighbors)
            {
                BoundaryMetric tmp = new BoundaryMetric(3);
                Vector2 _tp0 = new Vector2();
                Vector2 _tp1 = new Vector2();
                MeshSourceDblCtrler.GetAsVec2(poslist, wingedEdge.Vertex0, ref _tp0);
                MeshSourceDblCtrler.GetAsVec2(poslist, wingedEdge.Vertex1, ref _tp1);

                Vector3 tp0 = new Vector3();
                Vector3 tp1 = new Vector3();

                tp0[0] = _tp0[0];
                tp0[1] = _tp0[1];
                tp0[2] = 0.0;

                tp1[0] = _tp1[0];
                tp1[1] = _tp1[1];
                tp1[2] = 0.0;

                tmp.Compute(tp0, tp1);
                result += tmp;
            }
            return result;
        }

        /// <summary>
        /// オープンエッジに仮想的な平面を立てる事で、コストが増大するように仕向ける
        /// </summary>
        /// <param name="we"></param>
        /// <param name="plist"></param>
        private void ComputeVirtualPlaneQuadris(WingedEdge we, ref VectorN[] plist)
        {
            if (we.Face0 == int.MaxValue || we.Face1 == int.MaxValue)
            {
                //IsOnEdgeBreak(we, 187, 984, true);

                Vector3 v0 = new Vector3();
                Vector3 v1 = new Vector3();
                int sn = mMesh.SourceN;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);
                int no = mMesh.GetSourceOfset(SourceType.Normal, null);
                MeshSourceBase poslist = mMesh.GetSource(SourceType.Position, null);

                int fid = we.Face0 == int.MaxValue ? we.Face1 : we.Face0;

                plist[0].SetZero();
                plist[1].SetZero();
                plist[2].SetZero();
                GetFaceVertices(fid, ref plist);

                // エッジの両端以外の頂点を取得します
                MeshSourceDblCtrler.GetAsVec3(poslist, we.Vertex0, ref v0);
                MeshSourceDblCtrler.GetAsVec3(poslist, we.Vertex1, ref v1);
                Vector3 e = v1 - v0;
                Vector3 n = mMesh.mFaceNormal[fid];
                int vo = mMesh.mVoffs[fid];
                int vindex = int.MaxValue;
                for (int vi = 0; vi < 3; vi++)
                {
                    int vid = mMesh.mVbuffs[(vo + vi) * sn];
                    if (we.Vertex0 != vid && we.Vertex1 != vid)
                    {
                        vindex = vi;
                    }
                }

                // エッジの端点以外の頂点を法線方向に移動して、平面を構成
                if (vindex != int.MaxValue)
                {
                    VectorN vp0 = new VectorN((uint)3);
                    VectorN vp1 = new VectorN((uint)3);
                    VectorN vp2 = new VectorN((uint)3);
                    VectorN vp3 = new VectorN((uint)3);

                    VectorN p0 = new VectorN((uint)3);
                    VectorN p1 = new VectorN((uint)3);

                    // 仮平面を立てる
                    int pi0 = (vindex + 1) % 3;
                    int pi1 = (vindex + 2) % 3;
                    Vector3 d = n * e.Norm() * 3;
                    Vector3 n2 = d.Cross(e);
                    n2.Normalize();

                    vp0.Set(plist[pi0]);
                    vp0[po + 0] += d[0];
                    vp0[po + 1] += d[1];
                    vp0[po + 2] += d[2];

                    vp2.Set(plist[pi0]);
                    vp2[po + 0] -= d[0];
                    vp2[po + 1] -= d[1];
                    vp2[po + 2] -= d[2];

                    vp1.Set(plist[pi1]);
                    vp1[po + 0] += d[0];
                    vp1[po + 1] += d[1];
                    vp1[po + 2] += d[2];

                    vp3.Set(plist[pi1]);
                    vp3[po + 0] -= d[0];
                    vp3[po + 1] -= d[1];
                    vp3[po + 2] -= d[2];

                    p0.Set(plist[pi0]);
                    p1.Set(plist[pi1]);
                    // 一枚目
                    {
                        _tmpQ.Clear();
                        _tmpQ.DistOffset = DistOffset;
                        _tmpQ.Compute(p0, p1, vp0);

                        mQuads[we.Vertex0] = mQuads[we.Vertex0] + _tmpQ;
                        mQuads[we.Vertex1] = mQuads[we.Vertex1] + _tmpQ;
                    }

                    // 二枚目
                    {
                        _tmpQ.Clear();
                        _tmpQ.DistOffset = DistOffset;
                        _tmpQ.Compute(vp0, p1, vp1);
                        mQuads[we.Vertex0] = mQuads[we.Vertex0] + _tmpQ;
                        mQuads[we.Vertex1] = mQuads[we.Vertex1] + _tmpQ;
                    }
                    //検査
                    if (mQuads[we.Vertex0].IsNaN() || mQuads[we.Vertex1].IsNaN())
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"Quadrics is NaN.");
                    }
                }
            }
        }

        /// <summary>
        /// Quadrics を計算
        /// </summary>
        /// <param name="vtri"></param>
        private void ComputeQuadrics(VtxTriMesh vtri, WingedEdgeMesh wem)
        {
            try
            {
                int posN = mMesh.VertexN;

                mQuads.Resize(posN, null);
                // Faceについて、Quadricを計算する
                VectorN[] plist = new VectorN[3];
                for (int v = 0; v < 3; v++)
                {
                    plist[v] = new VectorN((uint)3);
                }
                _tmpQ = new Quadric(3, DistOffset);
                for (int v = 0; v < posN; v++)
                {
                    ComputeVertexQuadrics(vtri, v, ref plist);
                }
                // 境界線について架空のフェースを作る
                foreach (var we in wem.RepresentativeEdges)
                {
                    ComputeVirtualPlaneQuadris(we, ref plist);
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        /// <summary>
        /// オリジナルＵＶをコスト演算の為に蓄える
        /// </summary>
        private void StockOriginalUV()
        {
            List<MeshSourceBase> list = mMesh.GetSources(SourceType.Texture);

            foreach (var sourceBase in list)
            {
                mOrigUV.Add(sourceBase.Key, new ResizableList<Vector2>());

                int count = sourceBase.Count;
                mOrigUV[sourceBase.Key].Resize(count, null);

                Vector2 tmp = new Vector2();
                for (int i = 0; i < count; i++)
                {
                    mOrigUV[sourceBase.Key][i] = new Vector2();
                    MeshSourceDblCtrler.GetAsVec2(sourceBase, i, ref tmp);
                    mOrigUV[sourceBase.Key][i].Set(tmp);
                }
            }
        }

        /// <summary>
        /// プロパティの重複などをなくす写像を生成し、変換を行う
        /// </summary>
        private void CreatePropMaps(bool cleanUp)
        {
            List<MeshSourceBase> texList = mMesh.GetSources(SourceType.Texture);
            foreach (var meshSourceBase in texList)
            {
                PropMapTaransform pmap = new PropMapTaransform();
                pmap.Run(mMesh, meshSourceBase.Type, meshSourceBase.Name, cleanUp);
                propMaps.Add(pmap);
            }
        }

        /// <summary>
        /// サブメッシュ境界に接している面数を、連結Shapeリストごとに求める。
        /// </summary>
        private void CountUpSubmeshBorderFaces(int[] vertexBorder)
        {
            if (vertexBorder == null) { return; }

            var edges = WingEdedgeMesh.RepresentativeEdges;

            // サブメッシュの面数を取得します
            mMesh.GetSubMeshCount();

            var visted = new bool[mMesh.FaceN];
            Array.Clear(visted, 0, mMesh.FaceN);
            foreach (var edge in edges)
            {
                if (IsSubmeshBorderEdge(edge))
                {
                    if (edge.Face0 != int.MaxValue && !visted[edge.Face0])
                    {
                        visted[edge.Face0] = true;
                    }
                    if (edge.Face1 != int.MaxValue && !visted[edge.Face1])
                    {
                        visted[edge.Face1] = true;
                    }
                }
            }
        }

        /// <summary>
        /// 頂点が、簡略化可能な状態であるかどうかを取得する
        /// </summary>
        /// <returns></returns>
        public List<int[]> GetVertexProfile()
        {
            List<int[]> vertexState = null;
            PreExecute(ref vertexState);
            return vertexState;
        }

        /// <summary>
        /// 実行前に必要な処理
        /// </summary>
        private void PreExecute(ref List<int[]> vertexborder)
        {
            try
            {
#if DEBUG
                if (OptPosPolicy == QemOptPos.ZeroOne)
                {
                    MeshIndexTrack mit = new MeshIndexTrack();
                    mit.Run(mCloneMesh, mMesh, true);
                }
#endif
                // プロパティを扱いやすくする
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 1, 7));
                CreatePropMaps(OptPosPolicy != QemOptPos.ZeroOne);
                // 連結メッシュ単位に分割する
                mMesh.GetConnectedShapeFaceCounter();
                mMesh.GetMaterialShapeFaceCounter();
                OriginalMaterialShapeFaceCount = mMesh._MaterialShapeFaceCounter.ToArray();

                // 連結形状ごとの簡略化設定を作成
                // 設定されてない場合は、全体で設定された値を使用する
                var maxIndex = mMesh.MaterialShapeInformationList.Max(o => o.Index);
                RatePerAllMhapes = new float[ maxIndex + 1];
                Array.Clear(RatePerAllMhapes, 0, maxIndex + 1);
                for (int index = 0; index < mMesh.MaterialShapeInformationList.Count; index++)
                {
                    var msInfo = mMesh.MaterialShapeInformationList[index];
                    // 無設定の場合は、相対化をしないために限界まで削る
                    RatePerAllMhapes[msInfo.Index] = 0.0f;
                    foreach (var kvp in TargetRatePerMaterialShape)
                    {
                        if (Regex.IsMatch(msInfo.ShapeName, kvp.Key + QemSimp.MeshNameSymbol + "[0-9]*"))
                        {
                            RatePerAllMhapes[msInfo.Index] = kvp.Value;
                        }
                    }
                }

                // 隣接フェースリストを作成
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 2, 7));
                VtxTriMesh tmpVtm = null;
                CreateVTriInfo(ref tmpVtm);
                VtxTriMesh = tmpVtm;

                // Wedgeを作成します。
                // ハードエッジの検出を行います。
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 3, 7));
                MeshWedges tmpWedge = null;
                CreateWedge(ref tmpWedge);
                WedgeMesh = tmpWedge;

                // 最外周エッジの検出と、WingedEdgeの作成を行う
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 4, 7));
                WingedEdgeMesh tmpWingedEdge = null;
                CreateEdgeInfo(ref tmpWingedEdge);
                WingEdedgeMesh = tmpWingedEdge;

                // UVハードエッジが存在する＝アトラスが存在する
                // にもかかわらず、UVフラグが未セットの場合には警告メッセージを出力して、フラグを強制セット
                var uvheCnt = ComputeUvHardEdgeCount();
                if (uvheCnt > 0 && !IsUVEnabled)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\n" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_UV_AtlasError_Notify", SourceMesh.MeshName));
                    PropFlag |= (int)QemUseProp.TEX;
                    CheckProperties();
                }

                // 頂点の種別を決定
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 5, 7));
                ComputeVertexFlag(ref vertexborder);
                CountUpSubmeshBorderFaces(vertexborder.First());
                // Quadric を初期計算する
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 6, 7));
                ComputeQuadrics(VtxTriMesh, WingEdedgeMesh);
                // 元のＵＶを保存します
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\r" + _3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Init_Proc_Notify", 7, 7));
                StockOriginalUV();
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"..");
#if DEBUG
                var cornerCount = vertexborder[0].ToList().Count(o => (o & (int)QemVertexType.Corner) > 0);
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine($"Corner : {cornerCount}");
                var uvcrnrCount = vertexborder[0].ToList().Count(o => (o & (int)QemVertexType.UvCorner) > 0);
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine($"UvCorner : {uvcrnrCount}");
                var openCount = vertexborder[0].ToList().Count(o => (o & (int)QemVertexType.OpenEdge) > 0);
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine($"OpenEdge : {openCount}");
#endif
                var nmfdCount = vertexborder[0].ToList().Count(o => (o & (int)QemVertexType.Nonmanifold) > 0);
                var submeshCount = vertexborder[0].ToList().Count(o => (o & (int)QemVertexType.SubMesh) > 0);

                if (nmfdCount > 0)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Nonmanifold_Notify", nmfdCount, mMesh.VertexN));
                }

                if (submeshCount > 0)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_SubMeshBorder_Notify", submeshCount, mMesh.VertexN));
                }

                if (nmfdCount > 0 || submeshCount > 0)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Limited_Simplify_Notify"));
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }
    }
}

// ReSharper restore RedundantCast
// ReSharper restore CompareOfFloatsByEqualityOperator
// ReSharper restore SuggestUseVarKeywordEvident
// ReSharper restore RedundantAssignment
// ReSharper restore TooWideLocalVariableScope
// ReSharper restore PossibleIntendedRethrow
