﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text.RegularExpressions;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public partial class QemSimp
    {
        public const string MeshNameSymbol = "#mid";
        internal WingedEdgeMesh WingEdedgeMesh { get; set; } = null;
        internal VtxTriMesh VtxTriMesh { get; set; } = null;
        internal MeshWedges WedgeMesh { get; set; } = null;
        private ResizableList<PropMapTaransform> propMaps = new ResizableList<PropMapTaransform>();

        private Mesh mMesh = null;
        private ResizableList<bool> mPropuse = new ResizableList<bool>();
        private ResizableList<Quadric> mQuads = new ResizableList<Quadric>();
        private Dictionary<SourceKey, ResizableList<Vector2>> mOrigUV = new Dictionary<SourceKey, ResizableList<Vector2>>(); // ＵＶコストを計算する為のリスト
        internal BucketHeapList<QemHeapData> Heap { get; set; } = new BucketHeapList<QemHeapData>();
        private float[] RatePerAllMhapes;
        private int[] OriginalMaterialShapeFaceCount { get; set; } = null;

        /// 除外する面の数（簡略化可能なポリゴン数に影響がある）
        private int ExcludeFaceNum = 0;

        public double MaxCost { get; private set; }

        public double MinCost { get; private set; }

        private Mesh mCloneMesh = null;

        private VectorN _tmpUsedV0 = null;
        private VectorN _tmpUsedV1 = null;
        private VectorN _tmpFullV0 = null;
        private VectorN _tmpFullV1 = null;
        private Quadric _tmpQ;
        private ResizableList<Vector3> _faceAngle;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public QemSimp()
        {
            //パラメーター更新
            Params = new QemSimpParam();
        }

        /// <summary>
        /// 計算に必要なメモリ量を計算します
        /// </summary>
        /// <returns></returns>
        public ulong ComputeRequiredMemSize()
        {
            throw new NotImplementedException("opaque function");
        }

        /// <summary>
        /// 再利用すべき一時的なオブジェクトを作成します
        /// </summary>
        private void CreateTempObject()
        {
            _tmpUsedV0 = new VectorN((uint)DataDimension);
            _tmpUsedV1 = new VectorN((uint)DataDimension);
            _tmpFullV0 = new VectorN((uint)mMesh.GetPropDimension());
            _tmpFullV1 = new VectorN((uint)mMesh.GetPropDimension());
        }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="mesh">リダクションを行いたいメッシュ</param>
        /// <returns>成功した場合はtrue</returns>
        public bool Initialize(Mesh mesh)
        {
            try
            {
                if (mesh.FaceN == 0)
                {
                    return false;
                }
                //頂点だけは必ず使う
                PropFlag = (int)QemUseProp.VERT | PropFlag;
                mMesh = mesh;
                // 除外シェイプの更新
                SetUpExcludeShape();

                mMesh.FacePropsCountEqualityTest();

                mCloneMesh = mesh.Clone();
                mCloneMesh.ComputeBounds();

                mesh.ReComputeSubMeshArray(true);
                // 境界ボリュームの計算を行います
                mesh.ComputeBounds();

                mMesh.RecomputeFaceNormalsAndArea();

                ReComputeFaceAngles();
                CheckProperties();

                // 面積を考慮するか、どうか？
                Quadric.AreaWeighted = (Policy == QemPolicy.AreaBased || Policy == QemPolicy.AreaAngleBased);
                // 計算用一時オブジェクトの作成
                CreateTempObject();
                return true;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex);
            }
        }

        private void SetUpExcludeShape()
        {
            var shapeInfos = mMesh.MaterialShapeInformationList;

            // 除外情報を更新する
            for (var i = 0; i < shapeInfos.Count; i++)
            {
                shapeInfos[i].IsExcludeForSimplification = false;
                // パターンマッチするかどうか？
                if (ExcludeMeshNames.Any(pattern => Regex.IsMatch(shapeInfos[i].ShapeName, pattern + MeshNameSymbol + "[0-9]*")))
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Exclude_Message", shapeInfos[i].ShapeName));
                    shapeInfos[i].IsExcludeForSimplification = true;
                }
            }
        }

        /// <summary>
        /// 面積を一辺の長さの二乗で割る事で、形状の正三角形への近さを測る
        /// 正三角形の時が最大、面積がゼロで最少
        /// </summary>
        /// <param name="p0"></param>
        /// <param name="p1"></param>
        /// <param name="p2"></param>
        /// <returns></returns>
        private double FaceQuality(Vector3 p0, Vector3 p1, Vector3 p2)
        {
            Vector3 d10 = p1 - p0;
            Vector3 d20 = p2 - p0;
            Vector3 d12 = p1 - p2;
            Vector3 x = d10.Cross(d20);

            double a = x.Norm();
            if (a == 0)
            {
                return 0;
            }
            double b = d10.Dot(d10);
            if (b == 0)
            {
                return 0;
            }
            double t = b;
            t = d20.Dot(d20);
            if (b < t)
            {
                b = t;
            }
            t = d12.Dot(d12);
            if (b < t)
            {
                b = t;
            }
            return a / b;
        }

        /// <summary>
        /// 各面の角度を計算
        /// </summary>
        /// <param name="fi"></param>
        private void ComputeFaceAngles(int fi)
        {
            if (Policy != QemPolicy.AngleBased && Policy != QemPolicy.AreaAngleBased) { return; }

            int vn = 0;
            int vo = 0;
            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            MeshSource<double> pos = mMesh.GetSource(SourceType.Position, null) as MeshSource<double>;
            if (pos == null)
            {
                throw ExcepHandle.CreateException("Positions are not found in mesh");
            }

            Vector3[] p = new Vector3[3];
            {
                vn = mMesh.mVnums[fi];
                vo = mMesh.mVoffs[fi];
                if (vn != 3)
                {
                    throw ExcepHandle.CreateException("ReComputeFaceAngles:: mesh have polyhedron (over 3vtx)");
                }

                for (int i = 0; i < 3; i++)
                {
                    int vid = mMesh.mVbuffs[(vo + i) * sn + po];
                    p[i] = new Vector3();
                    MeshSourceDblCtrler.GetAsVec3(pos, vid, ref p[i]);
                }

                Vector3 pos1 = p[0];
                Vector3 pos2 = p[1];
                Vector3 pos3 = p[2];

                Vector3 te1 = pos2 - pos1;
                Vector3 te2 = pos3 - pos1;
                Vector3 te3 = pos3 - pos2;

                double l1 = te1.Norm();
                double l2 = te2.Norm();
                double l3 = te3.Norm();
                _faceAngle[fi].x = te1.Dot(te2) / (l1 * l2);
                _faceAngle[fi].y = te3.Dot(-te1) / (l3 * l1);
                _faceAngle[fi].z = te2.Dot(-te3) / (l2 * l3);
            }
        }

        /// <summary>
        /// フェース毎の角度を保存
        /// </summary>
        private void ReComputeFaceAngles()
        {
            if (Policy != QemPolicy.AngleBased && Policy != QemPolicy.AreaAngleBased) { return; }

            _faceAngle = new ResizableList<Vector3>();
            _faceAngle.Resize(mMesh.FaceN, null);
            int vn = 0;
            MeshSource<double> pos = mMesh.GetSource(SourceType.Position, null) as MeshSource<double>;
            if (pos == null)
            {
                throw ExcepHandle.CreateException("The value type of position stream is not double/float.");
            }

            for (int fi = 0; fi < mMesh.FaceN; fi++)
            {
                vn = mMesh.mVnums[fi];
                if (vn != 3)
                {
                    throw ExcepHandle.CreateException("ReComputeFaceAngles:: mesh have polyhedron (over 3vtx).");
                }

                _faceAngle[fi] = new Vector3();
                _faceAngle[fi].SetZero();

                ComputeFaceAngles(fi);
            }
        }

        /// <summary>
        /// フェース毎の角度を隣接フェースについて再計算
        /// </summary>
        private void ReComputeFaceAngles(VtxTriMesh vtm, int vid)
        {
            if (Policy != QemPolicy.AngleBased && Policy != QemPolicy.AreaAngleBased) { return; }

            // 角度を計算
            foreach (int afid in vtm.VTL[vid].TriList)
            {
                ComputeFaceAngles(afid);
            }
        }

        /// <summary>
        /// 各ソース種別を考慮するか？どうかフラグを設定して、
        /// データ全体の次元数を計算する
        /// </summary>
        private void CheckProperties()
        {
            if (SourceMesh == null)
            {
                throw ExcepHandle.CreateException("Source mesh is null.");
            }

            try
            {
                List<MeshSourceBase> list = null;
                DataDimension = 0;
                mPropuse.Resize(Enum.GetNames(typeof(SourceType)).Length, false);
                //---------------------------------------
                list = SourceMesh.GetSources(SourceType.Position);
                if (list == null)
                {
                    throw ExcepHandle.CreateException(
                        "CheckProperties :: for lack of position data, reduction process will be stopped.");
                }

                int posid = (int)SourceType.Position;
                mPropuse[posid] = ((PropFlag & (int)QemUseProp.VERT) != 0);

                // 頂点座標が複数ある場合はリダクションを実行しない
                if (list.Count != 1)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Merge_Vertex", "Reduction"));
                    DataDimension = 0;
                    return;
                }
                if (PropUse[posid])
                {
                    DataDimension += 3;
                }

                //---------------------------------------
                list = SourceMesh.GetSources(SourceType.Texture);
                if (list != null)
                {
                    mPropuse[(int)SourceType.Texture] = ((PropFlag & (int)QemUseProp.TEX) != 0);

                    // テクスチャ有効の場合は再計算も有効にする
                    if (mPropuse[(int)SourceType.Texture])
                    {
                        RecalcTangentBasis = true;
                    }

                    foreach (MeshSourceBase source in list)
                    {
                        if (PropUse[(int)SourceType.Texture])
                        {
                            DataDimension += source.Stride;
                        }
                    }
                }

#if VERBOSE
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(string.Format("Data Dimension : {0}", DataDimension));
#endif
            }
            catch
            {
                throw;
            }
        }

        /// <summary>
        /// 各頂点の価数と、隣接フェースを取得する
        /// </summary>
        /// <param name="vtMesh"> 隣接フェース情報　</param>
        private void CreateVTriInfo(ref VtxTriMesh vtMesh)
        {
            vtMesh = new VtxTriMesh();
            vtMesh.Create(mMesh);
        }
        /// <summary>
        /// Wedge構造を作成して、ハードエッジの検出を行います
        /// </summary>
        /// <param name="meshWedges"> Wedgeデータ　</param>
        private void CreateWedge(ref MeshWedges meshWedges)
        {
            meshWedges = new MeshWedges(mMesh);
            meshWedges.Create();
        }
        /// <summary>
        /// WingedEdgeを作成して、境界線上の頂点を決定する
        /// </summary>
        /// <param name="meshWEdges"> WingedEdgeオブジェクトへの参照　</param>
        private void CreateEdgeInfo(ref WingedEdgeMesh meshWEdges)
        {
            MeshSourceBase source = mMesh.GetSource(SourceType.Position, null);

            meshWEdges = new WingedEdgeMesh();
            meshWEdges.SetRepresentativeKey(source.Key.type, source.Key.name);
            meshWEdges.CreateAll(mMesh);
            // エッジ間参照を作成する
            meshWEdges.BuildEdgeReference();
        }
    }
}
