﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public partial class Mesh
    {
        public List<int> CoutUpUsedPropFG(SourceType type, string name, int fg = -1)
        {
            List<int> result = new List<int>();

            MeshSourceBase source = GetSource(type, name);
            int so = GetSourceOfset(type, name);
            int sn = SourceN;
            for (int fi = 0; fi < FaceN; fi++)
            {
                if (fg == -1 || fg == int.MaxValue || fg == _FaceMaterialShapeID[fi])
                {
                    int vo = mVoffs[fi];
                    for (int vi = 0; vi < 3; vi++)
                    {
                        int vid = mVbuffs[(vo + vi) * sn + so];
                        if (result.FindIndex((vj) => vj == vid) == -1)
                        {
                            result.Add(vid);
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// システムのテンポラリフォルダにファイル出力を行う
        /// </summary>
        /// <param name="filename"></param>
        [Conditional("LOCAL_TEST")]
        public void DebugOutput(string filename)
        {
            string tmpPath = System.IO.Path.GetTempPath();
            if (!string.IsNullOrEmpty(tmpPath))
            {
                ObjImporter.SaveMeshToObj(this, tmpPath + @"\" + filename);
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine("Debug Output To " + tmpPath + @"\" + filename);
            }
        }

        /// <summary>
        /// ソースメッシュに同じ名前のものがある場合は、致命的なエラー
        /// </summary>
        [Conditional("DEBUG")]
        internal void FacePropsCountEqualityTest()
        {
            if (mFaceNormal.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }
            if (mFaceColor.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }
            if (mFaceInvalid.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }

            if (mFaceSubMeshID.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }
            if (_FaceMaterialShapeID.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }
            if (mFaceArea.Count != FaceN)
            {
                throw new Exception("Face Props Count equality error.");
            }
        }

        /// <summary>
        /// 各プロパティのカウントが、頂点のカウントに一致していることを保証します
        /// </summary>
        [Conditional("DEBUG")]
        internal void PropsCountEqualityTest()
        {
            MeshSourceBase pos = GetSource(SourceType.Position, null);
            if (pos == null)
            {
                throw new Exception("Mesh doesnt have position");
            }

            foreach (var meshSourceBase in Sources)
            {
                if (meshSourceBase == pos) { continue; }
                if (meshSourceBase.Count != pos.Count)
                {
                    throw new Exception(string.Empty);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        [Conditional("DEBUG")]
        public void CheckSubMeshArray()
        {
            foreach (var node in mSubMeshInfo)
            {
                SubMeshInfo subMeshInfo = node.Value;
                if (SubMeshInfo.IsNull(subMeshInfo))
                {
                    continue;
                }
                int test = subMeshInfo.subMeshOffset + subMeshInfo.subMeshCount;
                if (test > FaceN)
                {
                    throw new Exception("caould not construct legal submesh info");
                }
                if (subMeshInfo.subMeshCount == 0)
                {
                    throw new Exception("illegal submesh have no face exist.");
                }
            }

            // サブメッシュIDが、ちゃんと順番にならんでいるか？
            // SubMeshInfo Array が崩壊してないか？
            List<int> cloneFaceSMId = new List<int>(mFaceSubMeshID);
            cloneFaceSMId.Sort();
            for (int i = 0; i < cloneFaceSMId.Count; i++)
            {
                if (!mSubMeshInfo.ContainsKey(mFaceSubMeshID[i]))
                {
                    throw new Exception("mesh don have submesh info " + mFaceSubMeshID[i].ToString());
                }
                if (cloneFaceSMId[i] != mFaceSubMeshID[i])
                {
                    throw new Exception("SubmeshId dont line up sequentially.");
                }
            }
        }

        [Conditional("DEBUG")]
        public void SystPrintBounds()
        {
            foreach (var boundingBox in mBBoxes)
            {
                Vector3 min = boundingBox.Value.Min;
                Vector3 max = boundingBox.Value.Max;
                Debug.WriteLine("BB{3} Min x:{0} y:{1} z:{2}", min.x, min.y, min.z, boundingBox.Key);
                Debug.WriteLine("BB{3} Max x:{0} y:{1} z:{2}", max.x, max.y, max.z, boundingBox.Key);
            }
            foreach (var boundingSquare in mUVBSquares)
            {
                int i = 0;
                foreach (var uvSquare in boundingSquare.Value)
                {
                    Vector2 min = uvSquare.Min;
                    Vector2 max = uvSquare.Max;
                    Debug.WriteLine("BB{2}[{3}] Min x:{0} y:{1}", min.x, min.y, boundingSquare.Key, i);
                    Debug.WriteLine("BB{2}[{3}] Max x:{0} y:{1}", max.x, max.y, boundingSquare.Key, i);
                    i++;
                }
            }
        }

        /// <summary>
        /// デバッグ時のみバウンディングを表示
        /// </summary>
        [Conditional("VERVOSE")]
        public void PrintBoundingBox()
        {
            foreach (var boundingBox in mBBoxes)
            {
                BoundingBox bb = boundingBox.Value;
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine($"POS Bound[{bb.Min.x}, {bb.Min.y}, {bb.Min.z}] [{bb.Max.x}, {bb.Max.y}, {bb.Max.z}]");
            }
        }

        /// <summary>
        /// 指定した２つのメッシュが完全に一致するかどうか？を調べます
        /// </summary>
        /// <param name="lm"></param>
        /// <param name="rm"></param>
        [Conditional("DEBUG")]
        public static void CheckMeshSourceEquality(Mesh lm, Mesh rm)
        {
            // 各プロパティの数を比較する
            if (lm.FaceN != rm.FaceN) { throw new Exception("face count is different"); }
            if (lm.VertexN != rm.VertexN) { throw new Exception("face count is different"); }
            if (lm.SourceN != rm.SourceN) { throw new Exception("source count is different"); }

            // ソースの数などを確認します
            foreach (var leftSource in lm.Sources)
            {
                MeshSourceBase rightSource = rm.GetSource(leftSource.Type, leftSource.Name);
                if (rightSource == null)
                {
                    string msg = string.Format(" source[{0}] dont find in right mesh", leftSource.Name);
                    throw new Exception(msg);
                }
                // 要素数が異なる
                if (leftSource.Count != rightSource.Count)
                {
                    throw new Exception("source count is different : " + leftSource.Name);
                }

                // 各値の比較
                for (int i = 0; i < leftSource.RowCount; i++)
                {
                    bool hit = false;
                    for (int j = 0; j < rightSource.RowCount; j++)
                    {
                        if (leftSource.GetRowValue(i).Equals(rightSource.GetRowValue(j)))
                        {
                            hit = true;
                            break;
                        }
                    }
                    if (!hit)
                    {
                        string msg = string.Format("source[{0}] value[{1}] is different", leftSource.Name, i);
                        throw new Exception(msg);
                    }
                }
            }
        }

        /// <summary>
        /// 全頂点において、各プロパティ毎の隣接フェースインデックスが一致している事を
        /// チェックすることが可能な関数
        /// </summary>
        /// <param name="mesh"></param>
        [Conditional("DEBUG")]
        public static void CheckSharedPropRegularity(Mesh mesh, bool chkByIndex)
        {
            var vtm = new VtxTriMesh();
            var wdg = new MeshWedges(mesh);

            vtm.Create(mesh);
            wdg.Create();
            var tmpList = new List<List<int>>();
            foreach (var wv in wdg.Wedges)
            {
                int sizeN = wv.Keys.Count;
                //複数プロパティがない場合は無視してもよろしい
                if (sizeN <= 1) { continue; }

                int vid = wv.Vertex;
                int propIndexN = wv.Props[0].Count;
                for (int pi = 0; pi < propIndexN; pi++)
                {
                    int testIndex = wv.Props[0][pi];
                    bool hit = true;
                    tmpList.Clear();
                    for (int si = 0; si < sizeN; si++)
                    {
                        int sid = wv.Props[si][pi];
                        if (testIndex != sid)
                        {
                            hit = false;
                        }

                        List<int> flist = null;
                        if (chkByIndex)
                        {
                            flist = vtm.GetPindexSharedFaceIndices(vid, wv.Keys[si].type, wv.Keys[si].name, sid);
                        }
                        else
                        {
                            flist = vtm.GetPdistSharedFaceIndices(vid, wv.Keys[si].type, wv.Keys[si].name, sid);
                        }
                        tmpList.Add(flist);
                    }

                    if (hit)
                    {
                        for (int si = 1; si < sizeN; si++)
                        {
                            if (tmpList[0].Count != tmpList[si].Count)
                            {
                                throw new Exception("a wedge has illegal property.");
                            }
                        }
                    }
                }
            }
        }
    }
}
