﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class BoundingBox
    {
        private Vector3 mMin = new Vector3();
        private Vector3 mMax = new Vector3();

        /// <summary>
        /// 最小座標
        /// </summary>
        public Vector3 Min
        {
            get { return mMin; }
            set { mMin = value; }
        }

        /// <summary>
        /// 最大座標
        /// </summary>
        public Vector3 Max
        {
            get { return mMax; }
            set { mMax = value; }
        }

        public BoundingBox()
        {
            Reset();
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Reset()
        {
            mMin.x = mMin.y = mMin.z = double.MaxValue;
            mMax.x = mMax.y = mMax.z = -double.MaxValue;
        }

        /// <summary>
        /// AABBの対角ベクトル
        /// </summary>
        /// <returns></returns>
        public Vector3 Size()
        {
            return mMax - mMin;
        }

        /// <summary>
        /// 非数チェッカ
        /// </summary>
        /// <returns></returns>
        public bool IsNaN()
        {
            for (int i = 0; i < 3; i++)
            {
                if (double.IsNaN(mMin[i]) || double.IsNaN(mMax[i]))
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 正当な値が入っているか？をチェックする
        /// </summary>
        /// <returns></returns>
        public bool IsValid()
        {
            for (int i = 0; i < 3; i++)
            {
                if (double.IsNaN(mMin[i]) || double.IsNaN(mMax[i]))
                {
                    return false;
                }
                if (double.IsInfinity(mMin[i]) || double.IsInfinity((mMax[i])))
                {
                    return false;
                }
                if (mMin[i] == double.MaxValue || mMax[i] == double.MinValue)
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// AABB同士の交差判定を行います
        /// </summary>
        /// <param name="bBox">テスト対象</param>
        /// <returns></returns>
        public bool Intersect(BoundingBox bBox)
        {
            return
            (
                mMax.x >= bBox.mMin.x &&
                mMin.x <= bBox.mMax.x &&
                mMax.y >= bBox.mMin.y &&
                mMin.y <= bBox.mMax.y &&
                mMax.z >= bBox.mMin.z &&
                mMin.z <= bBox.mMax.z);
        }

        /// <summary>
        /// AABBが完全に中に入っているか？
        /// </summary>
        /// <param name="bBox"></param>
        /// <returns></returns>
        public bool Inside(BoundingBox bBox)
        {
            return
            (
                mMax.x >= bBox.mMax.x &&
                mMin.x <= bBox.mMin.x &&
                mMax.y >= bBox.mMax.y &&
                mMin.y <= bBox.mMin.y &&
                mMax.z >= bBox.mMax.z &&
                mMin.z <= bBox.mMin.z);
        }

        /// <summary>
        /// 境界ボリュームを更新
        /// </summary>
        /// <param name="pos"></param>
        public void Update(Vector3 pos)
        {
            if (pos == null)
            {
                return;
            }
            mMin.x = Math.Min(pos.x, mMin.x);
            mMin.y = Math.Min(pos.y, mMin.y);
            mMin.z = Math.Min(pos.z, mMin.z);

            mMax.x = Math.Max(pos.x, mMax.x);
            mMax.y = Math.Max(pos.y, mMax.y);
            mMax.z = Math.Max(pos.z, mMax.z);
        }

        /// <summary>
        /// 指定倍拡大
        /// </summary>
        /// <param name="factor"></param>
        public void Stretch(double factor)
        {
            Vector3 size = Size();
            mMin -= size * (factor * 0.5);
            mMax += size * (factor * 0.5);
        }
    }
}
