﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.toollib.Ext;
using nw.g3d.nw4f_3dif;
using System.IO;
using System.Diagnostics;

namespace nw.g3d.optcvtr
{
    // スケルタルアニメオプティマイザ
    internal class SkeletalAnimOptimizer : Optimizer
    {
        // コンストラクタ
        internal SkeletalAnimOptimizer(g3doptcvtrParams fileOption)
            : base(fileOption)
        {
            // マテリアル圧縮
            CheckUnsupportedOption(fileOption, () => fileOption.CompressMaterial);

            // リジッドスキン変換
            CheckUnsupportedOption(fileOption, () => fileOption.RigidBodyToRigidSkinning);

            // 刈り込みボーン圧縮
            if (fileOption.CompressBoneCull)
            {
                this.Compressors.Add(new IfSkeletalAnimBoneCullCompressor());
            }

            // マージボーン圧縮
            if (fileOption.CompressBoneMerge)
            {
                this.Compressors.Add(new IfSkeletalAnimBoneMergeCompressor());
            }

            // 親子合体ボーン圧縮
            if (fileOption.CompressBoneUniteChild)
            {
                this.Compressors.Add(new IfSkeletalAnimBoneUniteChildCompressor());
            }

            // 合体ボーン圧縮
            if (fileOption.CompressBoneUnite)
            {
                this.Compressors.Add(new IfSkeletalAnimBoneUniteCompressor());
            }

            // 完全合体ボーン圧縮
            if (fileOption.CompressBoneUniteAll)
            {
                this.Compressors.Add(new IfSkeletalAnimBoneUniteAllCompressor());
            }

            // シェイプ圧縮
            CheckUnsupportedOption(fileOption, () => fileOption.CompressShape);

#if false
            // 中間ファイルの修正はデバッグ目的のみ使用
            ArgumentOption repair = fileOption.GetOption("--repair");
            if (repair != null)
            {
                RepairFile = true;
            }

            // 中間ファイルのチェックはデバッグ目的のみ使用
            ArgumentOption check = fileOption.GetOption("--check_stream");
            if (check != null)
            {
                CheckFile = true;
            }
#endif

            // マージ
            if (fileOption.Merge)
            {
                if (fileOption.MergeOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.MergeOptions));
                }

                var mergeOptions = fileOption.MergeOptions;

                // マージファイルの設定
                if (!File.Exists(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_OptionFileNotFound", mergeOptions.MergeFile);
                }
                if (!G3dPath.IsSkeletalAnimPath(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_Merge_InvalidFileType", mergeOptions.MergeFile);
                }

                this.MergeFilePath = mergeOptions.MergeFile;
            }

            // 同一頂点の削除
            CheckUnsupportedOption(fileOption, () => fileOption.DeleteNearVertex);

            // サブメッシュ分割
            CheckUnsupportedOption(fileOption, () => fileOption.UniteSubmesh);
            CheckUnsupportedOption(fileOption, () => fileOption.DivideSubmesh);

            // PolygonReduction
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReduction);
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReductionOptions);

            // プリミティブ最適化
            CheckUnsupportedOption(fileOption, () => fileOption.OptimizePrimitive);

            // LOD モデル結合・削除
            CheckUnsupportedOption(fileOption, () => fileOption.UniteLodModel);
            CheckUnsupportedOption(fileOption, () => fileOption.TrimLodModel);

            // マテリアルアニメ統合
            CheckUnsupportedOption(fileOption, () => fileOption.Combine);
            CheckUnsupportedOption(fileOption, () => fileOption.CombineOptions);

            // 量子化分析
            if (fileOption.QuantizationAnalysis)
            {
                this.Optimizers.Add(new IfSkeletalAnimQuantizationAnalysisOptimizer());
            }
        }

        //=====================================================================
        // 最適化
        internal override void Optimize()
        {
            // 読み込み
            Read();
            skeletal_animType skeletal_anim = this.nw4f_3dif.Item as skeletal_animType;

            // スケルタルアニメ圧縮
            foreach (IfSkeletalAnimCompressor compressor in this.Compressors)
            {
                compressor.Compress(skeletal_anim, this.Streams);

                // 最適化ログの追加
                IfProcessLogUtility.Add(this.nw4f_3dif,
                    compressor.Process, compressor.Argument, compressor.GetResult());
            }

            // データの正当性チェック
            Check(skeletal_anim);

            // 修正
            Repair(skeletal_anim);

            // マージ
            Merge(skeletal_anim);

            // スケルタルアニメ最適化
            foreach (IfSkeletalAnimOptimizer optimizer in this.Optimizers)
            {
                optimizer.Optimize(skeletal_anim, this.Streams);

                // 最適化ログの追加
                if (optimizer.EnableProcessLog)
                {
                    IfProcessLogUtility.Add(this.nw4f_3dif,
                        optimizer.Process, optimizer.Argument, optimizer.GetResult());
                }
            }

            IfProcessLogUtility.ToDistinctLatest(this.nw4f_3dif);

            // 書き出し
            Write();
        }

        //---------------------------------------------------------------------
        // 確認
        private void Check(skeletal_animType skeletalAnim)
        {
            if (!CheckFile) { return; }

            // ストリームインデックスの正当性を確認する
            string checkLog = string.Empty;
            bool checkResult =
                StreamChecker.CheckStreamIndex(out checkLog, skeletalAnim, this.Streams);
            // チェックが失敗したらログを例外として投げて表示させる
            if (!checkResult)
            {
                throw new Exception(checkLog);
            }
            else
            {
                Console.WriteLine("ok.");
            }
        }

        //---------------------------------------------------------------------
        // 修正
        private void Repair(skeletal_animType skeletalAnim)
        {
            if (!RepairFile) { return; }

            StreamUtility.SortStream(skeletalAnim, this.Streams);
        }

        //---------------------------------------------------------------------
        // マージ
        private void Merge(skeletal_animType newSkeletalAnim)
        {
            if (this.MergeFilePath == null) { return; }

            this.MergeStopwatch.Start();

            // マージファイルの読み込み
            List<G3dStream> oldStreams = new List<G3dStream>();
            nw4f_3difType oldIf = IfReadUtility.Read(
                oldStreams, this.MergeFilePath, this.XsdBasePath);
            skeletal_animType oldSkeletalAnim = oldIf.Item as skeletal_animType;

            // マージ
            IfSkeletalAnimMerger.Merge(newSkeletalAnim, this.Streams, oldSkeletalAnim, oldStreams);

            // 最適化ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "merge",
                Path.GetFullPath(this.MergeFilePath).Replace('\\', '/'), string.Empty); /// TODO: Result 対応

            this.MergeStopwatch.Stop();
        }

        //=====================================================================
        // ログの取得
        internal override string GetLog()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(base.GetLog());
            builder.AppendLine(GetReadLog());

            // スケルタルアニメ圧縮
            foreach (IfSkeletalAnimCompressor compressor in this.Compressors)
            {
                builder.AppendLine("    " + compressor.ToString());
            }

            // マージ
            if (this.MergeFilePath != null)
            {
                builder.AppendLine(Strings.Get("MergeLog",
                    this.MergeStopwatch.ElapsedMilliseconds,
                    this.MergeFilePath));
            }

            // スケルタルアニメ最適化
            foreach (IfSkeletalAnimOptimizer optimizer in this.Optimizers)
            {
                builder.AppendLine("    " + optimizer.ToString());
            }

            builder.Append(GetWriteLog());
            return builder.ToString();
        }

        //=====================================================================
        private List<IfSkeletalAnimCompressor> Compressors =
            new List<IfSkeletalAnimCompressor>();
        private readonly string MergeFilePath;
        private readonly bool RepairFile = false;
        private readonly bool CheckFile = false;
        protected readonly Stopwatch MergeStopwatch = new Stopwatch();
        private List<IfSkeletalAnimOptimizer> Optimizers =
            new List<IfSkeletalAnimOptimizer>();
    }
}
