﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.IO;
using nw.g3d.iflib;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Resources;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.toollib.Ext;

namespace nw.g3d.optcvtr
{
    public class g3doptcvtrParams
    {
        static g3doptcvtrParams()
        {
            resourceManager = new ResourceManager("nw.g3d.optcvtr.Resources.CommandLineHelp", typeof(g3doptcvtrParams).Assembly);
        }

        private static ResourceManager resourceManager;

        public static string DescriptionConverter(string description, string name)
        {
            var value = resourceManager.GetString(description.Replace("-", "_"));
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!string.IsNullOrEmpty(value));
            return value;
        }

        [CommandLineOption('s', "silent", Description = "silent", DescriptionConverterName = "DescriptionConverter")]
        public bool Silent { get; set; }

        [CommandLineOption('j', "jobs", Description = "jobs", DescriptionConverterName = "DescriptionConverter", ValueName = "number")]
        public int Job { get; set; }

        [CommandLineOption("disable-xsd-check", Description = "disable-xsd-check", DescriptionConverterName = "DescriptionConverter")]
        public bool DisableXsdCheck { get; set; }

        [CommandLineOption("disable-file-info", Description = "disable-file-info", DescriptionConverterName = "DescriptionConverter")]
        public bool DisableFileInfo { get; set; }

        [CommandLineOption("compress-material", Description = "compress-material", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressMaterial { get; set; }

        [CommandLineOption("rigid-body-to-rigid-skinning", Description = "rigid-body-to-rigid-skinning", DescriptionConverterName = "DescriptionConverter")]
        public bool RigidBodyToRigidSkinning { get; set; }

        [CommandLineOption("compress-bone-cull", Description = "compress-bone-cull", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressBoneCull { get; set; }

        [CommandLineOption("compress-bone-merge", Description = "compress-bone-merge", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressBoneMerge { get; set; }

        [CommandLineOption("compress-bone-unite-child", Description = "compress-bone-unite-child", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressBoneUniteChild { get; set; }

        [CommandLineOption("compress-bone-unite", Description = "compress-bone-unite", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressBoneUnite { get; set; }

        [CommandLineOption("compress-bone-unite-all", Description = "compress-bone-unite-all", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressBoneUniteAll { get; set; }

        [CommandLineOption("compress-shape", Description = "compress-shape", DescriptionConverterName = "DescriptionConverter")]
        public bool CompressShape { get; set; }

        [CommandLineOption("compress-shape-options", Description = "compress-shape-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "CompressShapeSubOptionsConverter")]
        public CompressShapeSubOptions CompressShapeOptions { get; set; }

        public static object CompressShapeSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<CompressShapeSubOptions>(value);
        }

        public class CompressShapeSubOptions
        {
            [CommandLineOption("ignore-skinning-count", Description = "d")]
            public bool IgnoreSkinningCount { get; set; }
        }

        [CommandLineOption('m', "merge", Description = "merge", DescriptionConverterName = "DescriptionConverter")]
        public bool Merge { get; set; }

        [CommandLineOption("merge-options", Description = "merge-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "MergeSubOptionsConverter")]
        public MergeSubOptions MergeOptions { get; set; }

        public static object MergeSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<MergeSubOptions>(value);
        }

        public class MergeSubOptions
        {
            [CommandLineOption("merge-file", Description = "d")]
            public string MergeFile { get; set; }

            [CommandLineOption("sampler-merge-priority", Description = "d")]
            public bool SamplerMergePriority { get; set; }

            [CommandLineOption("texture-srt-merge-priority", Description = "d")]
            public bool TextureSrtMergePriority { get; set; }

            [CommandLineOption("bone-visibility-merge-priority", Description = "d")]
            public bool BoneVisibilityMergePriority { get; set; }
        }

        [CommandLineOption("delete-near-vertex", Description = "delete-near-vertex", DescriptionConverterName = "DescriptionConverter")]
        public bool DeleteNearVertex { get; set; }

        [CommandLineOption("delete-near-vertex-options", Description = "delete-near-vertex-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "DeleteNearVertexSubOptionsConverter")]
        public DeleteNearVertexSubOptions DeleteNearVertexOptions { get; set; }

        public static object DeleteNearVertexSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<DeleteNearVertexSubOptions>(value);
        }

        public class DeleteNearVertexSubOptions
        {
            [CommandLineOption("position", Description = "d")]
            public float? Position { get; set; }

            // 度
            [CommandLineOption("normal", Description = "d")]
            public float? Normal { get; set; }

            // 度
            [CommandLineOption("tangent", Description = "d")]
            public float? Tangent { get; set; }

            // 度
            [CommandLineOption("binormal", Description = "d")]
            public float? Binormal { get; set; }

            [CommandLineOption("color", Description = "d")]
            public float? Color { get; set; }

            [CommandLineOption("uv", Description = "d")]
            public float? Uv { get; set; }

            [CommandLineOption("blend-index", Description = "d")]
            public float? BlendIndex { get; set; }

            [CommandLineOption("blend-weight", Description = "d")]
            public float? BlendWeight { get; set; }
        }

        [CommandLineOption("unite-submesh", Description = "unite-submesh", DescriptionConverterName = "DescriptionConverter")]
        public bool UniteSubmesh { get; set; }

        [CommandLineOption("divide-submesh", Description = "divide-submesh", DescriptionConverterName = "DescriptionConverter")]
        public bool DivideSubmesh { get; set; }

        [CommandLineOption("divide-submesh-options", Description = "divide-submesh-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "DivideSubmeshSubOptionsConverter")]
        public DivideSubmeshSubOptions DivideSubmeshOptions { get; set; }

        public static object DivideSubmeshSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<DivideSubmeshSubOptions>(value);
        }

        public class DivideSubmeshSubOptions
        {
            [CommandLineOption("max-bounding-box-width", Description = "d")]
            public float? MaxBoundingBoxWidth { get; set; }

            [CommandLineOption("max-division-depth", Description = "d")]
            public int? MaxDivisionDepth { get; set; }

            [CommandLineOption("min-triangle-count", Description = "d")]
            public int? MinTriangleCount { get; set; }
        }

        [CommandLineOption("trim-lod-model", Description = "trim-lod-model", DescriptionConverterName = "DescriptionConverter")]
        public bool TrimLodModel { get; set; }

        [CommandLineOption("polygon-reduction", Description = "polygon-reduction", DescriptionConverterName = "DescriptionConverter")]
        public bool PolygonReduction { get; set; }

        [CommandLineOption("polygon-reduction-options", Description = "polygon-reduction-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "PolygonReductionSubOptionsConverter")]
        public PolygonReductionSubOptions PolygonReductionOptions { get; set; }

        public static object PolygonReductionSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<PolygonReductionSubOptions>(value);
        }

        public class PolygonReductionSubOptions
        {
            [CommandLineArgFileOption("args-file", Description = "d")]
            public string ArgsFile { get; set; }

            [CommandLineOption("target-rate", Description = "d")]
            public string TargetRate { get; set; }

            [CommandLineOption("target-rate-lod-level1", Description = "d")]
            public float? TargetRateLodLevel1 { get; set; }

            [CommandLineOption("target-rate-lod-level2", Description = "d")]
            public float? TargetRateLodLevel2 { get; set; }

            [CommandLineOption("target-rate-lod-level3", Description = "d")]
            public float? TargetRateLodLevel3 { get; set; }

            [CommandLineOption("target-rate-lod-level4", Description = "d")]
            public float? TargetRateLodLevel4 { get; set; }

            [CommandLineOption("target-rate-lod-level5", Description = "d")]
            public float? TargetRateLodLevel5 { get; set; }

            [CommandLineOption("target-rate-lod-level6", Description = "d")]
            public float? TargetRateLodLevel6 { get; set; }

            [CommandLineOption("target-rate-lod-level7", Description = "d")]
            public float? TargetRateLodLevel7 { get; set; }

            [CommandLineOption("target-rate-lod-level8", Description = "d")]
            public float? TargetRateLodLevel8 { get; set; }

            [CommandLineOption("target-rate-lod-level9", Description = "d")]
            public float? TargetRateLodLevel9 { get; set; }

            [CommandLineOption("target-rate-per-shapes", Description = "d")]
            public string[] TargetRatePerMaterialShapes { get; set; }

            [CommandLineOption('q', "quality", Description = "d")]
            public double? Quality { get; set; }

            [CommandLineOption("property-flag", Description = "d")]
            public string[] PropertyFlag { get; set; }

            [CommandLineOption("feature-face-angle", Description = "d")]
            public double? FeatureFaceAngle { get; set; }

            [CommandLineOption("lock-open-edge-angle", Description = "d")]
            public double? LockOpenEdgeAngle { get; set; }

            [CommandLineOption("lock-uv-hard-edge-angle", Description = "d")]
            public double? LockUvHardEdgeAngle { get; set; }

            [CommandLineOption("feature-angle", Description = "d")]
            public double? OldFeatureAngle { get; set; }

            [CommandLineOption("open-edge-lock-angle", Description = "d")]
            public double? OldOpenEdgeLockAngle { get; set; }

            [CommandLineOption("lock-uv-hard-edge", Description = "d")]
            public double? OldLockUvHardEdge { get; set; }

            [CommandLineOption("optimize-position-policy", Description = "d")]
            public string OptimizePositionPolicy { get; set; }

            [CommandLineOption("smoothing", Description = "d")]
            public bool Smoothing { get; set; }

            [CommandLineOption("policy", Description = "d")]
            public string Policy { get; set; }

            [CommandLineOption("distance-offset", Description = "d")]
            public double? DistanceOffset { get; set; }

            [CommandLineOption("uv-weight", Description = "d")]
            public double? UvWeight { get; set; }

            [CommandLineOption("normal-weight", Description = "d")]
            public double? NormalWeight { get; set; }

            [CommandLineOption("anim-blend-weight", Description = "d")]
            public double? AnimBlendWeight { get; set; }

            [CommandLineOption("closed-vertex-threshold", Description = "d")]
            public double? ClosedVertexThreshold { get; set; }

            [CommandLineOption("merge-shape", Description = "d")]
            public bool MergeShape { get; set; }

            [CommandLineOption("exclude-shapes", Description = "d")]
            public string[] ExcludeShapes { get; set; }

            [CommandLineOption("disable-merge-shape", Description = "d")]
            public bool DisableMergeShape { get; set; }

            public string CreateProcessLogArgument()
            {
                StringBuilder processLogArgument = new StringBuilder();
                processLogArgument.Append($" --{GetOptionLongName(nameof(this.TargetRate))}");
                StringBuilder rateString = new StringBuilder();
                if (this.TargetRateLodLevel1 != null)
                {
                    float?[] rates = new[] {
                        this.TargetRateLodLevel1,
                        this.TargetRateLodLevel2,
                        this.TargetRateLodLevel3,
                        this.TargetRateLodLevel4,
                        this.TargetRateLodLevel5,
                        this.TargetRateLodLevel6,
                        this.TargetRateLodLevel7,
                        this.TargetRateLodLevel8,
                        this.TargetRateLodLevel9,
                    };
                    foreach (float? rate in rates)
                    {
                        if (rate == null)
                        {
                            break;
                        }
                        rateString.Append($" {rate.Value}");
                    }
                }
                else if (this.TargetRate != null)
                {
                    rateString.Append(this.TargetRate);
                }
                processLogArgument.Append($" '{rateString.ToString().Trim(' ')}'");

                if (this.TargetRatePerMaterialShapes != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.TargetRatePerMaterialShapes))} '{string.Join(",", this.TargetRatePerMaterialShapes)}'");
                }

                if (this.PropertyFlag != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.PropertyFlag))} {string.Join("|", this.PropertyFlag)}");
                }

                if (this.OptimizePositionPolicy != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.OptimizePositionPolicy))} {this.OptimizePositionPolicy}");
                }

                if (this.Policy != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.Policy))} {this.Policy}");
                }

                if (this.Quality != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.Quality))} {this.Quality.Value}");
                }

                if (this.DistanceOffset != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.DistanceOffset))} {this.DistanceOffset.Value}");
                }

                if (this.Smoothing)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.Smoothing))}");
                }

                if (this.LockUvHardEdgeAngle != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.LockUvHardEdgeAngle))} {this.LockUvHardEdgeAngle.Value}");
                }

                if (this.OldLockUvHardEdge != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.OldLockUvHardEdge))} {this.OldLockUvHardEdge.Value}");
                }

                if (this.LockOpenEdgeAngle != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.LockOpenEdgeAngle))} {this.LockOpenEdgeAngle.Value}");
                }

                if (this.OldOpenEdgeLockAngle != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.OldOpenEdgeLockAngle))} {this.OldOpenEdgeLockAngle.Value}");
                }

                if (this.FeatureFaceAngle != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.FeatureFaceAngle))} {this.FeatureFaceAngle.Value}");
                }

                if (this.OldFeatureAngle != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.OldFeatureAngle))} {this.OldFeatureAngle.Value}");
                }

                if (this.UvWeight != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.UvWeight))} {this.UvWeight.Value}");
                }

                if (this.NormalWeight != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.NormalWeight))} {this.NormalWeight.Value}");
                }

                if (this.AnimBlendWeight != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.AnimBlendWeight))} {this.AnimBlendWeight.Value}");
                }

                if (this.ClosedVertexThreshold != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.ClosedVertexThreshold))} {this.ClosedVertexThreshold.Value}");
                }

                if (this.ExcludeShapes != null)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.ExcludeShapes))} '{string.Join(",", this.ExcludeShapes)}'");
                }

                if (this.DisableMergeShape)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.DisableMergeShape))}");
                }
                return processLogArgument.ToString().Trim(' ');
            }

            private static string GetOptionLongName(string propertyName)
            {
                return g3doptcvtrParams.GetOptionLongName(typeof(g3doptcvtrParams.PolygonReductionSubOptions), propertyName);
            }
        }

        [CommandLineOption("unite-lod-model", Description = "unite-lod-model", DescriptionConverterName = "DescriptionConverter")]
        public bool UniteLodModel { get; set; }

        [CommandLineOption("unite-lod-model-options", Description = "unite-lod-model-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "UniteLodSubOptionsConverter")]
        public UniteLodSubOptions UniteLodOptions { get; set; }

        public static object UniteLodSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<UniteLodSubOptions>(value);
        }

        public class UniteLodSubOptions
        {
            [CommandLineOption("lod-level1", Description = "d")]
            public string LodLevel1 { get; set; }

            [CommandLineOption("lod-level2", Description = "d")]
            public string LodLevel2 { get; set; }

            [CommandLineOption("lod-level3", Description = "d")]
            public string LodLevel3 { get; set; }

            [CommandLineOption("lod-level4", Description = "d")]
            public string LodLevel4 { get; set; }

            [CommandLineOption("lod-level5", Description = "d")]
            public string LodLevel5 { get; set; }

            [CommandLineOption("lod-level6", Description = "d")]
            public string LodLevel6 { get; set; }

            [CommandLineOption("lod-level7", Description = "d")]
            public string LodLevel7 { get; set; }

            [CommandLineOption("lod-level8", Description = "d")]
            public string LodLevel8 { get; set; }

            [CommandLineOption("lod-level9", Description = "d")]
            public string LodLevel9 { get; set; }

            public string CreateProcessLogArgument()
            {
                StringBuilder argumentStringBuilder = new StringBuilder();
                string[] lodModelFilePaths = new[] {
                    null,
                    this.LodLevel1,
                    this.LodLevel2,
                    this.LodLevel3,
                    this.LodLevel4,
                    this.LodLevel5,
                    this.LodLevel6,
                    this.LodLevel7,
                    this.LodLevel8,
                    this.LodLevel9,
                };

                for (int lodLevel = 1; lodLevel <= IfModelUniteLodModelOptimizer.MaxLodLevel; ++lodLevel)
                {
                    string lodModelFilePath = lodModelFilePaths[lodLevel];
                    if (lodModelFilePath != null)
                    {
                        argumentStringBuilder.Append($"--lod-level{lodLevel} {lodModelFilePath} ");
                    }
                }

                return argumentStringBuilder.ToString().Trim(' ');
            }
        }

        [CommandLineOption("optimize-primitive", Description = "optimize-primitive", DescriptionConverterName = "DescriptionConverter")]
        public bool OptimizePrimitive { get; set; }

        [CommandLineOption("optimize-primitive-options", Description = "optimize-primitive-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "OptimizePrimitiveSubOptionsConverter")]
        public OptimizePrimitiveSubOptions OptimizePrimitiveOptions { get; set; }

        public static object OptimizePrimitiveSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<OptimizePrimitiveSubOptions>(value);
        }

        public class OptimizePrimitiveSubOptions
        {
            public enum OptimizeMode
            {
                Normal,
                Force,
                BruteForce,
                Forsyth
            }

            [CommandLineOption("mode", Description = "d")]
            public OptimizeMode Mode { get; set; }
        }

        [CommandLineOption("quantization-analysis", Description = "quantization-analysis", DescriptionConverterName = "DescriptionConverter")]
        public bool QuantizationAnalysis { get; set; }

        [CommandLineOption("combine", Description = "combine", DescriptionConverterName = "DescriptionConverter")]
        public bool Combine { get; set; }

        [CommandLineOption("combine-options", Description = "combine-options", DescriptionConverterName = "DescriptionConverter", ConverterName = "CombineSubOptionsConverter")]
        public CombineSubOptions CombineOptions { get; set; }

        public static object CombineSubOptionsConverter(string value, string name)
        {
            return CommandLineUtility.SubOptionConverter<CombineSubOptions>(value);
        }

        public class CombineSubOptions
        {
            [CommandLineValues(Description = "combine-file-paths")]//, ValueName = "combine-file-paths", DescriptionConverterName = "DescriptionConverter")]
            public string[] Paths { get; set; }

            [CommandLineOption("ignore-frame-loop-unmatch", Description = "d")]
            public bool IgnoreFrameLoopUnmatch { get; set; }

            [CommandLineOption("ignore-animaton-conflict", Description = "d")]
            public bool IgnoreAnimationConflict { get; set; }

            public string CreateProcessLogArgument()
            {
                StringBuilder processLogArgument = new StringBuilder();
                processLogArgument.Append(string.Join(" ", this.Paths.Select(s => System.IO.Path.GetFullPath(s).Replace('\\', '/'))));
                if (this.IgnoreFrameLoopUnmatch)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.IgnoreFrameLoopUnmatch))}");
                }

                if (this.IgnoreAnimationConflict)
                {
                    processLogArgument.Append($" --{GetOptionLongName(nameof(this.IgnoreAnimationConflict))}");
                }

                return processLogArgument.ToString().Trim(' ');
            }

            private static string GetOptionLongName(string propertyName)
            {
                return g3doptcvtrParams.GetOptionLongName(typeof(g3doptcvtrParams.CombineSubOptions), propertyName);
            }
        }

        [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
        public string Output { get; set; }

        [CommandLineValue(0, Description = "path", ValueName = "path", DescriptionConverterName = "DescriptionConverter")]
        public string Path { get; set; }

        [CommandLineArgFileOption("args-file", Description = "args-file", DescriptionConverterName = "DescriptionConverter")]
        public string ArgsFile { get; set; }

        public static string GetOptionLongName(Type type, string propertyName)
        {
            var prop = type.GetProperty(propertyName);
            var attr = (CommandLineOptionAttribute)Attribute.GetCustomAttribute(prop, typeof(CommandLineOptionAttribute));
            return attr.LongName;
        }
    }
}
