﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Runtime;
using System.Linq;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using Nintendo.Foundation.IO;
using System.Text;

namespace nw.g3d.optcvtr
{
    // 中間ファイル最適化コンバータ
    public class g3doptcvtr
    {
        /// <summary>
        /// テストから実行する際に使う引数です。
        /// </summary>
        public string Arguments { get; set; } = null;

        /// <summary>
        /// XSD パスです。テストから実行する際にはあらかじめ設定しておく必要がありますが、通常は相対パスで自動的に設定されます。
        /// </summary>
        public string XsdBasePath { get; set; } = null;

        // エントリポイント
        public static void Main()
        {
            string path = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            ProfileOptimization.SetProfileRoot(path);
            ProfileOptimization.StartProfile(Path.GetFileNameWithoutExtension(
                Assembly.GetExecutingAssembly().Location) + ".JIT.Profile");

            // エラー時にポーズしない
            G3dToolUtility.EnableErrorPause = false;
            g3doptcvtr g3doptcvtr = new g3doptcvtr();
            G3dToolUtility.Run(delegate { g3doptcvtr.Run(); });
        }

        // 実行
        public void Run()
        {
            // 全体計測
            Stopwatch swRun = new Stopwatch();
            swRun.Start();
            {
                // 正常終了できなければ、異常終了
                Environment.ExitCode = 1;

                // ローカライズ処理
                Strings.Initialize("nw.g3d.optcvtr.Resources.StringResource");
                G3dLocalization.SetCulture();

                // 初期化
                Stopwatch swInitialize = new Stopwatch();
                swInitialize.Start();
                {
                    if (!Initialize())
                    {
                        Environment.ExitCode = 0;
                        return;
                    }
                }
                swInitialize.Stop();
                this.WriteProgress("Initialize",
                    this.Optimizers.Count, swInitialize.ElapsedMilliseconds);

                // 処理
                Process();
                Environment.ExitCode = 0;
            }
            swRun.Stop();
            this.WriteProgress("Finish",
                this.Optimizers.Count, swRun.ElapsedMilliseconds);
        }

        //---------------------------------------------------------------------
        // 初期化
        private bool Initialize()
        {
            g3doptcvtrParams param;
            var setting = new CommandLineParserSettings()
            {
                ApplicationDescription = g3doptcvtrParams.DescriptionConverter("ApplicationDescription", null),
                // 例外処理に任せる
                ErrorWriter = x => { },
            };

            try
            {
                IEnumerable<string> arguments = string.IsNullOrEmpty(this.Arguments) ?
                    arguments = Environment.GetCommandLineArgs().Skip(1) :
                    SplitArgumentText(this.Arguments);
                if (!(new CommandLineParser(setting).ParseArgs(arguments, out param)))
                {
                    return false;
                }
            }
            catch
            {
                throw;
            }

            ProcessArgument(param);
            return true;
        }

        private void ProcessArgument(g3doptcvtrParams param)
        {
            // 進捗表示切り替え
            this.IsSilent = param.Silent;

            // 並列処理数
            G3dParallel.Job = param.Job;

            // スキーマチェック切り替え
            if (!param.DisableXsdCheck)
            {
                // スキーマベースパス不明をエラー扱いしない
                if (string.IsNullOrEmpty(this.XsdBasePath))
                {
                    this.XsdBasePath = G3dToolUtility.GetXsdBasePath();
                }
            }

            this.DisableFileInfo = param.DisableFileInfo;

            this.Optimizers.Add(Optimizer.Create(param));
        }

        //---------------------------------------------------------------------
        // 処理
        private void Process()
        {
            foreach (Optimizer optimizer in this.Optimizers)
            {
                optimizer.DisableFileInfo = this.DisableFileInfo;
            }

            G3dParallel.ForEach(this.Optimizers, delegate(Optimizer optimizer)
                {
                    Stopwatch swProcessFile = new Stopwatch();
                    swProcessFile.Start();
                    {
                        optimizer.XsdBasePath = this.XsdBasePath;
                        optimizer.Optimize();
                    }
                    swProcessFile.Stop();
                    if (!this.IsSilent)
                    {
                        this.WriteProgress("ProcessFile",
                            swProcessFile.ElapsedMilliseconds, optimizer.GetLog());
                    }

                    optimizer.Dispose();
                });
        }

        //---------------------------------------------------------------------
        // 進捗の書き出し
        internal void WriteProgress(string id, params object[] args)
        {
            if (this.IsSilent)
            {
                return;
            }

            string progress = Strings.Get(id, args);
            // Console、Debug はスレッドセーフ
            Console.WriteLine(progress);
            Debug.WriteLine(progress);
        }


        /// <summary>
        /// 自動テスト用の引数をパース単位に分割する処理です。テスト時にしか利用しないので制限付きでしか解析しません。
        /// </summary>
        /// <param name="argumentText">解析する引数テキスト</param>
        /// <returns>分割された引数</returns>
        private static IEnumerable<string> SplitArgumentText(string argumentText)
        {
            List<string> tmpArguments = new List<string>();
            var spaceSplitted = argumentText.Split(' ');
            StringBuilder tmpArg = new StringBuilder();
            foreach (string arg in spaceSplitted)
            {
                if (!string.IsNullOrEmpty(tmpArg.ToString()))
                {
                    tmpArg.Append($" {arg}");
                    if (!arg.EndsWith("\\\"") && arg.EndsWith("\""))
                    {
                        tmpArguments.Add(tmpArg.ToString().Trim('\"'));
                        tmpArg.Clear();
                    }
                }
                else
                {
                    if (arg.StartsWith("\""))
                    {
                        tmpArg.Append(arg);
                    }
                    else
                    {
                        tmpArguments.Add(arg.Trim('\"'));
                    }
                }
            }
            return tmpArguments.ToArray();
        }

        private bool IsSilent { get; set; }
        private bool DisableFileInfo;
        private readonly List<Optimizer> Optimizers = new List<Optimizer>();
    }
}
