﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System;

    /// <summary>
    /// 行列の回転順序が XYZ である演算関数です。
    /// </summary>
    public static class EulerXyz
    {
        /// <summary>
        /// X軸、Y軸、Z軸の順に回転させた回転行列に設定します。
        /// 回転の値をradianとして行列を求めます。
        /// </summary>
        /// <param name="target">設定する行列です。</param>
        /// <param name="xRadian">x 軸を中心とした回転角度 (Radian単位)です。</param>
        /// <param name="yRadian">y 軸を中心とした回転角度 (Radian単位)です。</param>
        /// <param name="zRadian">z 軸を中心とした回転角度 (Radian単位)です。</param>
        public static void SetRotate(
            IMatrix target, float xRadian, float yRadian, float zRadian)
        {
            Ensure.Argument.NotNull(target);

            Ensure.Argument.True(target.RowCount >= 3);
            Ensure.Argument.True(target.ColumnCount >= 3);

            // NW4RSystemLib 準拠 math_types.cpp MTX34RotXYZFIdx
            target.SetIdentity();

            float sinx = (float)Math.Sin(xRadian);
            float cosx = (float)Math.Cos(xRadian);
            float siny = (float)Math.Sin(yRadian);
            float cosy = (float)Math.Cos(yRadian);
            float sinz = (float)Math.Sin(zRadian);
            float cosz = (float)Math.Cos(zRadian);
            float f1, f2;

            target[2, 0] = -siny;
            target[0, 0] = cosz * cosy;
            target[1, 0] = sinz * cosy;
            target[2, 1] = cosy * sinx;
            target[2, 2] = cosy * cosx;

            f1 = cosx * sinz;
            f2 = sinx * cosz;

            target[0, 1] = (f2 * siny) - f1;
            target[1, 2] = (f1 * siny) - f2;

            f1 = sinx * sinz;
            f2 = cosx * cosz;
            target[0, 2] = (f2 * siny) + f1;
            target[1, 1] = (f1 * siny) + f2;
        }

        /// <summary>
        /// Transform を行列に設定します。
        /// 回転の値をradianとして行列を求めます。
        /// </summary>
        /// <param name="target">設定する行列です。</param>
        /// <param name="transform">Transform。</param>
        public static void SetTransform(IMatrix target, Transform transform)
        {
            Ensure.Argument.NotNull(target);
            Ensure.Argument.NotNull(transform);

            Matrix34 mtx34 = target as Matrix34;
            if (mtx34 != null)
            {
                mtx34.SetIdentity();
                mtx34.Scale(transform.Scale);
                mtx34.RotateX(transform.Rotate.X);
                mtx34.RotateY(transform.Rotate.Y);
                mtx34.RotateZ(transform.Rotate.Z);
                mtx34.Translate(transform.Translate);
                return;
            }

            Matrix44 mtx44 = target as Matrix44;
            if (mtx44 != null)
            {
                mtx44.SetIdentity();
                mtx44.Scale(transform.Scale);
                mtx44.RotateX(transform.Rotate.X);
                mtx44.RotateY(transform.Rotate.Y);
                mtx44.RotateZ(transform.Rotate.Z);
                mtx44.Translate(transform.Translate);
                return;
            }

            throw new ArgumentException();
        }

        /// <summary>
        /// クォータニオンをオイラー角に変換します。
        /// </summary>
        /// <param name="quat">変換するクォータニオンです。</param>
        /// <returns>変換した回転順序が XYZ であるオイラー角です。</returns>
        public static Vector3 ToEuler(Quaternion quat)
        {
            throw new NotImplementedException(); // TODO:実装
        }

        /// <summary>
        /// オイラー角をクォータニオンに変換します。
        /// </summary>
        /// <param name="euler">変換する回転順序が XYZ であるオイラー角です。</param>
        /// <returns>変換したクォータニオンです。</returns>
        public static Quaternion ToQuaternion(Vector3 euler)
        {
            throw new NotImplementedException(); // TODO:実装
        }

        /// <summary>
        /// オイラー角を度数からラジアンに変換します。
        /// </summary>
        /// <param name="euler">変換する回転順序が XYZ であるオイラー角です。</param>
        /// <returns>確度をラジアンに変換したオイラー角です。</returns>
        public static Vector3 ToRadian(Vector3 euler)
        {
            return new Vector3(
                (euler.X / 180.0f) * (float)Math.PI,
                (euler.Y / 180.0f) * (float)Math.PI,
                (euler.Z / 180.0f) * (float)Math.PI);
        }

        /// <summary>
        /// オイラー角をラジアンから度数に変換します。
        /// </summary>
        /// <param name="euler">変換する回転順序が XYZ であるオイラー角です。</param>
        /// <returns>確度を度数に変換したオイラー角です。</returns>
        public static Vector3 ToDegree(Vector3 euler)
        {
            return new Vector3(
                (euler.X / (float)Math.PI) * 180.0f,
                (euler.Y / (float)Math.PI) * 180.0f,
                (euler.Z / (float)Math.PI) * 180.0f);
        }
    }
}
