﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シェーダーパラメーターアニメーションの量子化分析
    public class IfShaderParamAnimQuantizationAnalysis
    {
        public enum Srt2dIndex
        {
            ScaleX = 0,
            ScaleY,
            Rotate,
            TranslateX,
            TranslateY
        }

        public enum TexsrtIndex
        {
            Mode = 0,
            ScaleX,
            ScaleY,
            Rotate,
            TranslateX,
            TranslateY
        }

        public enum Srt3dIndex
        {
            ScaleX = 0,
            ScaleY,
            ScaleZ,
            RotateX,
            RotateY,
            RotateZ,
            TranslateX,
            TranslateY,
            TranslateZ
        }

        // 量子化分析
        public static IfQuantizationAnalysisResult Analyse(
            IG3dShaderParamAnimInfo shaderAnimInfo,
            shader_param_typeType type,
            int component_index,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // frame の量子化を評価して type を選択する。

            // frametype が none 以外の場合は frame 量子化の判定を行わない。
            if (!forceAnalyse &&
                curve.frameType != curve_frame_typeType.none)
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / shaderAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            // カーブを量子化します。
            curve.QuantizeCurve();

            if (type == shader_param_typeType.@int ||
                type == shader_param_typeType.int2 ||
                type == shader_param_typeType.int3 ||
                type == shader_param_typeType.int4 ||
                type == shader_param_typeType.@uint ||
                type == shader_param_typeType.uint2 ||
                type == shader_param_typeType.uint3 ||
                type == shader_param_typeType.uint4)
            {
                curve.EvalQuantizeInt();
            }
            else if (
                type == shader_param_typeType.texsrt)
            {
                // 量子化結果を判定し反映します。
                if (component_index == (int)TexsrtIndex.Mode)
                {
                    curve.EvalQuantizeInt();
                }
                else if (component_index == (int)TexsrtIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    float tolerance_tex_rotate;

                    // texsrt, texsrt_ex の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)TexsrtIndex.ScaleX ||
                    component_index == (int)TexsrtIndex.ScaleY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)TexsrtIndex.TranslateX ||
                    component_index == (int)TexsrtIndex.TranslateY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt2d)
            {
                // 量子化結果を判定し反映します。
                // 回転値の量子化
                if (component_index == (int)Srt2dIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    float tolerance_tex_rotate;

                    // srt2d の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt2dIndex.ScaleX ||
                    component_index == (int)Srt2dIndex.ScaleY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt2dIndex.TranslateX ||
                    component_index == (int)Srt2dIndex.TranslateY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt3d)
            {
                if (
                    component_index == (int)Srt3dIndex.RotateX ||
                    component_index == (int)Srt3dIndex.RotateY ||
                    component_index == (int)Srt3dIndex.RotateZ)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    // 回転の量子化解析はランタイムの動作に合わせてキーを Radian に変換してから行うため
                    // tolerance の値を Radian に変換する必要がある。
                    float tolerance_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);
                    curve.EvalQuantizeFloat(
                        tolerance_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt3dIndex.ScaleX ||
                    component_index == (int)Srt3dIndex.ScaleY ||
                    component_index == (int)Srt3dIndex.ScaleZ)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt3dIndex.TranslateX ||
                    component_index == (int)Srt3dIndex.TranslateY ||
                    component_index == (int)Srt3dIndex.TranslateZ)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }

            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;

            return result;
        }

        // 量子化分析
        public static IfQuantizationAnalysisResult Analyse(
            material_anim_infoType shaderAnimInfo,
            shader_param_typeType type,
            int component_index,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // frame の量子化を評価して type を選択する。

            // frametype が none 以外の場合は frame 量子化の判定を行わない。
            if (!forceAnalyse &&
                curve.frameType != curve_frame_typeType.none)
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / shaderAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            // カーブを量子化します。
            curve.QuantizeCurve();

            if (type == shader_param_typeType.@int ||
                type == shader_param_typeType.int2 ||
                type == shader_param_typeType.int3 ||
                type == shader_param_typeType.int4 ||
                type == shader_param_typeType.@uint ||
                type == shader_param_typeType.uint2 ||
                type == shader_param_typeType.uint3 ||
                type == shader_param_typeType.uint4)
            {
                curve.EvalQuantizeInt();
            }
            else if (
                type == shader_param_typeType.texsrt)
            {
                // 量子化結果を判定し反映します。
                if (component_index == (int)TexsrtIndex.Mode)
                {
                    curve.EvalQuantizeInt();
                }
                else if (component_index == (int)TexsrtIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    float tolerance_tex_rotate;

                    // texsrt, texsrt_ex の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)TexsrtIndex.ScaleX ||
                    component_index == (int)TexsrtIndex.ScaleY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)TexsrtIndex.TranslateX ||
                    component_index == (int)TexsrtIndex.TranslateY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt2d)
            {
                // 量子化結果を判定し反映します。
                // 回転値の量子化
                if (component_index == (int)Srt2dIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    float tolerance_tex_rotate;

                    // srt2d の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt2dIndex.ScaleX ||
                    component_index == (int)Srt2dIndex.ScaleY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt2dIndex.TranslateX ||
                    component_index == (int)Srt2dIndex.TranslateY)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt3d)
            {
                if (
                    component_index == (int)Srt3dIndex.RotateX ||
                    component_index == (int)Srt3dIndex.RotateY ||
                    component_index == (int)Srt3dIndex.RotateZ)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    bool useAbsEval = true;
                    // 回転の量子化解析はランタイムの動作に合わせてキーを Radian に変換してから行うため
                    // tolerance の値を Radian に変換する必要がある。
                    float tolerance_rotate =
                        FloatUtility.DegreeToRadian(shaderAnimInfo.quantize_tolerance_tex_rotate);
                    curve.EvalQuantizeFloat(
                        tolerance_rotate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt3dIndex.ScaleX ||
                    component_index == (int)Srt3dIndex.ScaleY ||
                    component_index == (int)Srt3dIndex.ScaleZ)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
                else if (
                    component_index == (int)Srt3dIndex.TranslateX ||
                    component_index == (int)Srt3dIndex.TranslateY ||
                    component_index == (int)Srt3dIndex.TranslateZ)
                {
                    bool useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        shaderAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        shaderAnimInfo.frame_resolution);
                }
            }

            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;

            return result;
        }
    }
}
