﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class PseudoVertexCacheManager
    {
        public const int CafeCacheSize = G3dConstant.VertexCacheSize;
        private readonly Queue<int> vertexCacheQueue = new Queue<int>();
        private int totalVertexCount = 0;
        private int totalTriangleCount = 0;
        private int cacheMissCount = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PseudoVertexCacheManager()
        {
        }

        /// <summary>
        /// 頂点キャッシュのサイズです。
        /// </summary>
        public int VertexCacheSize
        {
            get
            {
                return CafeCacheSize;
            }
        }

        /// <summary>
        /// 描画を行った頂点数を取得します。
        /// </summary>
        public int TotalVertexCount
        {
            get
            {
                return this.totalVertexCount;
            }
        }

        /// <summary>
        /// 処理した三角形の数を取得します。
        /// </summary>
        public int TotalTriangleCount
        {
            get
            {
                return this.totalTriangleCount;
            }
        }

        /// <summary>
        /// キャッシュミスが発生した回数を取得します。
        /// </summary>
        public int CacheMissCount
        {
            get
            {
                return this.cacheMissCount;
            }
        }

        /// <summary>
        /// 全変換処理に対するキャッシュミスの割合です。
        /// </summary>
        public float CacheMissRatio
        {
            get
            {
                float cacheMissRatio = 0.0f;

                if (0 < this.TotalVertexCount)
                {
                    cacheMissRatio = (float)this.CacheMissCount / (float)this.TotalVertexCount;
                }

                return cacheMissRatio;
            }
        }

        /// <summary>
        /// 三角形を描くのに必要な頂点数の平均です。
        /// </summary>
        public float AverageCacheMissRatio
        {
            get
            {
                float averageCacheMissRatio = 0.0f;

                if (0 < this.TotalTriangleCount)
                {
                    averageCacheMissRatio = (float)this.CacheMissCount / (float)this.TotalTriangleCount;
                }

                return averageCacheMissRatio;
            }
        }

        /// <summary>
        /// トライアングルリストにおける擬似シェーディング動作を行い、その動作情報を更新します。
        /// </summary>
        /// <param name="triangles">動作を行うためのトライアングルリストです。</param>
        public void TransformVertices(IList<int> vertexIndices)
        {
            if ((vertexIndices.Count % 3) != 0)
            {
                return;
            }

            this.totalTriangleCount = vertexIndices.Count / 3;

            foreach (var posId in vertexIndices)
            {
                this.TransformVertex(posId);
                ++totalVertexCount;
            }
        }

        /// <summary>
        /// 計算結果の情報を初期状態に戻します。
        /// </summary>
        private void ClearResult()
        {
            this.totalVertexCount = 0;
            this.totalTriangleCount = 0;
            this.cacheMissCount = 0;
        }

        /// <summary>
        /// 頂点キャッシュと計算結果の情報を初期状態に戻します。
        /// </summary>
        public void Clear()
        {
            this.ClearResult();
            this.vertexCacheQueue.Clear();
        }

        /// <summary>
        /// 頂点の変換処理を行った場合において、頂点キャッシュの動作だけ行います。
        /// </summary>
        /// <param name="positionId">変換処理を行う座標ＩＤです。</param>
        /// <returns>キャッシュにヒットした場合は true が返ります。</returns>
        private bool TransformVertex(int positionId)
        {
            bool isHitCache = false;

            if (this.IsContains(positionId))
            {
                isHitCache = true;
            }
            else
            {
                ++this.cacheMissCount;
                this.EnqueuePosition(positionId);
            }

            return isHitCache;
        }

        /// <summary>
        /// 任意の座標IDがキャッシュされているかを取得します。
        /// </summary>
        /// <param name="positionId">確認したい座標IDです。</param>
        /// <returns>キャッシュされてれば true を返します。</returns>
        private bool IsContains(int positionId)
        {
            return this.vertexCacheQueue.Contains(positionId);
        }

        /// <summary>
        /// キューの末尾に座標IDを追加します。
        /// キャッシュ数を超えると先頭からIDを削除します。
        /// </summary>
        /// <param name="positionId">座標IDです。</param>
        private void EnqueuePosition(int positionId)
        {
            this.vertexCacheQueue.Enqueue(positionId);
            while (this.VertexCacheSize < this.vertexCacheQueue.Count)
            {
                int front = this.vertexCacheQueue.Dequeue();
            }
        }
    }
}
