﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.IO;
using System.Linq;

namespace nw.g3d.iflib
{
    // LOD モデル削除
    public class IfModelTrimLodModelOptimizer : IfModelOptimizer
    {
        // コンストラクタ
        public IfModelTrimLodModelOptimizer() :
            base("IfModelTrimLodModelOptimizer_Log") { }

        public override string ToString()
        {
            return $"{base.ToString()} {this.GetResult()}";
        }

        // プロセス
        public override string Process
        {
            get { return "trim_lod_model"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        // 最適化
        protected override void Optimize()
        {
            EnableProcessLog = false;
            // 2 つ目以降のメッシュを消す
            foreach (var shape in this.Target.shape_array.shape)
            {
                if (shape.mesh_array.length > 1)
                {
                    EnableProcessLog = true;
                    TrimLod(shape);
                }
                shape.shape_info.polygon_reduction_mode = string.Empty;
            }

            // 不要になったストリームはここで取り除かれる
            if (EnableProcessLog)
            {
                StreamUtility.SortStream(this.Target, this.Streams);
            }
        }

        public void TrimLod(shapeType shape)
        {
            shape.mesh_array.mesh = new meshType[] { shape.mesh_array.mesh[0] };
            vertexType vertex = this.Target.vertex_array.vertex[shape.shape_info.vertex_index];

            if (vertex.lod_offset == null)
            {
                return;
            }

            int baseVertexCount = G3dDataParser.ParseIntArray(vertex.lod_offset.Value).Min(p => p > 0 ? p : vertex.vtx_attrib_array.vtx_attrib[0].count);
            vertex.lod_offset = null;
            if (baseVertexCount <= 0)
            {
                return;
            }

            foreach (var vtx_attrib in vertex.vtx_attrib_array.vtx_attrib)
            {
                int removeCount;
                int removeOffset;
                switch (vtx_attrib.type)
                {
                    case vtx_attrib_typeType.@int:
                    case vtx_attrib_typeType.@uint:
                    case vtx_attrib_typeType.@float:
                        removeCount = (vtx_attrib.count - baseVertexCount) * 1;
                        removeOffset = baseVertexCount * 1;
                        break;
                    case vtx_attrib_typeType.int2:
                    case vtx_attrib_typeType.uint2:
                    case vtx_attrib_typeType.float2:
                        removeCount = (vtx_attrib.count - baseVertexCount) * 2;
                        removeOffset = baseVertexCount * 2;
                        break;
                    case vtx_attrib_typeType.int3:
                    case vtx_attrib_typeType.uint3:
                    case vtx_attrib_typeType.float3:
                        removeCount = (vtx_attrib.count - baseVertexCount) * 3;
                        removeOffset = baseVertexCount * 3;
                        break;
                    case vtx_attrib_typeType.int4:
                    case vtx_attrib_typeType.uint4:
                    case vtx_attrib_typeType.float4:
                        removeCount = (vtx_attrib.count - baseVertexCount) * 4;
                        removeOffset = baseVertexCount * 4;
                        break;
                    default:
                        throw new InvalidDataException();
                }

                G3dStream stream = this.Streams[vtx_attrib.stream_index];
                switch (stream.type)
                {
                    case stream_typeType.@float:
                        stream.FloatData.RemoveRange(removeOffset, removeCount);
                        break;
                    case stream_typeType.@int:
                        stream.IntData.RemoveRange(removeOffset, removeCount);
                        break;
                    case stream_typeType.@byte:
                        stream.ByteData.RemoveRange(removeOffset, removeCount);
                        break;
                    default:
                        throw new InvalidDataException();
                }

                vtx_attrib.count = baseVertexCount;
            }
        }
    }
}
