﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Diagnostics;
using System.Collections.ObjectModel;

namespace nw.g3d.iflib
{
    /// <summary>
    /// サブメッシュ分割のユーティリティです。
    /// </summary>
    public class IfDivideUtility
    {
        public enum ResultCode
        {
            NotDivide,
            Success,
            LessTriangle,
            TooDeep,
            ShortBox,
            TooSmall,
            NotReduction,
            Discrepancy,
            Exclusion
        }

        /// <summary>
        /// 分割用のメッシュ構造です。
        /// </summary>
        public class DivideMesh
        {
            public class AxisComparer : IComparer<ITriangle>
            {
                public AxisComparer(Vector3 axis)
                {
                    this.axis = axis;
                }

                public int Compare(ITriangle x, ITriangle y)
                {
                    return (int)(x.Center.Dot(axis) - y.Center.Dot(axis));
                }

                private Vector3 axis = null;
            }

            public List<ITriangle> Triangles
            {
                get { return triangles; }
            }

            public IBounding MeshBounding { get; private set; }

            public DivideMesh Parent { get; private set; }

            public DivideMesh[] Children
            {
                get
                {
                    return children;
                }
            }

            public int Depth { get; private set; }

            public Vector3[] Eigens
            {
                get
                {
                    return eigens;
                }
            }

            public ResultCode DivideResult { get; set; }

            public int BakeBreadthIndex { get; private set; }

            public int BakeDepthIndex { get; private set; }

            public bool HasChildren
            {
                get
                {
                    if (Children[0] != null &&
                        Children[1] != null)
                    {
                        return true;
                    }

                    return false;
                }
            }

            public bool HasParent
            {
                get
                {
                    return Parent != null;
                }
            }

            public bool HasExclusionPolygon { get; private set; }

            public IDivideAlgorithm DividedAlgorithm { get; private set; }

            /// <summary>
            /// 親ノードとバウンディングを設定するコンストラクタです。
            /// </summary>
            public DivideMesh(DivideMesh parent, IBounding bounding)
            {
                SetParent(parent);
                if (parent != null)
                {
                    Triangles.Capacity = parent.Triangles.Count;
                }
                MeshBounding = bounding;
                DivideResult = ResultCode.NotDivide;
                BakeBreadthIndex = BakeDepthIndex = 0;
                HasExclusionPolygon = false;
            }

            public void AddTriangle(ITriangle triangle)
            {
                triangles.Add(triangle);
                MeshBounding.AddTriangle(triangle);

                DivideTriangle dTriangle = triangle as DivideTriangle;
                if (dTriangle != null && dTriangle.IsExclusion)
                {
                    HasExclusionPolygon = true;
                }
            }

            /// <summary>
            /// 固有ベクトルを計算します。
            /// </summary>
            public void Calc()
            {
                CalcPCA(eigens, triangles);
                if (HasChildren)
                {
                    foreach (DivideMesh child in Children)
                    {
                        child.Calc();
                    }
                }
            }

            /// <summary>
            /// 固有ベクトルを基にソートを行います。
            /// </summary>
            public void SortFromEigen(List<ITriangle> triangles, int eigenIndex)
            {
                if (!eigens[eigenIndex].IsZero())
                {
                    // 主成分
                    triangles.Sort(new AxisComparer(eigens[eigenIndex]));
                }
                else
                {
                    // 固有値が計算できない場合はどんな軸で並べても問題ないので、軸方向でソートする。
                    Vector3 axis = new Vector3();
                    axis[eigenIndex] = 1.0f;
                    triangles.Sort(new AxisComparer(axis));
                }
            }

            /// <summary>
            /// メッシュを分割します。
            /// </summary>
            public void Divide(IDivideAlgorithm algorithm)
            {
                if (HasChildren)
                {
                    // 再帰的に分割する。
                    foreach (DivideMesh child in Children)
                    {
                        child.Divide(algorithm);
                    }
                }

                if (Triangles.Count <= 2)
                {
                    // トライアングルが２以下だと分割できない。
                    this.DivideResult = ResultCode.LessTriangle;
                    return;
                }

                this.DivideResult = algorithm.Divide(this);
                if (this.DivideResult == ResultCode.Success)
                {
                    // 再帰的に分割する。
                    foreach (DivideMesh child in Children)
                    {
                        child.DividedAlgorithm = algorithm;
                        child.Divide(algorithm);
                    }
                }
            }

            /// <summary>
            /// メッシュに別のメッシュが内包できるかどうかを判定します。
            /// </summary>
            public bool Test(DivideMesh testMesh)
            {
                // TODO: AABB 以外の場合は改良が必要
                DivideAABB targetAABB = this.MeshBounding as DivideAABB;
                DivideAABB testAABB = testMesh.MeshBounding as DivideAABB;

                if (targetAABB != null &&
                    testAABB != null)
                {
                    return targetAABB.TestAABB(testAABB, 1.0f);
                }
                else
                {
                    throw new InvalidCastException();
                }
            }

            /// <summary>
            /// 不要なノードを削除してツリーを組み替えます。
            /// </summary>
            public DivideMesh Merge(DivideMesh mergedMesh)
            {
                Debug.WriteLine("MergeTo: " + mergedMesh.Triangles.Count + ":" + this.Triangles.Count);

                // 細かく分割されすぎた Cube を他のノードへマージします。
                foreach (ITriangle triangle in mergedMesh.Triangles)
                {
                    this.AddTriangle(triangle);
                }

                // mesh は消滅するので後始末する。
                mergedMesh.Triangles.Clear();
                mergedMesh.MeshBounding.SetZero();

                DivideMesh mergedParent = mergedMesh.Parent;

                if (mergedParent == null)
                {
                    throw new InvalidProgramException();
                }

                DivideMesh brother = (mergedParent.Children[0] == mergedMesh) ? mergedParent.Children[1] : mergedParent.Children[0];

                // マージした結果、ルートノードのみになる
                if (mergedParent.Parent == null)
                {
                    foreach (DivideMesh child in mergedParent.Children)
                    {
                        child.Parent = null;
                    }
                    mergedParent.Children[0] = mergedParent.Children[1] = null;

                    return brother;
                }

                // 親の親ノードに兄弟ノードを吊るすことで無くなったノードをを排除する
                int parentIndex = 0;
                for (int index = 0; index < mergedParent.Parent.Children.Length; ++index)
                {
                    if (mergedParent.Parent.Children[index] == mergedParent)
                    {
                        parentIndex = index;
                        break;
                    }
                }
                DivideMesh newParent = mergedParent.Parent;

                // 無くなったノードを排除するように親子関係を組み替える
                newParent.Children[parentIndex] = brother;
                brother.Parent = newParent;

                return null;
            }

            /// <summary>
            ///  面積をもつかどうかを判定します。
            /// </summary>
            public bool HasArea()
            {
                foreach (var triangle in this.triangles)
                {
                    var v01 = triangle.Vertices[1] - triangle.Vertices[0];
                    var v02 = triangle.Vertices[2] - triangle.Vertices[0];
                    if (v01.Cross(v02).LengthSquared > float.Epsilon)
                    {
                        return true;
                    }
                }
                return false;
            }

            /// <summary>
            /// ツリーをリストに並べます。
            /// </summary>
            public static List<DivideMesh> Bake(DivideMesh rootMesh)
            {
                List<DivideMesh> bakedArray = new List<DivideMesh>();
                Queue<DivideMesh> nodeQueue = new Queue<DivideMesh>();
                nodeQueue.Enqueue(rootMesh);
                while (nodeQueue.Count != 0)
                {
                    DivideMesh mesh = nodeQueue.Dequeue();
                    bakedArray.Add(mesh);

                    if (mesh.HasChildren)
                    {
                        nodeQueue.Enqueue(mesh.Children[0]);
                        nodeQueue.Enqueue(mesh.Children[1]);
                    }
                }

                return bakedArray;
            }

            /// <summary>
            /// リーフノードのみベイクします。
            /// </summary>
            public static List<DivideMesh> BakeLeaf(DivideMesh rootMesh)
            {
                List<DivideMesh> bakedMesh = DivideMesh.Bake(rootMesh);
                List<DivideMesh> leafMesh = new List<DivideMesh>();
                foreach (DivideMesh mesh in bakedMesh)
                {
                    if (!mesh.HasChildren)
                    {
                        leafMesh.Add(mesh);
                    }
                }

                return leafMesh;
            }

            public static void BreadthSearch(DivideMesh rootMesh)
            {
                Queue<DivideMesh> nodeQueue = new Queue<DivideMesh>();
                nodeQueue.Enqueue(rootMesh);
                int bakeBreadthIndex = 0;
                while (nodeQueue.Count != 0)
                {
                    DivideMesh mesh = nodeQueue.Dequeue();
                    mesh.BakeBreadthIndex = bakeBreadthIndex;
                    ++bakeBreadthIndex;

                    if (mesh.HasChildren)
                    {
                        nodeQueue.Enqueue(mesh.Children[0]);
                        nodeQueue.Enqueue(mesh.Children[1]);
                    }
                }
            }

            public static void DepthSearch(DivideMesh rootMesh)
            {
                Stack<DivideMesh> nodeStack = new Stack<DivideMesh>();
                nodeStack.Push(rootMesh);
                int bakeDepthIndex = 0;
                while (nodeStack.Count != 0)
                {
                    DivideMesh mesh = nodeStack.Pop();
                    mesh.BakeDepthIndex = bakeDepthIndex;
                    ++bakeDepthIndex;

                    if (mesh.HasChildren)
                    {
                        nodeStack.Push(mesh.Children[1]);
                        nodeStack.Push(mesh.Children[0]);
                    }
                }
            }

            private void SetParent(DivideMesh parent)
            {
                Parent = parent;

                if (Parent == null)
                {
                    Depth = 0;
                }
                else
                {
                    Depth = Parent.Depth + 1;
                }
            }

            private List<ITriangle> triangles = new List<ITriangle>();
            // 二分木のみサポート
            private DivideMesh[] children = new DivideMesh[2];
            private Vector3[] eigens = new Vector3[3] { new Vector3(), new Vector3(), new Vector3() };

            #region PCA
            /// <summary>
            /// 主成分分析により固有値を計算します。
            /// </summary>
            /// <remarks>C3T_Common.cpp から移植</remarks>
            private static void CalcPCA(Vector3[] eigens, IList<ITriangle> triangles)
            {
                if (triangles.Count == 0)
                {
                    foreach (Vector3 eigen in eigens)
                    {
                        eigen.SetZero();
                    }

                    return;
                }

                // 平均位置を計算します。
                Vector3 meanPosition = new Vector3();
                int positionCount = 0;
                foreach (Triangle triangle in triangles)
                {
                    foreach (Vector3 pos in triangle.Vertices)
                    {
                        meanPosition += pos;
                        ++positionCount;
                    }
                }

                meanPosition /= positionCount;

                // 共分散行列を作成します。
                float[,] covarianceMtx = new float[3, 3];
                foreach (Triangle triangle in triangles)
                {
                    foreach (Vector3 pos in triangle.Vertices)
                    {
                        for (int i = 0; i < 3; i++)
                        {
                            for (int j = i; j < 3; j++)
                            {
                                covarianceMtx[i, j] +=
                                    (pos[i] - meanPosition[i]) * (pos[j] - meanPosition[j]);
                            }
                        }
                    }
                }

                for (int i = 0; i < 3; i++)
                {
                    for (int j = i; j < 3; j++)
                    {
                        covarianceMtx[i, j] /= positionCount;
                        if (i != j)
                        {
                            covarianceMtx[j, i] = covarianceMtx[i, j];
                        }
                    }
                }

                // jacobi 法で固有値 & 固有ベクトルを算出します。
                float[,] eigenVectors;
                float[] eigenValues;
                bool solved = JacobiCalc(covarianceMtx, out eigenVectors, out eigenValues);

                foreach (Vector3 eigen in eigens)
                {
                    eigen.SetZero();
                }

                if (solved)
                {
                    // 固有値の降順でソートします。
                    List<KeyValuePair<float, int>> order = new List<KeyValuePair<float, int>>();
                    for (int i = 0; i < 3; i++)
                    {
                        order.Add(new KeyValuePair<float, int>(eigenValues[i], i));
                    }

                    order.Sort((x, y) => y.Key.CompareTo(x.Key));

                    for (int j = 0; j < 3; j++)
                    {
                        for (int i = 0; i < 3; i++)
                        {
                            eigens[j][i] = eigenVectors[i, order[j].Value];
                        }

                        eigens[j].Normalize();
                    }
                }
            }

            /// <summary>
            /// Jacobi 法により固有値と固有ベクトルを求めます。
            /// </summary>
            /// <remarks>C3T_Common.cpp より移植、NUMERICAL RECIPES in C より</remarks>
            /// <param name="a">３次行列です。</param>
            /// <param name="v">固有ベクトルです。</param>
            /// <param name="d">固有値です。</param>
            /// <returns>解が求まればtrue を返します。</returns>
            private static bool JacobiCalc(float[,] a, out float[,] v, out float[] d)
            {
                int n = 3;
                int i, j, iq, ip;
                float tresh, theta, tau, t, sm, s, h, g, c;
                float[] b = new float[3];
                float[] z = new float[3];
                v = new float[3, 3];
                d = new float[3];

                for (ip = 0; ip < n; ip++)
                {
                    for (iq = 0; iq < n; iq++)
                    {
                        v[ip, iq] = 0.0f;
                    }

                    v[ip, ip] = 1.0f;
                }

                for (ip = 0; ip < n; ip++)
                {
                    b[ip] = d[ip] = a[ip, ip];
                    z[ip] = 0.0f;
                }

                for (i = 0; i < 50; i++)
                {
                    sm = 0.0f;
                    for (ip = 0; ip < n - 1; ip++)
                    {
                        for (iq = ip + 1; iq < n; iq++)
                        {
                            sm += Math.Abs(a[ip, iq]);
                        }
                    }

                    if (sm == 0.0f)
                    {
                        return true;
                    }

                    if (i < 3)
                    {
                        tresh = 0.2f * sm / (n * n);
                    }
                    else
                    {
                        tresh = 0.0f;
                    }

                    for (ip = 0; ip < n - 1; ip++)
                    {
                        for (iq = ip + 1; iq < n; iq++)
                        {
                            g = 100.0f * Math.Abs(a[ip, iq]);
                            if (i > 3 && (Math.Abs(d[ip]) + g) == Math.Abs(d[ip])
                                && (Math.Abs(d[iq]) + g) == Math.Abs(d[iq]))
                            {
                                a[ip, iq] = 0.0f;
                            }
                            else if (Math.Abs(a[ip, iq]) > tresh)
                            {
                                h = d[iq] - d[ip];
                                if ((Math.Abs(h) + g) == Math.Abs(h))
                                {
                                    t = a[ip, iq] / h;
                                }
                                else
                                {
                                    theta = 0.5f * h / a[ip, iq];
                                    t = 1 / (Math.Abs(theta) + (float)Math.Sqrt(1.0f + (theta * theta)));
                                    if (theta < 0.0f)
                                    {
                                        t = -t;
                                    }
                                }

                                c = 1.0f / (float)Math.Sqrt(1 + (t * t));
                                s = t * c;
                                tau = s / (1.0f + c);
                                h = t * a[ip, iq];
                                z[ip] -= h;
                                z[iq] += h;
                                d[ip] -= h;
                                d[iq] += h;
                                a[ip, iq] = 0.0f;

                                for (j = 0; j < ip; j++)
                                {
                                    JacobiRotate(a, s, tau, j, ip, j, iq);
                                }

                                for (j = ip + 1; j < iq; j++)
                                {
                                    JacobiRotate(a, s, tau, ip, j, j, iq);
                                }

                                for (j = iq + 1; j < n; j++)
                                {
                                    JacobiRotate(a, s, tau, ip, j, iq, j);
                                }

                                for (j = 0; j < n; j++)
                                {
                                    JacobiRotate(v, s, tau, j, ip, j, iq);
                                }
                            }
                        }
                    }

                    for (ip = 0; ip < n; ip++)
                    {
                        b[ip] += z[ip];
                        d[ip] = b[ip];
                        z[ip] = 0.0f;
                    }
                }

                return false;
            }

            private static void JacobiRotate(float[,] a, float s, float tau, int i, int j, int k, int l)
            {
                float h, g;
                g = a[i, j];
                h = a[k, l];
                a[i, j] = g - (s * (h + (g * tau)));
                a[k, l] = h + (s * (g - (h * tau)));
            }
            #endregion

            #region トライングル判定

            public static bool TestTriangleAABB(Triangle triangle, AABB aabb)
            {
                Vector3 trans = new Vector3(aabb.Center);
                Vector3 scale = new Vector3(1.0f / aabb.Width.X, 1.0f / aabb.Width.Y, 1.0f / aabb.Width.Z);
                Vector3[] v = new Vector3[3];
                Vector3[] e = new Vector3[3];
                Vector3[] u = new Vector3[3];
                float[] l = new float[3];

                for (int i = 0; i < 3; ++i)
                {
                    v[i] = new Vector3(triangle.Vertices[i] - trans);
                }

                // 三角形の辺
                e[0] = new Vector3(v[1] - v[0]);
                e[1] = new Vector3(v[2] - v[1]);
                e[2] = new Vector3(v[0] - v[2]);

                u[0] = new Vector3(1.0f, 0.0f, 0.0f);
                u[1] = new Vector3(0.0f, 1.0f, 0.0f);
                u[2] = new Vector3(0.0f, 0.0f, 1.0f);

                l[0] = aabb.Width.X / 2.0f;
                l[1] = aabb.Width.Y / 2.0f;
                l[2] = aabb.Width.Z / 2.0f;
                float[] p = new float[3];

                for (int i = 0; i < 3; ++i)
                {
                    for (int j = 0; j < 3; ++j)
                    {
                        Vector3 a = Vector3.Cross(u[i], e[j]);
                        if (a.IsZero())
                        {
                            //Console.Error.WriteLine("a is zero vector.");
                            continue;
                        }
                        float r = l[0] * Math.Abs(a.X) + l[1] * Math.Abs(a.Y) + l[2] * Math.Abs(a.Z);
                        p[0] = Vector3.Dot(v[0], a);
                        p[1] = Vector3.Dot(v[1], a);
                        p[2] = Vector3.Dot(v[2], a);
                        if (Math.Max(-Math.Max(Math.Max(p[0], p[1]), p[2]), Math.Min(Math.Min(p[0], p[1]), p[2])) > r)
                        {
                            return false;
                        }
                    }
                }

                if (Math.Max(Math.Max(v[0].X, v[1].X), v[2].X) < -l[0] || Math.Min(Math.Min(v[0].X, v[1].X), v[2].X) > l[0])
                {
                    return false;
                }

                if (Math.Max(Math.Max(v[0].Y, v[1].Y), v[2].Y) < -l[1] || Math.Min(Math.Min(v[0].Y, v[1].Y), v[2].Y) > l[1])
                {
                    return false;
                }

                if (Math.Max(Math.Max(v[0].Z, v[1].Z), v[2].Z) < -l[2] || Math.Min(Math.Min(v[0].Z, v[1].Z), v[2].Z) > l[2])
                {
                    return false;
                }

                Plane plane = new Plane();
                plane.N.Set(Vector3.Cross(e[0], e[1]));
                if (plane.N.IsZero())
                {
                    // 三角形なので法線が求められない場合は直線になる。
                    // これは分離軸にならないので判定を終了する。
                    // TODO: 単純に true を返して良いかどうか調査する。
                    //Console.Error.WriteLine("n is zero vector.");
                    return true;
                }
                else
                {
                    plane.N.Normalize();
                    plane.D = Vector3.Dot(plane.N, triangle.Vertices[0]);
                    bool testPlane = TestPlaneCube(plane, aabb);
                    return testPlane;
                }
            }

            private static bool TestPlaneCube(Plane plane, AABB aabb)
            {
                Vector3 e = aabb.Max - aabb.Center;
                float r = e.X * Math.Abs(plane.N.X) + e.Y * Math.Abs(plane.N.Y) + e.Z * Math.Abs(plane.N.Z);
                float s = Vector3.Dot(plane.N, aabb.Center) - plane.D;

                return Math.Abs(s) <= r;
            }
            #endregion
        }

        #region 分割アルゴリズム
        /// <summary>
        /// 分割アルゴリズムのインタフェースです。
        /// </summary>
        public interface IDivideAlgorithm
        {
            ResultCode Divide(DivideMesh mesh);
        }

        /// <summary>
        /// 分割してみてフィットした割合によって分割の可否判定を行うアルゴリズムです。
        /// </summary>
        public class DivideSquashing : IDivideAlgorithm
        {
            /// <summary>
            /// ノードの深さの最大値です。
            /// </summary>
            public int DepthMax { get; set; }

            /// <summary>
            /// 分割を行う最大の幅です。
            /// </summary>
            public float WidthMax { get; set; }

            /// <summary>
            /// 分割を行う最小の三角形数です。
            /// </summary>
            public int MinTriangleCount { get; set; }

            /// <summary>
            /// 縮小率の最小値です。
            /// </summary>
            public float ReductionTargetMin { get; set; }

            /// <summary>
            /// 縮小率の最大値です。
            /// </summary>
            public float ReductionTargetMax { get; set; }

            public ResultCode Divide(DivideMesh mesh)
            {
                // ツリーの深さが指定数より深いなら終了。
                if (mesh.Depth >= DepthMax)
                {
                    return ResultCode.TooDeep;
                }

                // ボックスの大きさが指定値より小さいなら終了。
                if (WidthMax > Math.Max(Math.Max(mesh.MeshBounding.Width.X, mesh.MeshBounding.Width.Y), mesh.MeshBounding.Width.Z))
                {
                    return ResultCode.ShortBox;
                }

                // 三角形の数が指定値より少ないなら終了。
                if (mesh.Triangles.Count < MinTriangleCount)
                {
                    return ResultCode.LessTriangle;
                }

                const int AxisCount = 3;

                float[] reductionRateArray = new float[AxisCount];
                int[][] indices = new int[AxisCount][];
                List<ITriangle>[] triangles = new List<ITriangle>[AxisCount];

                for (int axis = 0; axis < AxisCount; ++axis)
                {
                    indices[axis] = new int[mesh.Triangles.Count];
                    triangles[axis] = new List<ITriangle>(mesh.Triangles);
                    mesh.SortFromEigen(triangles[axis], axis);
                    reductionRateArray[axis] = CalcReductionRate(indices[axis], triangles[axis], mesh.MeshBounding.Area, mesh.MeshBounding.Volume);

                    Debug.WriteLine("reduction: " + reductionRateArray[axis].ToString());
                }

                int bestAxis = -1;
                for (int axis = 0; axis < AxisCount; ++axis)
                {
                    if (reductionRateArray[axis] > ReductionTargetMin &&
                        reductionRateArray[axis] < ReductionTargetMax)
                    {
                        if ((bestAxis == -1) ||
                            (reductionRateArray[axis] < reductionRateArray[bestAxis]))
                        {
                            bestAxis = axis;
                        }
                    }
                }

                if (bestAxis == -1)
                {
                    // ReductionTarget の範囲の場合は Children を作成する。
                    if (reductionRateArray[0] <= ReductionTargetMin &&
                        reductionRateArray[1] <= ReductionTargetMin &&
                        reductionRateArray[2] <= ReductionTargetMin)
                    {
                        // 線などが誤判定されるのを防ぐ
                        return ResultCode.TooSmall;
                    }
                    else
                    {
                        return ResultCode.NotReduction;
                    }
                }

                // 子ノード初期化
                mesh.Children[0] = new DivideMesh(mesh, new DivideAABB());
                mesh.Children[1] = new DivideMesh(mesh, new DivideAABB());

                // 三角形を追加していく
                for (int i = 0; i < mesh.Triangles.Count; ++i)
                {
                    int index = indices[bestAxis][i];
                    mesh.Children[index].AddTriangle(triangles[bestAxis][i]);
                }

                // 実際にトライアングルを格納してみて、片側のみに振り分けられた場合は分割を中止する。
                // これは判定と実際の格納で AABB が異なることに起因する。
                if (mesh.Children[0].Triangles.Count == 0 ||
                    mesh.Children[1].Triangles.Count == 0)
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.Discrepancy;
                }

                if (!mesh.Children[0].HasArea() || !mesh.Children[1].HasArea())
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.TooSmall;
                }

                mesh.Triangles.Clear();
                mesh.Calc();

                Debug.WriteLine("DivideTo: " + mesh.Children[0].Triangles.Count * 3 + ":" + mesh.Children[1].Triangles.Count * 3);

                return ResultCode.Success;
            }

            /// <summary>
            /// 縮小率を計算します。
            /// </summary>
            private float CalcReductionRate(int[] indices, List<ITriangle> triangles, float area, float volume)
            {
                AABB aBox = new AABB();
                aBox.AddTriangle(triangles[0]);
                AABB bBox = new AABB();
                bBox.AddTriangle(triangles[triangles.Count - 1]);

                for (int i = 0; i < triangles.Count / 2; ++i)
                {
                    int aIndex = i;
                    int bIndex = triangles.Count - 1 - i;

                    // 両端のトライアングルを AABB に入れていく。
                    ITriangle a = triangles[aIndex];
                    ITriangle b = triangles[bIndex];

                    int aResult = DivideAABB.TestTriangle(aBox, bBox, a);
                    indices[aIndex] = aResult;

                    if (aResult == 0)
                    {
                        aBox.AddTriangle(a);
                    }
                    else
                    {
                        bBox.AddTriangle(a);
                    }

                    int bResult = DivideAABB.TestTriangle(aBox, bBox, b);
                    indices[bIndex] = bResult;

                    if (bResult == 0)
                    {
                        aBox.AddTriangle(b);
                    }
                    else
                    {
                        bBox.AddTriangle(b);
                    }
                }

                // 奇数の場合
                if (triangles.Count % 2 != 0)
                {
                    int aIndex = triangles.Count / 2;
                    ITriangle a = triangles[aIndex];

                    int aResult = DivideAABB.TestTriangle(aBox, bBox, a);
                    indices[aIndex] = aResult;

                    if (aResult == 0)
                    {
                        aBox.AddTriangle(a);
                    }
                    else
                    {
                        bBox.AddTriangle(a);
                    }
                }

                // Volume の減少率を計算する
                float reductionRate = 0.0f;
                if (volume <= FloatUtility.Epsilon)
                {
                    if (area <= FloatUtility.Epsilon)
                    {
                        // 線は分割しない
                        reductionRate = -float.MinValue;
                    }
                    // 面積基準で比較する。
                    reductionRate = ((aBox.Area + bBox.Area) / area) * 100.0f;
                }
                else
                {
                    reductionRate = ((aBox.Volume + bBox.Volume) / volume) * 100.0f;
                }

                return reductionRate;
            }
        }

        /// <summary>
        /// 例外ポリゴンを分割するアルゴリズムです。
        /// </summary>
        public class DivideExculusion : IDivideAlgorithm
        {
            /// <summary>
            /// ノードの深さの最大値です。
            /// </summary>
            public int DepthMax { get; set; }

            public ResultCode Divide(DivideMesh mesh)
            {
                // ツリーの深さが指定数より深いなら終了。
                if (mesh.Depth >= DepthMax)
                {
                    return ResultCode.TooDeep;
                }

                // 例外ポリゴンが含まれない場合は分割終了
                if (!mesh.HasExclusionPolygon)
                {
                    return ResultCode.NotDivide;
                }

                mesh.Children[0] = new DivideMesh(mesh, new DivideAABB());
                mesh.Children[1] = new DivideMesh(mesh, new DivideAABB());

                for (int i = 0; i < mesh.Triangles.Count; ++i)
                {
                    DivideTriangle triangle = mesh.Triangles[i] as DivideTriangle;
                    if (triangle.IsExclusion)
                    {
                        mesh.Children[0].AddTriangle(triangle);
                    }
                    else
                    {
                        mesh.Children[1].AddTriangle(triangle);
                    }
                }

                if (mesh.Children[0].Triangles.Count == 0 ||
                    mesh.Children[1].Triangles.Count == 0)
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.LessTriangle;
                }

                if (!mesh.Children[0].HasArea() || !mesh.Children[1].HasArea())
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.TooSmall;
                }

                mesh.Triangles.Clear();
                mesh.Calc();
                return ResultCode.Success;
            }
        }

        /// <summary>
        /// 主成分ベクトルを用いて分割するアルゴリズムです。
        /// </summary>
        public class DivideEigen : IDivideAlgorithm
        {
            /// <summary>
            /// ノードの深さの最大値です。
            /// </summary>
            public int DepthMax { get; set; }

            /// <summary>
            /// 分割を行う最小の幅です。
            /// </summary>
            public float WidthMax { get; set; }

            /// <summary>
            /// 分割を行う最小の三角形数です。
            /// </summary>
            public int MinTriangleCount { get; set; }

            public ResultCode Divide(DivideMesh mesh)
            {
                // ツリーの深さが指定数より深いなら終了。
                if (mesh.Depth >= DepthMax)
                {
                    return ResultCode.TooDeep;
                }

                // ボックスの大きさが指定値より小さいなら終了。
                if (WidthMax > Math.Max(Math.Max(mesh.MeshBounding.Width.X, mesh.MeshBounding.Width.Y), mesh.MeshBounding.Width.Z))
                {
                    return ResultCode.ShortBox;
                }

                // 三角形の数が指定値より少ないなら終了。
                if (mesh.Triangles.Count < MinTriangleCount)
                {
                    return ResultCode.LessTriangle;
                }

                // 除外ポリゴンが含まれる場合は分割を中止する。
                if (mesh.HasExclusionPolygon)
                {
                    return ResultCode.Exclusion;
                }

                // 主成分方向でソート
                mesh.SortFromEigen(mesh.Triangles, 0);

                mesh.Children[0] = new DivideMesh(mesh, new DivideAABB());
                mesh.Children[1] = new DivideMesh(mesh, new DivideAABB());

                ITriangle head = mesh.Triangles[0];
                ITriangle tail = mesh.Triangles[mesh.Triangles.Count - 1];
                float axisHalfLength = (tail.Center - head.Center).Length / 2.0f;
                for (int i = 0; i < mesh.Triangles.Count; ++i)
                {
                    ITriangle triangle = mesh.Triangles[i];
                    float length = (triangle.Center - head.Center).Length;
                    if (length < axisHalfLength)
                    {
                        mesh.Children[0].AddTriangle(triangle);
                    }
                    else
                    {
                        mesh.Children[1].AddTriangle(triangle);
                    }
                }

                if (mesh.Children[0].Triangles.Count == 0 || mesh.Children[1].Triangles.Count == 0)
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.LessTriangle;
                }

                if (!mesh.Children[0].HasArea() || !mesh.Children[1].HasArea())
                {
                    mesh.Children[0] = mesh.Children[1] = null;
                    return ResultCode.TooSmall;
                }

                mesh.Triangles.Clear();
                mesh.Calc();

                return ResultCode.Success;
            }
        }
        #endregion

        public static Vector3 MinVector3(Vector3 result, Vector3 lhs, Vector3 rhs)
        {
            result.X = Math.Min(lhs.X, rhs.X);
            result.Y = Math.Min(lhs.Y, rhs.Y);
            result.Z = Math.Min(lhs.Z, rhs.Z);

            return result;
        }

        public static Vector3 MaxVector3(Vector3 result, Vector3 lhs, Vector3 rhs)
        {
            result.X = Math.Max(lhs.X, rhs.X);
            result.Y = Math.Max(lhs.Y, rhs.Y);
            result.Z = Math.Max(lhs.Z, rhs.Z);

            return result;
        }

        // 出力用の構造
        public class MeshNode
        {
            public MeshNode(DivideMesh baseMesh)
            {
                ParentBreadthIndex = (baseMesh.Parent == null) ? -1 : baseMesh.Parent.BakeBreadthIndex;
                BreadthIndex = baseMesh.BakeBreadthIndex;
                ParentDepthIndex = (baseMesh.Parent == null) ? -1 : baseMesh.Parent.BakeDepthIndex;
                DepthIndex = baseMesh.BakeDepthIndex;
                IndexCount = baseMesh.Triangles.Count * 3;
                triangles = baseMesh.Triangles;
                SubmeshIndex = -1;
            }

            public int IndexCount
            {
                get;
                private set;
            }
            public int IndexOffset
            {
                get;
                private set;
            }

            public int ParentBreadthIndex
            {
                get;
                private set;
            }

            public int BreadthIndex
            {
                get;
                private set;
            }

            public int ParentDepthIndex
            {
                get;
                private set;
            }

            public int DepthIndex
            {
                get;
                private set;
            }

            public int SubmeshIndex
            {
                get;
                private set;
            }

            private List<ITriangle> triangles = null;

            public bool AddToIndexStream(List<int> indexStream, int submeshIndex)
            {
                if (triangles.Count > 0)
                {
                    SubmeshIndex = submeshIndex;
                    IndexOffset = indexStream.Count;

                    foreach (Triangle triangle in triangles)
                    {
                        indexStream.AddRange(triangle.Indices);
                    }

                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// トライアングルです。
        /// </summary>
        public interface ITriangle
        {
            int[] Indices { get; }
            Vector3[] Vertices { get; }
            Vector3 Center { get; }
        }

        /// <summary>
        /// トライアングルです。
        /// </summary>
        public class Triangle : ITriangle
        {
            public int[] Indices
            {
                get
                {
                    return indices;
                }
            }

            public Vector3[] Vertices
            {
                get
                {
                    return vertices;
                }
            }

            public Vector3 Center
            {
                get
                {
                    return center;
                }
            }

            private int[] indices = new int[3];
            private Vector3[] vertices = new Vector3[3];
            private Vector3 center = new Vector3();
        }

        /// <summary>
        /// 平面です。
        /// </summary>
        public class Plane
        {
            public Vector3 N
            {
                get
                {
                    return n;
                }
            }

            public float D { get; set; }

            private Vector3 n = new Vector3();
        }

        /// <summary>
        /// 分割用トライアングルです。
        /// </summary>
        public class DivideTriangle : Triangle
        {
            public bool IsExclusion { get; set; }
        }

        /// <summary>
        /// バウンディングです。
        /// </summary>
        public interface IBounding
        {
            Vector3 Width { get; }

            float Volume { get; }

            float Area { get; }

            void AddTriangle(ITriangle triangle);

            void SetZero();
        }

        /// <summary>
        /// AABB です。
        /// </summary>
        public class AABB : IBounding
        {
            public Vector3 Max
            {
                get
                {
                    return max;
                }
            }
            public Vector3 Min
            {
                get
                {
                    return min;
                }
            }

            public Vector3 Width
            {
                get
                {
                    return width;
                }
            }
            public Vector3 Center
            {
                get
                {
                    return center;
                }
            }

            public float Volume { get; private set; }

            public float Area
            {
                get
                {
                    return Math.Max(Math.Max(Width.X * Width.Y, Width.X * Width.Z), Width.Y * Width.Z);
                }
            }

            public AABB()
            {
                Init();
            }

            public void Init()
            {
                Max.Set(float.MinValue, float.MinValue, float.MinValue);
                Min.Set(float.MaxValue, float.MaxValue, float.MaxValue);
                Width.SetZero();
                Center.SetZero();
                Volume = 0.0f;
            }

            public void Set(AABB value)
            {
                Max.Set(value.Max);
                Min.Set(value.Min);
                Width.Set(value.Width);
                Center.Set(value.Center);
                Volume = value.Volume;
            }

            public void Set(Vector3 max, Vector3 min)
            {
                Max.Set(max);
                Min.Set(min);

                Calc();
            }

            public void Set(ITriangle triangle)
            {
                Max.Set(float.MinValue, float.MinValue, float.MinValue);
                Min.Set(float.MaxValue, float.MaxValue, float.MaxValue);
                for (int i = 0; i < 3; ++i)
                {
                    Max.X = Math.Max(triangle.Vertices[i].X, Max.X);
                    Max.Y = Math.Max(triangle.Vertices[i].Y, Max.Y);
                    Max.Z = Math.Max(triangle.Vertices[i].Z, Max.Z);

                    Min.X = Math.Min(triangle.Vertices[i].X, Min.X);
                    Min.Y = Math.Min(triangle.Vertices[i].Y, Min.Y);
                    Min.Z = Math.Min(triangle.Vertices[i].Z, Min.Z);
                }

                Calc();
            }

            public void AddTriangle(ITriangle triangle)
            {
                for (int i = 0; i < 3; ++i)
                {
                    Vector3 vertex = triangle.Vertices[i];
                    if (vertex.X > Max.X) { Max.X = vertex.X; }
                    if (vertex.X < Min.X) { Min.X = vertex.X; }
                    if (vertex.Y > Max.Y) { Max.Y = vertex.Y; }
                    if (vertex.Y < Min.Y) { Min.Y = vertex.Y; }
                    if (vertex.Z > Max.Z) { Max.Z = vertex.Z; }
                    if (vertex.Z < Min.Z) { Min.Z = vertex.Z; }
                }

                Calc();
            }

            public void SetZero()
            {
                Max.Set(Vector3.Zero);
                Min.Set(Vector3.Zero);

                Calc();
            }

            protected void Calc()
            {
                Width.Set(Max - Min);
                Center.Set((Max + Min) / 2.0f);
                Volume = Width.X * Width.Y * Width.Z;
            }

            private Vector3 max = new Vector3();
            private Vector3 min = new Vector3();
            private Vector3 width = new Vector3();
            private Vector3 center = new Vector3();
        }

        /// <summary>
        /// 分割用 AABB を表すクラスです。
        /// </summary>
        public class DivideAABB : AABB
        {
            public static int TestTriangle(AABB aBox, AABB bBox, ITriangle tri)
            {
                AABB aTempBox = new AABB();
                aTempBox.Set(aBox);
                AABB bTempBox = new AABB();
                bTempBox.Set(bBox);

                aTempBox.AddTriangle(tri);
                bTempBox.AddTriangle(tri);

                bool isAboxSmall = false;
                // 平面の場合に誤判定する可能性があるので特殊処理
                if (aTempBox.Volume == 0.0f && bTempBox.Volume == 0.0f)
                {
                    if (aTempBox.Area - aBox.Area < bTempBox.Area - bBox.Area)
                    {
                        isAboxSmall = true;
                    }
                }
                else
                {
                    if (aTempBox.Volume - aBox.Volume < bTempBox.Volume - bBox.Volume)
                    {
                        isAboxSmall = true;
                    }
                }

                return isAboxSmall ? 0 : 1;
            }

            public bool TestAABB(DivideAABB box, float ratio)
            {
                // AABB を内包しているかどうかをテストします。
                DivideAABB tempBox = new DivideAABB();
                tempBox.Set(this);
                tempBox.Inflate(ratio);

                if (tempBox.Max.X > box.Max.X &&
                    tempBox.Max.Y > box.Max.Y &&
                    tempBox.Max.Z > box.Max.Z &&
                    tempBox.Min.X < box.Min.X &&
                    tempBox.Min.Y < box.Min.Y &&
                    tempBox.Min.Z < box.Min.Z)
                {
                    return true;
                }

                return false;
            }

            public void Inflate(float ratio)
            {
                // 原点に移動
                Max.Set(Max - Center);
                Min.Set(Min - Center);

                // Center の値との情報落ち対策として
                // 指数部はそのままで、仮数部を 0x1 とした値以上の値を加える。

                Vector3 trivialCenter =
                    new Vector3(GetTrivialFloat(Center.X), GetTrivialFloat(Center.Y), GetTrivialFloat(Center.Z));

                Vector3 minmax = new Vector3();

                Max.Set(Max + MaxVector3(minmax, Max * (ratio / 200.0f), trivialCenter) + Center);
                Min.Set(Min + MinVector3(minmax, Min * (ratio / 200.0f), trivialCenter * -1.0f) + Center);

                Calc();
            }

            private static float GetTrivialFloat(float value)
            {
                byte[] bytes = BitConverter.GetBytes(value);
                bytes[0] = 0x1;
                bytes[1] = 0x0;
                bytes[2] = (byte)(bytes[2] & 0x80);
                bytes[3] = (byte)(bytes[3] & 0x7F); // 符号正
                float tempA = BitConverter.ToSingle(bytes, 0);
                bytes[0] = 0x0;
                float tempB = BitConverter.ToSingle(bytes, 0);

                return tempA - tempB;
            }
        }

        /// <summary>
        /// コマンドラインの解析用ユーティリティです。
        /// </summary>
        public class ArgumentProcessor
        {
            public ArgumentProcessor(string arg)
            {
                Arguments = new List<ArgumentValue>();
                Interpret(arg);
            }

            private void Interpret(string arg)
            {
                int startIndex = arg.IndexOf('(');
                if (startIndex < 0)
                {
                    return;
                }

                Function = arg.Substring(0, startIndex).Trim();
                Regex regex = new Regex("\\(.*\\)");
                Match match = regex.Match(arg);

                if (!match.Success)
                {
                    return;
                }

                string arguments = match.Value;
                if (arguments.IndexOf('(') != arguments.LastIndexOf('(') ||
                    arguments.IndexOf(')') != arguments.LastIndexOf(')'))
                {
                    IfStrings.Throw("IfCheck_Error_Arguments", arguments);
                }

                int startIdx = arguments.IndexOf('(') + 1;
                int endIdx = arguments.IndexOf(')') - 1;
                string tempArg = arguments.Substring(startIdx, endIdx);

                string[] argSplit = tempArg.Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                for (int i = 0; i < argSplit.Length; ++i)
                {
                    int idx = argSplit[i].IndexOf(':');
                    if (idx < 0)
                    {
                        IfStrings.Throw("IfCheck_Error_Arguments", arguments);
                    }
                    ArgumentValue argValue = new ArgumentValue();
                    argValue.arg = argSplit[i].Substring(0, idx).Trim();
                    argValue.value = argSplit[i].Substring(idx + 1).Trim();

                    Arguments.Add(argValue);
                }
            }

            public class ArgumentValue
            {
                public string arg { get; set; }
                public string value { get; set; }
            }

            public string Function { get; private set; }
            public List<ArgumentValue> Arguments { get; private set; }
        }
    }
}
